package gov.va.med.ars.service.impl;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.mockito.Mockito.when;

import java.util.ArrayList;
import java.util.List;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Matchers;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;

import gov.va.med.ars.dao.ars.HccsCdRepository;
import gov.va.med.ars.dao.ars.LoincCdRepository;
import gov.va.med.ars.dao.ars.LoincModCdRepository;
import gov.va.med.ars.exceptions.GenericException;
import gov.va.med.ars.model.request.CodeAndModifierRequest;
import gov.va.med.ars.model.request.GenericRequest;
import gov.va.med.ars.model.response.CodeAndModifierResponse;
import gov.va.med.domain.ars.HccsCd;
import gov.va.med.domain.ars.LoincCd;
import gov.va.med.domain.ars.LoincModCd;

@RunWith(MockitoJUnitRunner.class)
public class CodeAndModifierServiceImplTest {

	@Mock
	LoincCdRepository loincCdRepository;

	@Mock
	LoincModCdRepository loincModCdRepository;

	@Mock
	HccsCdRepository hccsCdRepository;

	@InjectMocks
	CodeAndModifierServiceImpl codeAndModService;

	@Test
	public void testMockIsWorking() {
		assertNotNull("CodeAndModifierServiceImpl Injection Failed", codeAndModService);
	}

	@SuppressWarnings("unchecked")
	@Test
	public void testGetCodeAndModifier_loinc_success() throws GenericException {

		// arrange (service)
		String url = "/api/v1/populate/loinc";
		List<String> codeActive = new ArrayList<>();
		codeActive.add("Y");
		codeActive.add("N");

		// arrange (repository)
		LoincCd loinc1 = new LoincCd();
		loinc1.setLoincId(999l);
		loinc1.setIsactive('Y');
		loinc1.setStcloincCd("loincCd");
		loinc1.setStcloincCdDesc("loincCd_desc");
		List<LoincCd> loincCdList = new ArrayList<>();
		loincCdList.add(loinc1);

		when(loincCdRepository.findByisactiveIn(Matchers.anyList())).thenReturn(loincCdList);

		// act (service)
		List<CodeAndModifierResponse> response = (List<CodeAndModifierResponse>) codeAndModService
				.getCodeAndModifier(url, codeActive);

		assertEquals(1, response.size());
		assertEquals("loincCd", response.get(0).getStcCd());
		assertEquals("loincCd_desc", response.get(0).getStcCdDesc());
		assertEquals(new Long(999), response.get(0).getCd());

	}

	@SuppressWarnings("unchecked")
	@Test
	public void testGetCodeAndModifier_loincMod_success() throws GenericException {

		// arrange (service)
		String url = "/api/v1/populate/loincmod";
		List<String> codeActive = new ArrayList<>();
		codeActive.add("Y");
		codeActive.add("N");

		// arrange (repository)
		LoincModCd loincMod1 = new LoincModCd();
		loincMod1.setLoincModId(888l);
		loincMod1.setIsactive('Y');
		loincMod1.setStcloincCdMod("loincCdMod");
		loincMod1.setStcloincCdModDesc("loincCdMod_desc");
		List<LoincModCd> loincCdModList = new ArrayList<>();
		loincCdModList.add(loincMod1);

		when(loincModCdRepository.findByisactiveIn(Matchers.anyList())).thenReturn(loincCdModList);

		// act (service)
		List<CodeAndModifierResponse> response = (List<CodeAndModifierResponse>) codeAndModService
				.getCodeAndModifier(url, codeActive);

		assertEquals(1, response.size());
		assertEquals("loincCdMod", response.get(0).getStcCd());
		assertEquals("loincCdMod_desc", response.get(0).getStcCdDesc());
		assertEquals(new Long(888), response.get(0).getCd());

	}

	@SuppressWarnings("unchecked")
	@Test
	public void testGetCodeAndModifier_hccs_success() throws GenericException {

		// arrange (service)
		String url = "/api/v1/populate/hccs";
		List<String> codeActive = new ArrayList<>();
		codeActive.add("Y");
		codeActive.add("N");

		// arrange (repository)
		HccsCd hccsCd1 = new HccsCd();
		hccsCd1.setHccsId(777l);
		hccsCd1.setIsactive('Y');
		hccsCd1.setStchccstatuscatCd("hccsCd");
		hccsCd1.setStchccstatuscatCdDesc("hccsCd_desc");
		List<HccsCd> hccsCdList = new ArrayList<>();
		hccsCdList.add(hccsCd1);

		when(hccsCdRepository.findByisactiveIn(Matchers.anyList())).thenReturn(hccsCdList);

		// act (service)
		List<CodeAndModifierResponse> response = (List<CodeAndModifierResponse>) codeAndModService
				.getCodeAndModifier(url, codeActive);

		assertEquals(1, response.size());
		assertEquals("hccsCd", response.get(0).getStcCd());
		assertEquals("hccsCd_desc", response.get(0).getStcCdDesc());
		assertEquals(new Long(777), response.get(0).getCd());

	}

	@SuppressWarnings({ "unchecked", "unused" })
	@Test(expected = GenericException.class)
	public void testGetCodeAndModifier_fail() throws GenericException {

		String url = "/api/v1/populate/apples";
		List<String> codeActive = new ArrayList<>();
		codeActive.add("Y");
		codeActive.add("N");

		List<CodeAndModifierResponse> response = (List<CodeAndModifierResponse>) codeAndModService
				.getCodeAndModifier(url, codeActive);

	}

	@Test
	public void test_checkSubmittedModificationsForDupes() {
		// arrange (service)
		CodeAndModifierRequest loinc = new CodeAndModifierRequest();
		loinc.setStcCd("loinc");
		List<CodeAndModifierRequest> loincList = new ArrayList<>();
		loincList.add(loinc);

		CodeAndModifierRequest loincMod = new CodeAndModifierRequest();
		loincMod.setStcCd("loincMod");
		List<CodeAndModifierRequest> loincModList = new ArrayList<>();
		loincModList.add(loincMod);
		
		CodeAndModifierRequest hccsCd = new CodeAndModifierRequest();
		hccsCd.setStcCd("hccs");
		List<CodeAndModifierRequest> hccList = new ArrayList<>();
		hccList.add(hccsCd);
		
		GenericRequest request = new GenericRequest();
		request.setLoincList(loincList);
		request.setHccList(hccList);
		request.setLoincModList(loincModList);

		// arrange(repository)
		LoincCd _loincCd = new LoincCd();
		_loincCd.setLoincId(999l);
		_loincCd.setStcloincCd("loincCd");
		_loincCd.setStcloincCdDesc("loincCd_desc");
		
		LoincModCd _loincModCd = new LoincModCd();
		_loincModCd.setLoincModId(999l);
		_loincModCd.setStcloincCdMod("loincModCd");
		_loincModCd.setStcloincCdMod("loincModCd_desc");
		
		HccsCd _hccsCd = new HccsCd();
		_hccsCd.setHccsId(999l);
		_hccsCd.setStchccstatuscatCd("hccsCd");
		_hccsCd.setStchccstatuscatCdDesc("hccsCd_desc");

		when(loincCdRepository.findOneByStcloincCd(Matchers.any(String.class))).thenReturn(_loincCd);
		when(loincModCdRepository.findOneByStcloincCdMod(Matchers.any(String.class))).thenReturn(_loincModCd);
		when(hccsCdRepository.findOneByStchccstatuscatCd(Matchers.any(String.class))).thenReturn(_hccsCd);

		// act (repository)
		List<String> duplicateErrorList = codeAndModService.checkSubmittedModificationsForDupes(request);

		assertEquals(3, duplicateErrorList.size());
		assertEquals("Duplicate Found: loincMod of type LOINC Modifier Code ", duplicateErrorList.get(0));
		assertEquals("Duplicate Found: loinc of type LOINC Code ", duplicateErrorList.get(1));
		assertEquals("Duplicate Found: hccs of type HCCS Code ", duplicateErrorList.get(2));
	}

}
