package gov.va.med.ars.controller;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Matchers;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.test.context.web.WebAppConfiguration;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.ResultActions;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.SerializationFeature;

import gov.va.med.ars.configuration.AppConfig;
import gov.va.med.ars.configuration.spring.SpringMvcConfig;
import gov.va.med.ars.constants.ErrorMessages;
import gov.va.med.ars.exceptions.GenericException;
import gov.va.med.ars.filter.CORSFilter;
import gov.va.med.ars.model.request.UserSearchRequest;
import gov.va.med.ars.model.response.GenericResponse;
import gov.va.med.ars.model.response.UserDetailsResponse;
import gov.va.med.ars.service.IUserAdministrationService;
import gov.va.med.ars.service.impl.UserAdministrationServiceImpl;
import net.minidev.json.JSONObject;

import static org.mockito.Mockito.when;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.put;

import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;

import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.content;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;

import static org.mockito.Matchers.any;

@WebAppConfiguration
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(classes = { AppConfig.class, SpringMvcConfig.class })
public class UserAdministrationRestControllerTest {
	private static final Logger logger = LogManager.getLogger(UserAdministrationRestControllerTest.class);
	private MockMvc mockMvc;
	public static final MediaType APPLICATION_JSON_UTF8 = new MediaType(MediaType.APPLICATION_JSON.getType(),
			MediaType.APPLICATION_JSON.getSubtype(), Charset.forName("utf8"));

	@Mock
	private IUserAdministrationService userAdministrationService;

	@Mock
	private GenericException genericException = new GenericException(ErrorMessages.INVALID_REQUEST,
			"At least one field information needs to be added", HttpStatus.NOT_FOUND);

	@InjectMocks
	private UserAdministrationRestController userAdministrationRestController;

	@Before
	public void init() throws Exception {
		MockitoAnnotations.initMocks(this);
		mockMvc = MockMvcBuilders.standaloneSetup(userAdministrationRestController).addFilters(new CORSFilter())
				.build();
	}

	@Test
	public void testsearchAllUsers() throws Exception {
		UserSearchRequest userSearchRequest = new UserSearchRequest();
		userSearchRequest.setAdmin("true");
		userSearchRequest.setCreatedBy("");
		userSearchRequest.setDescending(false);
		userSearchRequest.setDomain("");
		userSearchRequest.setEdit("");
		userSearchRequest.setEmail("");
		userSearchRequest.setExport("");
		userSearchRequest.setFirstName("");
		userSearchRequest.setLastName("");
		userSearchRequest.setPageNumber(1);
		userSearchRequest.setPageSize(10);
		userSearchRequest.setPhone("");
		userSearchRequest.setSortColumn("");
		userSearchRequest.setStatus("");
		userSearchRequest.setUserName("PII");
		userSearchRequest.setExport("Y");
		userSearchRequest.setEdit("Y");
		userSearchRequest.setAdmin("Y");

		UserDetailsResponse userDetailsResponse = new UserDetailsResponse();
		List<UserDetailsResponse> listuserDetails = new ArrayList<>();
		userDetailsResponse.setStatus("active");
		userDetailsResponse.setDomain("Base");
		userDetailsResponse.setUserName("PII");
		userDetailsResponse.setExport("y");
		userDetailsResponse.setAdmin("y");
		userDetailsResponse.setEdit("y");
		listuserDetails.add(userDetailsResponse);

		GenericResponse genericResponse = new GenericResponse();
		genericResponse.setpageNumber(1);
		genericResponse.setpageSize(10);
		genericResponse.setsortColumn("userName");
		genericResponse.settotalNumberOfResults((long) 3);
		genericResponse.setResponse(listuserDetails);

		when(userAdministrationService.getAllUserDetails(any(UserSearchRequest.class))).thenReturn(genericResponse);
		
		ObjectMapper mapper = new ObjectMapper();
		mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
		ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
		String requestJson = ow.writeValueAsString(userSearchRequest);
		mockMvc.perform(post("/api/v1/userSearch").contentType(MediaType.APPLICATION_JSON).content(requestJson))
				.andDo(print()).andExpect(status().isOk())
				.andExpect(content().contentType("application/json;charset=UTF-8"))
				.andExpect(jsonPath("$.pageNumber", is(1))).andExpect(jsonPath("$.pageSize", is(10)))
				.andExpect(jsonPath("$.sortColumn", is("userName")))
				.andExpect(jsonPath("$.totalNumberOfResults", is(3)))
				.andExpect(jsonPath("$.response[0].status", is("active")))
				.andExpect(jsonPath("$.response[0].domain", is("Base")))
				.andExpect(jsonPath("$.response[0].userName", is("PII")))
				.andExpect(jsonPath("$.response[0].export", is("y")))
				.andExpect(jsonPath("$.response[0].admin", is("y"))).andExpect(jsonPath("$.response[0].edit", is("y")));

		verify(userAdministrationService, times(1)).getAllUserDetails(Matchers.refEq(userSearchRequest));
		verifyNoMoreInteractions(userAdministrationService);

	}

	@Test
	public void testEditUserSuccess() throws Exception {
		UserSearchRequest userSearchRequest = new UserSearchRequest();
		userSearchRequest.setAdmin("true");
		userSearchRequest.setCreatedBy("");
		userSearchRequest.setDescending(false);
		userSearchRequest.setDomain("");
		userSearchRequest.setEdit("");
		userSearchRequest.setEmail("");
		userSearchRequest.setExport("");
		userSearchRequest.setFirstName("");
		userSearchRequest.setLastName("");
		userSearchRequest.setPageNumber(1);
		userSearchRequest.setPageSize(10);
		userSearchRequest.setPhone("");
		userSearchRequest.setSortColumn("");
		userSearchRequest.setStatus("");
		userSearchRequest.setUserName("PII");
		userSearchRequest.setExport("Y");
		userSearchRequest.setEdit("Y");
		userSearchRequest.setAdmin("Y");

		when(userAdministrationService.editUserByUserName(userSearchRequest)).thenReturn(true);

		ObjectMapper mapper = new ObjectMapper();
		mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
		ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
		String requestJson = ow.writeValueAsString(userSearchRequest);
		ResultActions andExpect = mockMvc
				.perform(post("/api/v1/editUser").contentType(MediaType.APPLICATION_JSON).content(requestJson))
				.andDo(print()).andExpect(status().is(200));
		logger.info("status results:" + andExpect.andReturn().getResponse().getContentAsString());

		verify(userAdministrationService, times(1)).editUserByUserName(Matchers.refEq(userSearchRequest));
		verifyNoMoreInteractions(userAdministrationService);

	}

	/*@Test
	public void testgetUserByIdSuccess() throws Exception {
		String userId = "100";
		UserDetailsResponse userDetailsResponse = new UserDetailsResponse();
		userDetailsResponse.setStatus("User");
		userDetailsResponse.setDomain("Base");
		userDetailsResponse.setUserName("PII");
		userDetailsResponse.setExport("y");
		userDetailsResponse.setAdmin("y");
		userDetailsResponse.setEdit("y");

		when(userAdministrationService.getUserById(userId)).thenReturn(userDetailsResponse);
		ObjectMapper mapper = new ObjectMapper();
		mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
		ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
		String requestJson = ow.writeValueAsString(userId);
		mockMvc.perform(
				get("/api/v1/userDetails/" + userId).contentType(MediaType.APPLICATION_JSON).content(requestJson))
				.andDo(print()).andExpect(status().isOk())
				.andExpect(jsonPath("$.domain", is("Base")))
				.andExpect(jsonPath("$.userName", is("PII")))
				.andExpect(jsonPath("$.export", is("y")))
				.andExpect(jsonPath("$.admin", is("y")))
				.andExpect(jsonPath("$.edit", is("y")))
				.andExpect(jsonPath("$.status", is("User")));

		verify(userAdministrationService, times(1)).getUserById(Matchers.refEq(userId));
		verifyNoMoreInteractions(userAdministrationService);

	}*/

	@Test
	public void testdeleteUserSuccess() throws Exception {
		Long userId = (long) 100;
		UserDetailsResponse userDetailsResponse = new UserDetailsResponse();
		userDetailsResponse.setStatus("User");
		userDetailsResponse.setDomain("Base");
		userDetailsResponse.setUserName("PII");
		userDetailsResponse.setExport("1");
		userDetailsResponse.setAdmin("!");
		userDetailsResponse.setEdit("2");

		when(userAdministrationService.deleteUserByUserName(userId)).thenReturn(true);
		ObjectMapper mapper = new ObjectMapper();
		mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
		ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
		String requestJson = ow.writeValueAsString(userId);
		mockMvc.perform(
				get("/api/v1/userDelete/" + userId).contentType(MediaType.APPLICATION_JSON).content(requestJson))
				.andDo(print()).andExpect(status().isOk());
		verify(userAdministrationService, times(1)).deleteUserByUserName(Matchers.refEq(userId));
		verifyNoMoreInteractions(userAdministrationService);
	}

}
