package gov.va.med.ars.controller;

import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.when;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Matchers;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.runners.MockitoJUnitRunner;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.SerializationFeature;

import gov.va.med.ars.filter.CORSFilter;
import gov.va.med.ars.model.request.RfaiLookupRequest;
import gov.va.med.ars.model.response.GenericResponse;
import gov.va.med.ars.model.response.RfaiLookUpResponse;
import gov.va.med.ars.service.IRfaiLookupService;

@RunWith(MockitoJUnitRunner.class)
public class RfaiLookupControllerTest {

	private MockMvc mockMvc;
	public static final MediaType APPLICATION_JSON_UTF8 = new MediaType(MediaType.APPLICATION_JSON.getType(),
			MediaType.APPLICATION_JSON.getSubtype(), Charset.forName("utf8"));

	@Mock
	private IRfaiLookupService rfaiLookupService;

	@InjectMocks
	private RfaiLookupController rfaiLookupController;

	@Before
	public void init() throws Exception {
		MockitoAnnotations.initMocks(this);
		mockMvc = MockMvcBuilders.standaloneSetup(rfaiLookupController).addFilters(new CORSFilter()).build();
	}

	@Test
	public void test_getSimpleSearchResults_advSearchFalse() throws Exception {

		RfaiLookupRequest lookupRequest = new RfaiLookupRequest();
		lookupRequest.setPatientIdentifier("999l");

		List<RfaiLookUpResponse> responseList = new ArrayList<>();
		RfaiLookUpResponse response = new RfaiLookUpResponse();
		response.setPatientId("999");
		responseList.add(response);
		GenericResponse simpleSearchResponse = new GenericResponse(1, 1, "sortColumn", 1l);
		simpleSearchResponse.setResponse(responseList);

		when(rfaiLookupService.getRfaiSimpleSearchResults(lookupRequest)).thenReturn(simpleSearchResponse);

		ObjectMapper mapper = new ObjectMapper();
		mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
		ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
		String jsonLookupRequest = ow.writeValueAsString(lookupRequest);

		mockMvc.perform(post("/api/v1/searchRfai/search?advSearch=false").contentType(MediaType.APPLICATION_JSON)
				.content(jsonLookupRequest)).andDo(print()).andExpect(status().isOk());
		// .andExpect(content().contentType(MediaType.APPLICATION_JSON_UTF8_VALUE));

		verify(rfaiLookupService, times(1)).getRfaiSimpleSearchResults(Matchers.refEq(lookupRequest));
		verifyNoMoreInteractions(rfaiLookupService);

	}

	@Test
	public void test_getSimpleSearchResults_noParamsPassed() throws Exception {

		RfaiLookupRequest lookupRequest = new RfaiLookupRequest();
		lookupRequest.setPatientIdentifier("999l");

		List<RfaiLookUpResponse> responseList = new ArrayList<>();
		RfaiLookUpResponse response = new RfaiLookUpResponse();
		response.setPatientId("999");
		responseList.add(response);
		GenericResponse simpleSearchResponse = new GenericResponse(1, 1, "sortColumn", 1l);
		simpleSearchResponse.setResponse(responseList);

		when(rfaiLookupService.getRfaiSimpleSearchResults(lookupRequest)).thenReturn(simpleSearchResponse);

		ObjectMapper mapper = new ObjectMapper();
		mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
		ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
		String jsonLookupRequest = ow.writeValueAsString(lookupRequest);

		mockMvc.perform(
				post("/api/v1/searchRfai/search").contentType(MediaType.APPLICATION_JSON).content(jsonLookupRequest))
				.andDo(print()).andExpect(status().isOk());
		// .andExpect(content().contentType(MediaType.APPLICATION_JSON_UTF8_VALUE));

		verify(rfaiLookupService, times(1)).getRfaiSimpleSearchResults(Matchers.refEq(lookupRequest));
		verifyNoMoreInteractions(rfaiLookupService);

	}

	@Test
	public void test_getSimpleSearchResults_advSearchTrue() throws Exception {

		RfaiLookupRequest lookupRequest = new RfaiLookupRequest();
		lookupRequest.setPatientIdentifier("999l");

		List<RfaiLookUpResponse> responseList = new ArrayList<>();
		RfaiLookUpResponse response = new RfaiLookUpResponse();
		response.setPatientId("999");
		responseList.add(response);
		GenericResponse simpleSearchResponse = new GenericResponse(1, 1, "sortColumn", 1l);
		simpleSearchResponse.setResponse(responseList);

		when(rfaiLookupService.getRfaiAdvResults(lookupRequest)).thenReturn(simpleSearchResponse);

		ObjectMapper mapper = new ObjectMapper();
		mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
		ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
		String jsonLookupRequest = ow.writeValueAsString(lookupRequest);

		mockMvc.perform(post("/api/v1/searchRfai/search?advSearch=true").contentType(MediaType.APPLICATION_JSON)
				.content(jsonLookupRequest)).andDo(print()).andExpect(status().isOk());
		// .andExpect(content().contentType(MediaType.APPLICATION_JSON_UTF8_VALUE));

		verify(rfaiLookupService, times(1)).getRfaiAdvResults(Matchers.refEq(lookupRequest));
		verifyNoMoreInteractions(rfaiLookupService);

	}

}
