package gov.va.med.ars.controller;

import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.when;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.content;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

import java.math.BigDecimal;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.runners.MockitoJUnitRunner;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

import gov.va.med.ars.filter.CORSFilter;
import gov.va.med.ars.model.response.RfaiLineItemResponse;
import gov.va.med.ars.model.response.RfaiResponse;
import gov.va.med.ars.service.IRfaiService;

@RunWith(MockitoJUnitRunner.class)
public class RfaiControllerTest {

	private MockMvc mockMvc;
	public static final MediaType APPLICATION_JSON_UTF8 = new MediaType(MediaType.APPLICATION_JSON.getType(),
			MediaType.APPLICATION_JSON.getSubtype(), Charset.forName("utf8"));

	@Mock
	private IRfaiService rfaiService;

	@InjectMocks
	private RfaiController rfaiController;

	@Before
	public void init() throws Exception {
		MockitoAnnotations.initMocks(this);
		mockMvc = MockMvcBuilders.standaloneSetup(rfaiController).addFilters(new CORSFilter()).build();
	}

	@Test
	public void test_getClaim() throws Exception {

		BigDecimal number = new BigDecimal(9985);
		List<RfaiLineItemResponse> lineItemList = new ArrayList<>();
		RfaiLineItemResponse rfaiLineItemResponse = new RfaiLineItemResponse(10240L, "09/10/2017", "09/13/2017", "12",
				"10240", "GP,RT,AS,T5", number);
		lineItemList.add(rfaiLineItemResponse);

		RfaiResponse rfaiResponse = new RfaiResponse(1L, "Veteran's Affairs", "12115", "10240", "01/16/2018",
				"RIVERSIDEMETHHOS", "1000000009", "111111120", "Holden", "Ines", "000100090", "10000240", "10240",
				"211", "051817242879014", "", "09/10/2017", "09/15/2017", lineItemList);
		// when service is called within controller method, mock it to return
		// RfaiResponse object
		when(rfaiService.populateRfaiInfo(1l)).thenReturn(rfaiResponse);

		mockMvc.perform(get("/api/v1/populateClaim/{id}", (long) 1)).andDo(print()).andExpect(status().isOk())
				.andExpect(content().contentType(MediaType.APPLICATION_JSON_UTF8_VALUE))
				.andExpect(jsonPath("$.payerName", is("Veteran's Affairs")))
				.andExpect(jsonPath("$.payerId", is("12115"))).andExpect(jsonPath("$.informationReceiver", is("10240")))
				.andExpect(jsonPath("$.responseDate", is("01/16/2018")))
				.andExpect(jsonPath("$.providerInformation", is("RIVERSIDEMETHHOS")))
				.andExpect(jsonPath("$.providerNpi", is("1000000009")))
				.andExpect(jsonPath("$.providerTin", is("111111120")))
				.andExpect(jsonPath("$.patientLastname", is("Holden")))
				.andExpect(jsonPath("$.patientFirstName", is("Ines")))
				.andExpect(jsonPath("$.patientIdentifier", is("000100090")))
				.andExpect(jsonPath("$.patientcontrolNumber", is("10000240")))
				.andExpect(jsonPath("$.payerClaimControlNumber", is("10240")))
				.andExpect(jsonPath("$.billType", is("211")))
				.andExpect(jsonPath("$.clearingHouseId", is("051817242879014")))
				.andExpect(jsonPath("$.medicalRecordNumber", is("")))
				.andExpect(jsonPath("$.serviceFromDate", is("09/10/2017")))
				.andExpect(jsonPath("$.serviceToDate", is("09/15/2017")))
				.andExpect(jsonPath("$.rfaiLineItemResponse[0].serviceLineId", is(10240)))
				.andExpect(jsonPath("$.rfaiLineItemResponse[0].serviceFrom", is("09/10/2017")))
				.andExpect(jsonPath("$.rfaiLineItemResponse[0].serviceTo", is("09/13/2017")))
				.andExpect(jsonPath("$.rfaiLineItemResponse[0].revenueCode", is("12")))
				.andExpect(jsonPath("$.rfaiLineItemResponse[0].procedureCode", is("10240")))
				.andExpect(jsonPath("$.rfaiLineItemResponse[0].modifiers", is("GP,RT,AS,T5")))
				.andExpect(jsonPath("$.rfaiLineItemResponse[0].chargeAmount", is(9985)));

		verify(rfaiService, times(1)).populateRfaiInfo((long) 1);
		verifyNoMoreInteractions(rfaiService);
	}

	
	/**
	 * @throws Exception
	 */
/*	@Test(expected = Exception.class)
	public void testGetClaim_Fail() throws Exception {
		long claimIndex = 1L;
		when(rfaiService.populateRfaiInfo(claimIndex)).thenThrow(new gov.va.med.ars.exceptions.GenericException(ErrorMessages.NOT_FOUND, "No claim found with claim id " + claimIndex,
				HttpStatus.NOT_FOUND));

		mockMvc.perform(get("/api/v1/populateClaim/{id}", (long) 1)).andDo(print()).andExpect(status().isNotFound());
		verifyNoMoreInteractions(rfaiController);

	}*/
}
