package gov.va.med.ars.controller;

import static org.hamcrest.Matchers.is;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.anyBoolean;
import static org.mockito.Mockito.when;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.content;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.test.context.web.WebAppConfiguration;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.SerializationFeature;

import gov.va.med.ars.configuration.AppConfig;
import gov.va.med.ars.configuration.spring.SpringMvcConfig;
import gov.va.med.ars.constants.ErrorMessages;
import gov.va.med.ars.exceptions.GenericException;
import gov.va.med.ars.filter.CORSFilter;
import gov.va.med.ars.model.request.SearchReportRequest;
import gov.va.med.ars.model.response.GenericResponse;
import gov.va.med.ars.model.response.ReportResponse;
import gov.va.med.ars.service.IReportService;

@WebAppConfiguration
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(classes = { AppConfig.class, SpringMvcConfig.class })
public class ReportControllerTest {

	private MockMvc mockMvc;
	public static final MediaType APPLICATION_JSON_UTF8 = new MediaType(MediaType.APPLICATION_JSON.getType(),
			MediaType.APPLICATION_JSON.getSubtype(), Charset.forName("utf8"));

	@Mock
	private IReportService reportService;

	@InjectMocks
	private ReportController reportController;

	@Before
	public void init() throws Exception {
		MockitoAnnotations.initMocks(this);
		mockMvc = MockMvcBuilders.standaloneSetup(reportController).addFilters(new CORSFilter()).build();
	}

	@Test
	public void test_searchReports() throws Exception {

		// arrange
		SearchReportRequest searchRequest = new SearchReportRequest();
		searchRequest.setPageNumber(1);
		searchRequest.setPageSize(10);
		searchRequest.setDescending(false);
		searchRequest.setSortColumn("attachmentId");
		searchRequest.setStartDate("03/16/2018");
		searchRequest.setEndDate("03/16/2018");
		searchRequest.setPayerId("121150001");

		List<ReportResponse> responseList = new ArrayList<>();
		ReportResponse report1 = new ReportResponse();
		report1.setAttachmentId(999);
		responseList.add(report1);
		GenericResponse searchResponse = new GenericResponse(1, 10, "attachmentId", 1l);
		searchResponse.setResponse(responseList);

		when(reportService.getReports(any(SearchReportRequest.class), anyBoolean())).thenReturn(searchResponse);

		ObjectMapper mapper = new ObjectMapper();
		mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
		ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
		String jsonSearchRequest = ow.writeValueAsString(searchRequest);

		// act & assert
		mockMvc.perform(post("/api/v1/reports/search?matched=false").contentType(MediaType.APPLICATION_JSON)
				.content(jsonSearchRequest)).andDo(print()).andExpect(status().isOk())
				.andExpect(content().contentType("application/json;charset=UTF-8"))
				.andExpect(jsonPath("$.pageNumber", is(1))).andExpect(jsonPath("$.pageSize", is(10)))
				.andExpect(jsonPath("$.sortColumn", is("attachmentId")))
				.andExpect(jsonPath("$.totalNumberOfResults", is(1)))
				.andExpect(jsonPath("$.response[0].attachmentId", is(999)));

	}

	@Test
	public void test_searchReports_fail() throws Exception {

		// arrange (service)
		GenericException genericException = new GenericException(ErrorMessages.INVALID_REQUEST,
				"At least one field information needs to be added", HttpStatus.NOT_FOUND);

		when(reportService.getReports(any(SearchReportRequest.class), anyBoolean())).thenThrow(genericException);

		// arrange (controller)
		List<ReportResponse> responseList = new ArrayList<>();
		ReportResponse report1 = new ReportResponse();
		report1.setAttachmentId(999);
		responseList.add(report1);
		GenericResponse searchResponse = new GenericResponse(1, 10, "attachmentId", 1l);

		SearchReportRequest searchRequest = new SearchReportRequest();
		searchResponse.setResponse(responseList);

		ObjectMapper mapper = new ObjectMapper();
		mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
		ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
		String jsonSearchRequest = ow.writeValueAsString(searchRequest);

		// act & assert
		mockMvc.perform(post("/api/v1/reports/search?matched=false").content(jsonSearchRequest)).andDo(print())
				.andExpect(status().is(415));

	}

}
