package gov.va.med.ars.controller;

import static org.hamcrest.Matchers.is;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.anyList;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.test.context.web.WebAppConfiguration;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;
import org.springframework.web.util.NestedServletException;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.SerializationFeature;

import gov.va.med.ars.configuration.AppConfig;
import gov.va.med.ars.configuration.spring.SpringMvcConfig;
import gov.va.med.ars.constants.ErrorMessages;
import gov.va.med.ars.exceptions.GenericException;
import gov.va.med.ars.filter.CORSFilter;
import gov.va.med.ars.model.request.CodeAndModifierRequest;
import gov.va.med.ars.model.request.GenericRequest;
import gov.va.med.ars.service.ICodeAndModifierService;
import gov.va.med.domain.ars.LoincCd;

@WebAppConfiguration
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(classes = { AppConfig.class, SpringMvcConfig.class })
public class CodeAndModifiersControllerTest {

	private MockMvc mockMvc;
	public static final MediaType APPLICATION_JSON_UTF8 = new MediaType(MediaType.APPLICATION_JSON.getType(),
			MediaType.APPLICATION_JSON.getSubtype(), Charset.forName("utf8"));

	@Mock
	ICodeAndModifierService codeAndModService;

	@InjectMocks
	private CodeAndModifiersController codesAndModsController;

	@Before
	public void init() throws Exception {
		MockitoAnnotations.initMocks(this);
		mockMvc = MockMvcBuilders.standaloneSetup(codesAndModsController).addFilters(new CORSFilter()).build();
	}

	@SuppressWarnings("unchecked")
	@Test
	public void test_getCodes() throws Exception {

		// arrange (request)
		HttpServletRequest request = mock(HttpServletRequest.class);
		when(request.getRequestURI()).thenReturn("/api/v1/populate/loinc");

		List<String> codesToProcess = new ArrayList<>();
		codesToProcess.add("Y");

		ObjectMapper mapper = new ObjectMapper();
		mapper.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);
		mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
		ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
		String requestBodyJson = ow.writeValueAsString(codesToProcess);

		// arrange (response)
		List<LoincCd> loincCodes = new ArrayList<>();
		LoincCd lcd1 = new LoincCd();
		lcd1.setLoincId(1l);
		lcd1.setStcloincCd("999");
		lcd1.setStcloincCdDesc("test loinc 1 description");
		loincCodes.add(lcd1);

		when(codeAndModService.getCodeAndModifier(any(String.class), anyList())).thenReturn(loincCodes);

		mockMvc.perform(post("/api/v1/populate/loinc").contentType(MediaType.APPLICATION_JSON).content(requestBodyJson))
				.andDo(print()).andExpect(status().isOk()).andExpect(jsonPath("$.[0].loincId", is(1)))
				.andExpect(jsonPath("$.[0].stcloincCd", is("999")))
				.andExpect(jsonPath("$.[0].stcloincCdDesc", is("test loinc 1 description")));

	}

	@SuppressWarnings("unchecked")
	@Test(expected = NestedServletException.class)
	public void test_getCodes_fail() throws Exception {

		// arrange (service)
		GenericException genericException = new GenericException(ErrorMessages.INVALID_REQUEST, "error",
				HttpStatus.NOT_FOUND);

		when(codeAndModService.getCodeAndModifier(any(String.class), anyList())).thenThrow(genericException);

		// arrange (controller)
		HttpServletRequest request = mock(HttpServletRequest.class);
		when(request.getRequestURI()).thenReturn("/api/v1/populate/loinc");

		List<String> codesToProcess = new ArrayList<>();
		codesToProcess.add("Y");

		ObjectMapper mapper = new ObjectMapper();
		mapper.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);
		mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
		ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
		String requestBodyJson = ow.writeValueAsString(codesToProcess);

		// act & assert
		mockMvc.perform(post("/api/v1/populate/loinc").contentType(MediaType.APPLICATION_JSON).content(requestBodyJson))
				.andDo(print()).andExpect(status().is(415));

	}

	@Test
	public void test_addOrModifyCodes() throws Exception {

		// arrange (request)
		HttpServletRequest request = mock(HttpServletRequest.class);
		when(request.getRequestURI()).thenReturn("/api/v1/populate/add");

		List<CodeAndModifierRequest> hccList = new ArrayList<>();
		CodeAndModifierRequest codeAndModReq = new CodeAndModifierRequest();
		codeAndModReq.setCd(1l);
		codeAndModReq.setFlag(true);
		codeAndModReq.setStcCd("hccs_1");
		codeAndModReq.setStcCdDesc("hccs_1_desc");
		hccList.add(codeAndModReq);

		GenericRequest requestBody = new GenericRequest();
		requestBody.setHccList(hccList);

		ObjectMapper mapper = new ObjectMapper();
		mapper.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);
		mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
		ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
		String requestBodyJson = ow.writeValueAsString(requestBody);

		// arrange (response)
		List<String> dupCheck = new ArrayList<>();

		when(codeAndModService.checkSubmittedModificationsForDupes(any(GenericRequest.class))).thenReturn(dupCheck);
		when(codeAndModService.addOrModifyCodeAndModifier(any(GenericRequest.class))).thenReturn(true);

		mockMvc.perform(post("/api/v1/populate/add").contentType(MediaType.APPLICATION_JSON).content(requestBodyJson))
				.andDo(print()).andExpect(status().isOk()).andExpect(jsonPath("$.result", is(true)));

	}
}
