package gov.va.med.ars.controller;

import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.when;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.put;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.content;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.http.MediaType;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.test.context.web.WebAppConfiguration;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.SerializationFeature;

import gov.va.med.ars.configuration.AppConfig;
import gov.va.med.ars.configuration.spring.SpringMvcConfig;
import gov.va.med.ars.filter.CORSFilter;
import gov.va.med.ars.model.request.EmailTemplateRequest;
import gov.va.med.ars.service.IArsEmailTemplateService;


@WebAppConfiguration
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(classes = { AppConfig.class, SpringMvcConfig.class })
public class ArsEmailTemplateControllerTest {
	
	private MockMvc mockMvc;
	public static final MediaType APPLICATION_JSON_UTF8 = new MediaType(MediaType.APPLICATION_JSON.getType(),
			MediaType.APPLICATION_JSON.getSubtype(), Charset.forName("utf8"));

	@Mock
	private IArsEmailTemplateService iArsEmailTemplateService;

	@InjectMocks
	private ArsEmailTemplateController arsEmailTemplateController;

	@Before
	public void init() throws Exception {
		MockitoAnnotations.initMocks(this);
		mockMvc = MockMvcBuilders.standaloneSetup(arsEmailTemplateController).addFilters(new CORSFilter()).build();
	}
	
	@Test 
	public void testGetEmailTemplate() throws Exception {
		List<EmailTemplateRequest> emailTemplates = new ArrayList<EmailTemplateRequest>();
		EmailTemplateRequest templateRes = new EmailTemplateRequest();
		templateRes.setHeading("This is a pending template heading.");
		templateRes.setMessage("This is a test message.");
		templateRes.setTemplateName("Pending");
		templateRes.setTemplateType("Pending Type");
		emailTemplates.add(templateRes);
		
		when(iArsEmailTemplateService.getEmailTemplate()).thenReturn(emailTemplates);
		
		mockMvc.perform(get("/api/v1/email/templates")).andDo(print()).andExpect(status().isOk())
		.andExpect(content().contentType(MediaType.APPLICATION_JSON_UTF8_VALUE))
		.andExpect(jsonPath("$.emailTemplates[0].heading", is("This is a pending template heading.")))
		.andExpect(jsonPath("$.emailTemplates[0].templateName", is("Pending")))
		.andExpect(jsonPath("$.emailTemplates[0].templateType", is("Pending Type")))
		.andExpect(jsonPath("$.emailTemplates[0].message", is("This is a test message.")));
		
		verify(iArsEmailTemplateService, times(1)).getEmailTemplate();
		verifyNoMoreInteractions(iArsEmailTemplateService);		
	}
	
	@Test
	public void testUpdateEmailTemplate() throws Exception {
		
		EmailTemplateRequest request = new EmailTemplateRequest();
		request.setHeading("This is a pending template heading.");
		request.setMessage("This is a message.");
		request.setTemplateName("Pending");
		request.setTemplateType("Pending Type");
		
		Boolean response = Boolean.TRUE;
		when(iArsEmailTemplateService.updateEmailTemplate(request)).thenReturn(response);
		
		mockMvc.perform(put("/api/v1/email/updateTemplate").contentType(MediaType.APPLICATION_JSON)
				.content(getRequestJson(request))).andDo(print());
		verify(iArsEmailTemplateService, times(1)).updateEmailTemplate(request);
	}
	
	private String getRequestJson(Object request) {
		String requestJson = null;
		try {
			ObjectMapper mapper = new ObjectMapper();
		    mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
		    ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
			requestJson = ow.writeValueAsString(request);
			System.out.println("requestJson - " + requestJson);
		
		} catch (JsonProcessingException e) {
			e.printStackTrace();
		}
		return requestJson;
	}	
}