package gov.va.med.ewv.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import gov.va.med.domain.fee.EwvClaims;
import gov.va.med.domain.fee.EwvSlProvider;
import gov.va.med.domain.fee.EwvSvcLineTeeth;
import gov.va.med.domain.fee.EwvServiceLineOhiAdjtmts;
import gov.va.med.domain.fee.EwvServiceLines;
import gov.va.med.ewv.util.ProviderType;
import gov.va.med.ewv.model.response.Provider;
import gov.va.med.domain.fee.EwvServiceLinesDrugs;
import gov.va.med.ewv.util.OhiType;
import gov.va.med.domain.fee.EwvServiceLineOhis;
import gov.va.med.domain.fee.EwvClaimOhis;
import gov.va.med.domain.fee.EwvClaimOhis;
import gov.va.med.ewv.model.response.AmbulanceCert;
import gov.va.med.ewv.model.response.DrugLine;
import gov.va.med.ewv.model.response.Provider;
import gov.va.med.ewv.model.response.ServiceLineOhiAdjustment;
import gov.va.med.ewv.model.response.ToothLine;
import gov.va.med.ewv.util.AdjustmentType;
import gov.va.med.ewv.util.DrugCategory_5010;
import gov.va.med.ewv.util.EwvUtils;


public class DTOBase
{
    public List<Provider> createProviders(Set<EwvSlProvider> EwvSlProvider)
    {
        List<Provider> providers = new ArrayList<Provider>();
        
        for (EwvSlProvider prov : EwvSlProvider)
        {
            String name = EwvUtils.defaultToSpace(prov.getProviderLname()) 
                          + ", " 
                          + EwvUtils.defaultToSpace(prov.getProviderFname())
                          + " "
                          + EwvUtils.defaultToSpace(prov.getProviderMname());
            String postalcode_5010 = prov.getPostalCode5010();
            
            Provider provider = new Provider(ProviderType.typeOf(prov.getProviderType()),
                                             prov.getProviderId(),
                                             prov.getPostalCode5010(),
                                             name,
                                             postalcode_5010, 
                                             prov.getTaxonomyCode());
            
            providers.add(provider);
        }
        
        return providers;
    }
    
    public List<ToothLine> createToothLines(Set<EwvSvcLineTeeth> toothLineSet)
    {
        List<ToothLine> returnTeeth = new ArrayList<ToothLine>();
        
        for (EwvSvcLineTeeth tooth : toothLineSet)
        {
            String toothNumber = tooth.getToothNumber();
            String surface1 = tooth.getSurface1();
            String surface2 = tooth.getSurface2();
            String surface3 = tooth.getSurface3();
            String surface4 = tooth.getSurface4();
            String surface5 = tooth.getSurface5();
            
            ToothLine teethLine = new ToothLine(toothNumber,
                                                surface1,
                                                surface2, 
                                                surface3,
                                                surface4,
                                                surface5);
            
            returnTeeth.add(teethLine);
        }
        
        // order the teeth list by tooth number
        Collections.sort(returnTeeth);
        
        return returnTeeth;
    }
    
    public String getQuantityOrType(EwvServiceLines line)
    {
        return EwvUtils.defaultToSpace(line.getQuantity()) + " " + EwvUtils.defaultToSpace(line.getType());
    }
    
    public List<DrugLine> createDrugLines(Set<EwvServiceLinesDrugs> listOfDrugLines)
    {
        List<DrugLine> drugs = new ArrayList<DrugLine>();
        
        for (EwvServiceLinesDrugs drug : listOfDrugLines)
        {
            String drugRx = drug.getDrugRx();
            String drugRxNdcCode = drug.getDrugRxNdcCode();
            String drugRxQuantityPlusType = EwvUtils.defaultToSpace(drug.getDrugRxQuantity())
                                          + " " + EwvUtils.defaultToSpace(drug.getDrugRxType());
            String drugRxQuantity = drug.getDrugRxQuantity();
            String drugRxType = drug.getDrugRxType();
            String drugRxCharge = drug.getDrugRxCharge();
            String drugCategoryString = drug.getDrugRxCategory5010();
            DrugCategory_5010 drugCategory_5010 = DrugCategory_5010.typeOf(drugCategoryString);
            
            DrugLine line = new DrugLine(drugRx,
            		                     drugRxNdcCode,
            		                     drugRxQuantityPlusType,
            		                     drugRxQuantity,
            		                     drugRxType,
            		                     drugRxCharge,
            		                     drugCategory_5010);
            
            drugs.add(line);
        }
        
        return drugs;
    }


    public List<AmbulanceCert> populateAmbulanceCert(List<AmbulanceCert> certs,
                                                        String certIndicator,
                                                        String... conditionCodesArray)
    {
      // Start up a new cert
      AmbulanceCert newCert = new AmbulanceCert();
      
      // Fill its fields
      newCert.setCertIndicator(certIndicator);
      String conditionCodes = EwvUtils.concatWithSeparator(", ", conditionCodesArray);
      newCert.setConditionCodes(conditionCodes);
      
      // If there is any data there, add it to the incoming
      if (!newCert.isEmpty()) {
        certs.add(newCert);
      }
      
      return certs;
    }
    
    public OhiType determineOhiType(EwvServiceLineOhis lineOhi, EwvClaims claim) {
      OhiType returnType = null;
      
      // 4010 Phase 1 - get the first adjudication for this ohi, and use it to determine type
      EwvServiceLineOhiAdjtmts firstAdjustment = null;
      //OhiType payerType = null;
      
      Set<EwvServiceLineOhiAdjtmts> adjustments = lineOhi.getEwvServiceLineOhiAdjtmtses();
      if (adjustments != null && !adjustments.isEmpty()) {
        for (EwvServiceLineOhiAdjtmts nextAdjustment : adjustments) {
          firstAdjustment = nextAdjustment;
          break;
        }
        String payerId = firstAdjustment.getPrimaryPayerId();
        returnType = determinePayerType(payerId, claim);
      
      }

      // 4010 Phase 2 - just use the field on the service line ohi
      if (returnType == null) {
        returnType = OhiType.PRIMARY;
      }
      
      return returnType;
    }

    public OhiType determinePayerType(String payerId, EwvClaims claim) {
      OhiType returnType = null;
      
      // Go through the claim OHIs, looking for the payerId sent in
      Set<EwvClaimOhis> claimOhis = claim.getEwvClaimOhises();
      for (EwvClaimOhis claimOhi : claimOhis) {
        String thisPayerId = claimOhi.getOhiPayerId();
        if (thisPayerId != null && thisPayerId.equals(payerId)) {
          String typeIndicator = claimOhi.getPaymentSequenceIndicator();
          if (typeIndicator != null) {
            returnType = OhiType.typeOf(typeIndicator);
          }
        }
      }
      
      return returnType;
    }

    public List<ServiceLineOhiAdjustment> populateServiceLineAdjustments
      (Set<EwvServiceLineOhiAdjtmts> adjustments) {
      List<ServiceLineOhiAdjustment> returnAdjustments = new ArrayList<ServiceLineOhiAdjustment>();
      
      // Go through the adjustments
      for (EwvServiceLineOhiAdjtmts nextAdjustment : adjustments) {
        ServiceLineOhiAdjustment newAdjustment = createOhiServiceLineAdjustment(nextAdjustment);
        if (newAdjustment != null) {
          returnAdjustments.add(newAdjustment);
        }
      }
      
      return returnAdjustments;
    }

    private ServiceLineOhiAdjustment createOhiServiceLineAdjustment(EwvServiceLineOhiAdjtmts adj)
    {
        //String payerId = adj.getPrimaryPayerId();
        String descriptionLong = adj.getDescription();
        String descriptionShort = EwvUtils.shorten(descriptionLong,15);
        String reasonCode = adj.getSvcLineAdjustmtReasonCode();
        String quantity = adj.getQuantity();
        String coAmount = adj.getCoAmt();
        String crAmount = adj.getCrAmt();
        String oaAmount = adj.getOaAmt();
        String piAmount = adj.getPiAmt();
        String prAmount = adj.getPrAmt();
        
        ServiceLineOhiAdjustment ret
          = new ServiceLineOhiAdjustment(descriptionLong, 
                                         descriptionShort,
                                         reasonCode,
                                         quantity, 
                                         coAmount,
                                         crAmount, 
                                         oaAmount,
                                         piAmount,
                                         prAmount);
        
        if (coAmount != null)
        {
            ret.setAdjustmentType(AdjustmentType.CO);
        }
        else if (crAmount != null)
        {
            ret.setAdjustmentType(AdjustmentType.CR);
        }
        else if (oaAmount != null)
        {
            ret.setAdjustmentType(AdjustmentType.OA);
        }
        else if (piAmount != null)
        {
            ret.setAdjustmentType(AdjustmentType.PI);
        }
        else if (prAmount != null)
        {
            ret.setAdjustmentType(AdjustmentType.PR);
        }
        
        return ret;
    }
    
}
