package gov.va.med.ewv.service.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.stereotype.Service;

import gov.va.med.domain.fee.EwvAmbulanceInfo5010;
import gov.va.med.domain.fee.EwvClaims;
import gov.va.med.domain.fee.EwvServiceLineOhis;
import gov.va.med.domain.fee.EwvServiceLines;
import gov.va.med.domain.fee.EwvSvcLineDme;
import gov.va.med.ewv.exceptions.NonUniqueRecordsException;
import gov.va.med.ewv.model.response.AmbulanceCert;
import gov.va.med.ewv.model.response.AmbulanceInfo;
import gov.va.med.ewv.model.response.DrugLine;
import gov.va.med.ewv.model.response.MedicalEquipmentLine;
import gov.va.med.ewv.model.response.Provider;
import gov.va.med.ewv.model.response.ServiceLineOhiAdjustment;
import gov.va.med.ewv.model.response.ToothLine;
import gov.va.med.ewv.service.IEwvAmbulanceAndServiceLinesInfoService;
import gov.va.med.ewv.util.ClaimType;
import gov.va.med.ewv.util.DTOBase;
import gov.va.med.ewv.util.EwvUtils;
import gov.va.med.ewv.util.OhiType;
import gov.va.med.ewv.util.ServiceLine;

@Service
public class EwvBillingAndServiceLinesInfoServiceImpl extends DTOBase
		implements IEwvAmbulanceAndServiceLinesInfoService {
	private static final Logger logger = LogManager.getLogger(EwvBillingAndServiceLinesInfoServiceImpl.class);

	@Override
	public List<ServiceLine> populateServiceLines(EwvClaims claim) throws NonUniqueRecordsException {
		logger.debug("claim.getGuid() - " + claim.getGuid());
		
		List<EwvServiceLines> serviceLineList = new ArrayList<>(claim.getEwvServiceLineses());

		// Need to check in the model for the equals and hashcode methods
		// Collections.sort(serviceLineList);
		List<ServiceLine> listOfServiceLines = new ArrayList<>();

		for (EwvServiceLines line : serviceLineList) {
			ServiceLine bean = populateServiceLine(line, claim);
			listOfServiceLines.add(bean);
		}

		return listOfServiceLines;
	}

	public String getQuantityOrType(EwvServiceLines line) {
		return EwvUtils.defaultToSpace(line.getQuantity()) + " " + EwvUtils.defaultToSpace(line.getType());
	}

	@Override
	public ServiceLine populateServiceLine(EwvServiceLines line, EwvClaims claim) throws NonUniqueRecordsException {
		String dateOfService = null;
		String beginDateOfService = null;
		String endDateOfService = null;
		String revCode = line.getRevenueCode();
		String procedureCode = line.getProcedureCode();

		String ndcCode = line.getNdc();
		String modifiers = EwvUtils.combineModifiers(", ", line.getModifier1(), line.getModifier2(),
				line.getModifier3(), line.getModifier4());
		String quantityOrType = getQuantityOrType(line);

		// Added quantity and type for phase II
		String quantity = line.getQuantity();
		String type = line.getType();

		String serviceTax = line.getServiceTax();
		String facilityTax = line.getFacilityTax();
		String charge = line.getCharge();

		String lineNumber = String.valueOf(line.getLineNumber());

		List<DrugLine> listOfDrugs = null;
		List<ToothLine> listOfTeeth = null;
		List<MedicalEquipmentLine> listOfEquipment = null;
		List<Provider> listOfProviderInfo = null;

		String dxDtPointers = EwvUtils.combineModifiers(", ", line.getDx1(), line.getDx2(), line.getDx3(),
				line.getDx4());
		String salesTax = line.getSalesTax();
		String pos = line.getPos();

		// Begin and End, new for Phase II, do not depend on claim type
		beginDateOfService = line.getBeginDateOfService();
		endDateOfService = line.getEndDateOfService();

		// 5010 Stuff
		AmbulanceInfo serviceLineLevelAmbulanceInfo = populateServiceLineLevelAmbulanceInfo(line);

		ClaimType claimType = ClaimType.typeOf(claim.getClaimType());
		if (claimType.equals(ClaimType.INSTITUTIONAL)) {
			dateOfService = EwvUtils.formatDateRangeEWV2(line.getBeginDateOfService(), line.getEndDateOfService(),
					" - ", true);
			// drugs accur in prof lines or inst
			listOfDrugs = createDrugLines(line.getEwvServiceLinesDrugses());//ewvServiceLinesDrugsRepository.getEwvServiceLinesDrugs(line.getGuid()));
			listOfProviderInfo = createProviders(line.getEwvSlProviders());//ewvSlProviderRepository.getEwvSlProviders(line.getGuid()));
		} else if (claimType.equals(ClaimType.PROFESSIONAL)) {
			dateOfService = EwvUtils.formatDateRangeEWV2(line.getBeginDateOfService(), line.getEndDateOfService(),
					" - ", true);

			// drugs accur in prof lines or inst
			listOfDrugs = createDrugLines(line.getEwvServiceLinesDrugses());//ewvServiceLinesDrugsRepository.getEwvServiceLinesDrugs(line.getGuid()));
			// medical equipment occurs only on professional svc lines
			listOfEquipment = createMedicalEquipmentLines(line.getEwvSvcLineDmes());//ewvSvcLineDmeRepository.getEwvSvcLineDme(line.getGuid()));
			listOfProviderInfo = createProviders(line.getEwvSlProviders());//ewvSlProviderRepository.getEwvSlProviders(line.getGuid()));
		} else if (claimType.equals(ClaimType.DENTAL)) {
			// dental is it's own
			dateOfService = line.getBeginDateOfService();
			listOfTeeth = createToothLines(line.getEwvSvcLineTeeths());//ewvSvcLineTeethRepository.getEwvSvcLineTeeth(line.getGuid()));
			listOfProviderInfo = createProviders(line.getEwvSlProviders());//ewvSlProviderRepository.getEwvSlProviders(line.getGuid()));
		}

		// Put together service line Ohi stuff
		String primaryPayerPaidAmount = null;
		List<ServiceLineOhiAdjustment> primaryOhiServiceLineAdjustments = null;
		String secondaryPayerPaidAmount = null;
		List<ServiceLineOhiAdjustment> secondaryOhiServiceLineAdjustments = null;

		// Go through the ohis for this service line
		Set<EwvServiceLineOhis> serviceLineOhis = line.getEwvServiceLineOhises();//ewvServiceLineOhisRepository.getEwvServiceLineOhis(line.getGuid());
		for (EwvServiceLineOhis lineOhi : serviceLineOhis) {
			// Is this a primary, secondary, or tertiary OHI?
			OhiType ohiType = determineOhiType(lineOhi, claim);

			// Check for primary
			if (ohiType.equals(OhiType.PRIMARY)) {
				primaryPayerPaidAmount = lineOhi.getOhiPaid();
				primaryOhiServiceLineAdjustments = populateServiceLineAdjustments(lineOhi.getEwvServiceLineOhiAdjtmtses());
						//ewvServiceLineOhiAdjtmtRepository.getEwvServiceLineOhiAdjtmts(lineOhi.getGuid()));
			}

			// Check for secondary
			else if (ohiType.equals(OhiType.SECONDARY)) {
				secondaryPayerPaidAmount = lineOhi.getOhiPaid();
				secondaryOhiServiceLineAdjustments = populateServiceLineAdjustments(lineOhi.getEwvServiceLineOhiAdjtmtses());
						//ewvServiceLineOhiAdjtmtRepository.getEwvServiceLineOhiAdjtmts(lineOhi.getGuid()));
			}

		}

		ServiceLine newServiceLine = new ServiceLine(dateOfService, beginDateOfService, endDateOfService, revCode,
				procedureCode, ndcCode, modifiers, quantityOrType, quantity, type, primaryPayerPaidAmount,
				primaryOhiServiceLineAdjustments, secondaryPayerPaidAmount, secondaryOhiServiceLineAdjustments,
				serviceTax, facilityTax, charge, lineNumber, listOfDrugs, listOfTeeth, listOfEquipment, dxDtPointers,
				salesTax, listOfProviderInfo, pos, serviceLineLevelAmbulanceInfo);

		return newServiceLine;
	}

	@Override
	public AmbulanceInfo populateServiceLineLevelAmbulanceInfo(EwvServiceLines serviceLineBean)
			throws NonUniqueRecordsException {
		AmbulanceInfo returnInfo = null;

		// Get the ambulance info from the service line
		Set<EwvAmbulanceInfo5010> ambulanceInfoSet = serviceLineBean.getEwvAmbulanceInfo5010s();
				//ewvAmbulanceInfo5010Repository.getEwvAmbulanceInfo5010s(serviceLineBean.getGuid());

		// There will be no more than one entry in the set
		int ambulanceLineCounter = 0;
		for (EwvAmbulanceInfo5010 ambulanceInfo : ambulanceInfoSet) {
			ambulanceLineCounter++;
			if (ambulanceLineCounter > 1) {
				throw new NonUniqueRecordsException("Too many service-line-level ambulance lines");
			}

			returnInfo = new AmbulanceInfo();

			// Pickup Address
			returnInfo.setPickupAddressLine1(ambulanceInfo.getPickupAddressLine1());
			returnInfo.setPickupAddressLine2(ambulanceInfo.getPickupAddressLine2());
			returnInfo.setPickupCity(ambulanceInfo.getPickupCity());
			returnInfo.setPickupState(ambulanceInfo.getPickupState());
			returnInfo.setPickupZip(ambulanceInfo.getPickupZip());

			// Dropoff Address
			returnInfo.setDropoffAddressLine1(ambulanceInfo.getDropoffAddressLine1());
			returnInfo.setDropoffAddressLine2(ambulanceInfo.getDropoffAddressLine2());
			returnInfo.setDropoffCity(ambulanceInfo.getDropoffCity());
			returnInfo.setDropoffState(ambulanceInfo.getDropoffState());
			returnInfo.setDropoffZip(ambulanceInfo.getDropoffZip());

			// Transport Fields
			returnInfo.setTransportReasonCode(ambulanceInfo.getTransportReasonCode());
			returnInfo.setTransportDistance(EwvUtils.formatAsInteger(ambulanceInfo.getTransportDistance()));

			returnInfo.setRoundTripPurpose(ambulanceInfo.getRoundTripPurpose());
			returnInfo.setRoundTripPurposeShort(EwvUtils.shorten(returnInfo.getRoundTripPurpose(), 50));
			returnInfo.setUseRoundTripPurposeModalPanel(returnInfo.getRoundTripPurposeShort().endsWith("..."));
		
			returnInfo.setStretcherPurpose(ambulanceInfo.getStretcherPurpose());
			returnInfo.setStretcherPurposeShort(EwvUtils.shorten(returnInfo.getStretcherPurpose(), 50));
			returnInfo.setUseStretcherPurposeModalPanel(returnInfo.getStretcherPurposeShort().endsWith("..."));

			// // Transport Fields
			// returnInfo.setTransportReasonCode(ambulanceInfo.getTransportReasonCode());
			// returnInfo.setRoundTripPurpose(ambulanceInfo.getRoundTripPurpose());
			// returnInfo.setTransportDistance(ambulanceInfo.getTransportDistance());
			// returnInfo.setStretcherPurpose(ambulanceInfo.getStretcherPurpose());
			//
			// Three possible Certs are slammed together into a single database
			// row
			List<AmbulanceCert> certs = new ArrayList<AmbulanceCert>();

			// Cert #1
			populateAmbulanceCert(certs, ambulanceInfo.getCertIndicator1(), ambulanceInfo.getCertIndicator1(),
					ambulanceInfo.getConditionCode12(), ambulanceInfo.getConditionCode13(),
					ambulanceInfo.getConditionCode14(), ambulanceInfo.getConditionCode15());

			// Cert #2
			populateAmbulanceCert(certs, ambulanceInfo.getCertIndicator2(), ambulanceInfo.getConditionCode21(),
					ambulanceInfo.getConditionCode22(), ambulanceInfo.getConditionCode23(),
					ambulanceInfo.getConditionCode24(), ambulanceInfo.getConditionCode25());

			// Cert #2
			populateAmbulanceCert(certs, ambulanceInfo.getCertIndicator3(), ambulanceInfo.getConditionCode31(),
					ambulanceInfo.getConditionCode32(), ambulanceInfo.getConditionCode33(),
					ambulanceInfo.getConditionCode34(), ambulanceInfo.getConditionCode35());

			// Attach the certs to the ambulance info, if there are any certs
			if (certs.size() > 0) {
				returnInfo.setCerts(certs);
			}
		}

		return returnInfo;
	}

	@Override
	public List<MedicalEquipmentLine> createMedicalEquipmentLines(Set<EwvSvcLineDme> list) {
		List<MedicalEquipmentLine> equips = new ArrayList<MedicalEquipmentLine>();

		for (EwvSvcLineDme dme : list) {
			String lengthOfMedicalNecessity = dme.getDmeLengthMedicalNecessity();
			String frequency = dme.getDmeFrequency();
			String rentalAmt = dme.getDmeRentalAmt();
			String purchaseAmt = dme.getDmePurchaseAmt();

			equips.add(new MedicalEquipmentLine(lengthOfMedicalNecessity, frequency, rentalAmt, purchaseAmt));
		}

		return equips;
	}

	@Override
	public AmbulanceInfo populateClaimLevelAmbulanceInfo(EwvClaims claimBean) throws NonUniqueRecordsException {
		AmbulanceInfo returnInfo = null;
		boolean okToRender = false;

		// Get the ambulance info from the claim
		Set<EwvAmbulanceInfo5010> ambulanceInfoSet = claimBean.getEwvAmbulanceInfo5010s();
				//ewvAmbulanceInfo5010Repository.getEwvAmbulanceInfo5010s(claimBean.getGuid());

		// There will be no more than one entry in the set
		int ambulanceLineCounter = 0;
		for (EwvAmbulanceInfo5010 ambulanceInfo : ambulanceInfoSet) {
			ambulanceLineCounter++;
			if (ambulanceLineCounter > 1) {
				throw new NonUniqueRecordsException("Too many claim-level ambulance lines");
			}

			returnInfo = new AmbulanceInfo();

			// Pickup Address
			returnInfo.setPickupAddressLine1(ambulanceInfo.getPickupAddressLine1());
			returnInfo.setPickupAddressLine2(ambulanceInfo.getPickupAddressLine2());
			returnInfo.setPickupCity(ambulanceInfo.getPickupCity());
			returnInfo.setPickupState(ambulanceInfo.getPickupState());
			returnInfo.setPickupZip(ambulanceInfo.getPickupZip());

			// Dropoff Address
			returnInfo.setDropoffAddressLine1(ambulanceInfo.getDropoffAddressLine1());
			returnInfo.setDropoffAddressLine2(ambulanceInfo.getDropoffAddressLine2());
			returnInfo.setDropoffCity(ambulanceInfo.getDropoffCity());
			returnInfo.setDropoffState(ambulanceInfo.getDropoffState());
			returnInfo.setDropoffZip(ambulanceInfo.getDropoffZip());

			// Transport Fields
			returnInfo.setTransportReasonCode(ambulanceInfo.getTransportReasonCode());
			returnInfo.setTransportDistance(EwvUtils.formatAsInteger(ambulanceInfo.getTransportDistance()));

			returnInfo.setRoundTripPurpose(ambulanceInfo.getRoundTripPurpose());
			returnInfo.setRoundTripPurposeShort(EwvUtils.shorten(returnInfo.getRoundTripPurpose(), 50));
			returnInfo.setUseRoundTripPurposeModalPanel(returnInfo.getRoundTripPurposeShort().endsWith("..."));
			;

			returnInfo.setStretcherPurpose(ambulanceInfo.getStretcherPurpose());
			returnInfo.setStretcherPurposeShort(EwvUtils.shorten(returnInfo.getStretcherPurpose(), 50));
			returnInfo.setUseStretcherPurposeModalPanel(returnInfo.getStretcherPurposeShort().endsWith("..."));

			// Three possible Certs are slammed together into a single database
			// row
			List<AmbulanceCert> certs = new ArrayList<AmbulanceCert>();

			// Cert #1
			populateAmbulanceCert(certs, ambulanceInfo.getCertIndicator1(), ambulanceInfo.getConditionCode11(),
					ambulanceInfo.getConditionCode12(), ambulanceInfo.getConditionCode13(),
					ambulanceInfo.getConditionCode14(), ambulanceInfo.getConditionCode15());

			// Cert #2
			populateAmbulanceCert(certs, ambulanceInfo.getCertIndicator2(), ambulanceInfo.getConditionCode21(),
					ambulanceInfo.getConditionCode22(), ambulanceInfo.getConditionCode23(),
					ambulanceInfo.getConditionCode24(), ambulanceInfo.getConditionCode25());

			// Cert #2
			populateAmbulanceCert(certs, ambulanceInfo.getCertIndicator3(), ambulanceInfo.getConditionCode31(),
					ambulanceInfo.getConditionCode32(), ambulanceInfo.getConditionCode33(),
					ambulanceInfo.getConditionCode34(), ambulanceInfo.getConditionCode35());

			// Attach the certs to the ambulance info, if there are any certs
			if (certs.size() > 0) {
				returnInfo.setCerts(certs);
			}
		}

		return returnInfo;
	}

}
