package gov.va.med.ars.service.impl;


import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import gov.va.med.ars.constants.ErrorMessages;
import gov.va.med.ars.dao.ars.IArsRole;
import gov.va.med.ars.dao.ars.IArsUserRepository;
import gov.va.med.ars.dao.ars.IArsUserRoleUsages;
import gov.va.med.ars.exceptions.GenericException;
import gov.va.med.ars.model.response.GenericResponse;
import gov.va.med.ars.model.response.UserDetailsResponse;
import gov.va.med.ars.model.request.UserSearchRequest;
import gov.va.med.ars.service.IUserAdministrationService;
import gov.va.med.domain.ars.ArsRole;
import gov.va.med.domain.ars.ArsUser;
import gov.va.med.domain.ars.ArsUserRoleUsages;
import gov.va.med.domain.ars.ArsUserRoleUsagesId;

@Service
public class UserAdministrationServiceImpl implements IUserAdministrationService {

	private static final Logger logger = LogManager.getLogger(UserAdministrationServiceImpl.class);

	@Autowired
	IArsUserRepository arsUserRepository;
	@Autowired
	IArsUserRoleUsages arsUserRoleUsages;
	@Autowired
	IArsRole arsRole;

	@Override
	public UserDetailsResponse getUserById(String userId) {
		ArsUser userDetail = arsUserRepository.findByUserNameIgnoreCase(userId);
		UserDetailsResponse response = null ;
		if (userDetail == null) {
			return response;
		}
		response= new UserDetailsResponse();
		response.setUserId(userDetail.getArsUserId());
		response.setUserName(userDetail.getUserName());
		response.setDomain(userDetail.getDomain());

		if (userDetail.getEnabled().equalsIgnoreCase("Y"))
			response.setStatus("Active");
		else
			response.setStatus("Inactive");

		if (userDetail.getEnableAdmin())
			response.setAdmin("Yes");
		else
			response.setAdmin("No");

		if (userDetail.getEnableEdit())
			response.setEdit("Yes");
		else
			response.setEdit("No");

		if (userDetail.getEnableExport())
			response.setExport("Yes");
		else
			response.setExport("No");

		response.setFirstName(userDetail.getFirstName());
		response.setLastName(userDetail.getLastName());
		response.setEmail(userDetail.getEmailAddress());
		response.setPhone(userDetail.getPhoneNumber());
		response.setCreatedBy(userDetail.getCreatedBy());
		response.setModifiedBy(userDetail.getModifiedBy());
		return response;
	}

	@Override
	public GenericResponse getAllUserDetails(UserSearchRequest userSearchRequest) throws GenericException {
		GenericResponse searchResponse = null;
		PageRequest pageable = generatePagination(userSearchRequest);

		try {
			Long countElements = arsUserRepository.count(new Specification<ArsUser>() {

				@Override
				public Predicate toPredicate(Root<ArsUser> root, CriteriaQuery<?> query, CriteriaBuilder cb) {
					List<Predicate> predicates = new ArrayList<>();
					
					getQueryData(userSearchRequest, root, cb, predicates);
					
					return cb.and(predicates.toArray(new Predicate[0]));
				}

			});
			logger.info("The result count is " + countElements);
			int pageNumber = userSearchRequest.getPageNumber();
			pageNumber = resetPageNumbersOnCountElements(userSearchRequest, countElements, pageNumber);
			if (countElements > 0) {
				logger.info("Result Count : " + countElements);
				searchResponse = setGenericJsonResponse(userSearchRequest, pageable, countElements, pageNumber);
			} else {
				logger.info("No data found");
				GenericResponse emptyResponse = new GenericResponse(userSearchRequest.getPageNumber(),
						userSearchRequest.getPageSize(), userSearchRequest.getSortColumn(), countElements);
				return emptyResponse;
			}

		} catch (Exception e) {
			throw new GenericException(ErrorMessages.DATA_ACCESS_ERROR, e.getMessage(),
					HttpStatus.INTERNAL_SERVER_ERROR);
		}
		return searchResponse;
	}

	private void getQueryData(UserSearchRequest userSearchRequest, Root<ArsUser> root, CriteriaBuilder cb,
			List<Predicate> predicates) {		
		logger.info("The vale of include inactive active users : "+userSearchRequest.isInActiveUsers());
		if (!userSearchRequest.isInActiveUsers()){
			predicates.add(cb.equal(cb.lower(root.get("enabled")),"y"));
		}
	}
	private PageRequest generatePagination(UserSearchRequest userSearchRequest) {
		PageRequest pageable;
		String sortColumn = validateSort(userSearchRequest.getSortColumn());
		logger.info("getAll275SearchResult(): The sort column is " + sortColumn);
		Sort.Order order;
		if (userSearchRequest.getDescending()) {
			/*
			 * pageable = new
			 * PageRequest(userSearchRequest.getPageNumber().intValue() - 1,
			 * userSearchRequest.getPageSize().intValue(), Sort.Direction.DESC,
			 * sortColumn);
			 */
			order = new Sort.Order(Sort.Direction.DESC, sortColumn).ignoreCase();

			pageable = new PageRequest(userSearchRequest.getPageNumber().intValue() - 1,
					userSearchRequest.getPageSize().intValue(), new Sort(order));
		} else {
			/*
			 * pageable = new
			 * PageRequest(userSearchRequest.getPageNumber().intValue() - 1,
			 * userSearchRequest.getPageSize().intValue(), Sort.Direction.ASC,
			 * sortColumn);
			 */
			order = new Sort.Order(Sort.Direction.ASC, sortColumn).ignoreCase();
			pageable = new PageRequest(userSearchRequest.getPageNumber().intValue() - 1,
					userSearchRequest.getPageSize().intValue(), new Sort(order));
		}
		return pageable;
	}

	private String validateSort(String sortColumn) {
		sortColumn.toUpperCase();
		logger.info("The requested Sort Column is " + sortColumn);
		if (sortColumn != null && !(sortColumn.isEmpty())) {
			switch (sortColumn) {
			case "userName":
				return "userName";
			case "domain":
				return "domain";
			case "status":
				return "enabled";
			case "export":
				return "enableExport";
			case "edit":
				return "enableEdit";
			case "admin":
				return "enableAdmin";
			case "firstName":
				return "firstName";
			case "lastName":
				return "lastName";
			case "email":
				return "emailAddress";
			case "phone":
				return "phoneNumber";
			default:
				return "userName";
			}
		} else {
			return "userName";
		}
	}

	private int resetPageNumbersOnCountElements(UserSearchRequest userSearchRequest, Long countElements,
			int pageNumber) {
		if (countElements <= (userSearchRequest.getPageSize() * userSearchRequest.getPageNumber())) {
			Integer resultCountIntVal = countElements.intValue();
			float pageNumberFloat = (float) resultCountIntVal / (float) userSearchRequest.getPageSize();
			pageNumber = (int) Math.ceil(pageNumberFloat);

			logger.debug("getAllUserDetails() : pageNumber is :" + pageNumber + " and pageSize is :"
					+ userSearchRequest.getPageSize());
		}
		return pageNumber;
	}

	private GenericResponse setGenericJsonResponse(UserSearchRequest userSearchRequest, PageRequest pageable,
			Long countElements, int pageNumber) {
		GenericResponse searchResponse;
		if (pageNumber == 0) {
			List<UserDetailsResponse> responseList = querySpecification(userSearchRequest, pageable);
			searchResponse = new GenericResponse(userSearchRequest.getPageNumber(), userSearchRequest.getPageSize(),
					userSearchRequest.getSortColumn(), countElements);
			searchResponse.setResponse(responseList);

		} else {
			List<UserDetailsResponse> responseList = querySpecification(userSearchRequest, pageable);
			searchResponse = new GenericResponse(userSearchRequest.getPageNumber().intValue(),
					userSearchRequest.getPageSize().intValue(), userSearchRequest.getSortColumn(), countElements);
			searchResponse.setResponse(responseList);

		}
		return searchResponse;
	}

	private List<UserDetailsResponse> querySpecification(UserSearchRequest userSearchRequest, PageRequest pageable) {
		Page<ArsUser> cm = (Page<ArsUser>) arsUserRepository.findAll(new Specification<ArsUser>() {

			@Override
			public Predicate toPredicate(Root<ArsUser> root, CriteriaQuery<?> query, CriteriaBuilder cb) {
				List<Predicate> predicates = new ArrayList<>();
				getQueryData(userSearchRequest, root, cb, predicates);
				return cb.and(predicates.toArray(new Predicate[0]));
			}
		}, pageable);
		return assignData(cm);
	}

	private List<UserDetailsResponse> assignData(Page<ArsUser> cm) {
		List<UserDetailsResponse> responseList = new ArrayList<>();
		for (ArsUser cs : cm) {
			UserDetailsResponse response = new UserDetailsResponse();
			response.setUserId(cs.getArsUserId());
			response.setUserName(cs.getUserName());
			// response.setStatus(cs.getEnabled());
			//logger.info("--- " + cs.getEnabled());
			if (cs.getEnabled().equalsIgnoreCase("Y"))
				response.setStatus("Active");
			else
				response.setStatus("Inactive");

			response.setDomain(cs.getDomain());

			if (cs.getEnableExport())
				response.setExport("Yes");
			else
				response.setExport("No");

			if (cs.getEnableEdit())
				response.setEdit("Yes");
			else
				response.setEdit("No");

			if (cs.getEnableAdmin())
				response.setAdmin("Yes");
			else
				response.setAdmin("No");

			response.setFirstName(cs.getFirstName());
			response.setLastName(cs.getLastName());
			response.setEmail(cs.getEmailAddress());
			response.setPhone(cs.getPhoneNumber());
			response.setCreatedBy(cs.getCreatedBy());
			response.setModifiedBy(cs.getModifiedBy());
			responseList.add(response);
		}
		return responseList;
	}

	@Override
	public boolean editUserByUserName(UserSearchRequest userSearchRequest) throws GenericException {

		// search the user by name
		// if exsists
		// alter that row
		// else
		// send not available error
		ArsUser userDetail = arsUserRepository.findByArsUserId(userSearchRequest.getUserId());
		boolean enableExport = false;
		boolean enableEdit = false;
		boolean enableAdmin = false;
		if (userSearchRequest.getExport() != null) {
			if (userSearchRequest.getExport().equalsIgnoreCase("Y")) {
				enableExport = true;
			} else {
				enableExport = false;
			}
		} else
			throw new GenericException("Enable Export Input Unavailable");

		if (userSearchRequest.getEdit() != null && !userSearchRequest.getEdit().isEmpty()) {
			if (userSearchRequest.getEdit().equalsIgnoreCase("Y")) {
				enableEdit = true;
			} else {
				enableEdit = false;
			}
		} else
			throw new GenericException("Enable Edit Input Unavailable");

		if (userSearchRequest.getAdmin() != null) {
			if (userSearchRequest.getAdmin().equalsIgnoreCase("Y") && !userSearchRequest.getAdmin().isEmpty()) {
				enableAdmin = true;
			} else {
				enableAdmin = false;
			}
		} else
			throw new GenericException("Enable Admin Input Unavailable");

		if (userSearchRequest.getModifiedBy() == null || userSearchRequest.getModifiedBy().isEmpty()) {
			throw new GenericException(
					"CreatedBy cannot be a null or empty, Please enter a valid Created by user name");
		}
		if (userSearchRequest.getStatus() == null || userSearchRequest.getStatus().isEmpty()) {
			throw new GenericException("Status cannot be  null or empty, Please enter a valid status");
		}

		if (userDetail != null) {

			ArsUser persisted = new ArsUser(userDetail.getArsUserId(), userSearchRequest.getUserName(),
					userSearchRequest.getStatus(), userSearchRequest.getLastName(), userSearchRequest.getFirstName(),
					userDetail.getMiddleName(), userSearchRequest.getPhone(), userSearchRequest.getEmail(),
					userDetail.getLastLoginDate(), userDetail.getDeactivationReasonCd(),
					userDetail.getDeactivationComments(), userDetail.getCreatedBy(), userDetail.getDateCreated(),
					userSearchRequest.getModifiedBy(),new Date() , enableExport, enableEdit, enableAdmin,
					userSearchRequest.getDomain(), userDetail.getArsUserRoleUsageses());

			arsUserRepository.save(persisted);

			/*******
			 * Another way to update user
			 * arsUserRepository.updateUserByUserName(userSearchRequest.getUserName().toUpperCase(),
			 * userSearchRequest.getDomain(),enableExport ,enableEdit ,
			 * enableAdmin);
			 */
		} else {
			throw new GenericException("User not available to edit, Please edit a valid user");
		}

		/*
		 * Update the Bridging table (ARS_ROLE_USER_USAGES)now accordingly if
		 * Admin checked if not user not mapped to admin role add a row mapping
		 * user id to role id of admin else if there is a row exsist in bridging
		 * table (ARS_ROLE_USER_USAGES) that maps user to admi delete it
		 */

		ArsUserRoleUsages userrolemapping = new ArsUserRoleUsages();
		ArsRole role = arsRole.findByRoleNameIgnoreCase("ADMIN");
		logger.info("Role Id " + role.getArsRoleId() + " Role Name" + role.getRoleName());
		userrolemapping.setId(new ArsUserRoleUsagesId(role.getArsRoleId(), userDetail.getArsUserId()));
		userrolemapping.setArsRole(role);
		userrolemapping.setArsUser(userDetail);
		userrolemapping.setCreatedBy("Admin");
		userrolemapping.setDateCreated(new Date());

		if (enableAdmin) {
			arsUserRoleUsages.save(userrolemapping);
		} else {
			arsUserRoleUsages.delete(userrolemapping);
		}

		return true;
	}

	@Override
	public boolean deleteUserByUserName(Long userId) throws GenericException {
		logger.info("Delete Service layer hit" + userId);
		ArsUser userDetail = arsUserRepository.findByArsUserId(userId);
		if (userDetail != null) {
			logger.info("User found, deleting the user");
			arsUserRepository.delete(userDetail);
			logger.info("User deleted");
		} else {
			logger.info("Exception User not present");
			throw new GenericException("User not available to delete, Please delete a valid user");
		}
		return true;
	}

}
