package gov.va.med.ars.service.impl;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.stream.Stream;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

import org.apache.commons.lang.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import gov.va.med.ars.constants.ErrorMessages;
import gov.va.med.ars.constants.IClaimAttachmentsViewConstants;
import gov.va.med.ars.dao.ars.IClaimAttachmentsViewRepository;
import gov.va.med.ars.exceptions.GenericException;
import gov.va.med.ars.model.request.SearchClaimAttachments275Request;
import gov.va.med.ars.model.response.GenericResponse;
import gov.va.med.ars.model.response.SearchClaimAttachments275Response;
import gov.va.med.ars.service.ISearchClaimAttachments275Service;
import gov.va.med.domain.ars.ClaimattachmentsView;

@Service
public class SearchClaimAttachments275ServiceImpl implements ISearchClaimAttachments275Service {

	private static final Logger logger = LogManager.getLogger(SearchClaimAttachments275ServiceImpl.class);

	DateFormat dateFormat = new SimpleDateFormat("MM/dd/yyyy");
	private static final String CURRENT_DATE_FORMAT = "yyyyMMdd";

	@Autowired
	IClaimAttachmentsViewRepository claimAttachmentsViewRepository;

	@Override
	public GenericResponse getAll275SearchResult(SearchClaimAttachments275Request searchClaimAttachments275Request)
			throws GenericException {

		GenericResponse searchResponse = null;
		PageRequest pageable = generatePagination(searchClaimAttachments275Request);

		try {
			boolean value = checkObjectNullOrEmpty(searchClaimAttachments275Request);

			if (value) {
				Long countElements = claimAttachmentsViewRepository.count(new Specification<ClaimattachmentsView>() {

					@Override
					public Predicate toPredicate(Root<ClaimattachmentsView> root, CriteriaQuery<?> query,
							CriteriaBuilder cb) {
						List<Predicate> predicates = new ArrayList<>();

						getQueryData(searchClaimAttachments275Request, root, cb, predicates);

						return cb.and(predicates.toArray(new Predicate[0]));
					}

				});
				logger.info("The result count is " + countElements);
				int pageNumber = searchClaimAttachments275Request.getPageNumber();
				pageNumber = resetPageNumbersOnCountElements(searchClaimAttachments275Request, countElements,
						pageNumber);
				if (countElements > 0) {
					logger.info("Result Count : " + countElements);
					searchResponse = setGenericJsonResponse(searchClaimAttachments275Request, pageable, countElements,
							pageNumber);
				} else {
					logger.info("No data found");
					GenericResponse emptyResponse = new GenericResponse(
							searchClaimAttachments275Request.getPageNumber(),
							searchClaimAttachments275Request.getPageSize(),
							searchClaimAttachments275Request.getSortColumn(), countElements);
					return emptyResponse;
				}
			} else {
				logger.info("Please enter at least one field to search the 275 attachments");
				throw new GenericException(ErrorMessages.INVALID_REQUEST,
						"At least one field information needs to be added", HttpStatus.NOT_FOUND);
			}

		} catch (Exception e) {
			throw new GenericException(ErrorMessages.DATA_ACCESS_ERROR, e.getMessage(),
					HttpStatus.INTERNAL_SERVER_ERROR);
		}
		logger.info("The search response object is" + searchResponse.toString());
		logger.info("The search results are " + searchResponse.getResponse().toString());
		return searchResponse;
	}

	private PageRequest generatePagination(SearchClaimAttachments275Request searchClaimAttachments275Request) {
		PageRequest pageable;
		String sortColumn = validateSort(searchClaimAttachments275Request.getSortColumn());
		logger.info("getAll275SearchResult(): The sort column is " + sortColumn);
		if (searchClaimAttachments275Request.getDescending()) {
			pageable = new PageRequest(searchClaimAttachments275Request.getPageNumber().intValue() - 1,
					searchClaimAttachments275Request.getPageSize().intValue(), Sort.Direction.DESC, sortColumn);
		} else {
			pageable = new PageRequest(searchClaimAttachments275Request.getPageNumber().intValue() - 1,
					searchClaimAttachments275Request.getPageSize().intValue(), Sort.Direction.ASC, sortColumn);
		}
		return pageable;
	}

	private String validateSort(String sortColumn) {
		logger.info("The requested Sort Column is " + sortColumn);
		if (sortColumn != null && !(sortColumn.isEmpty())) {
			switch (sortColumn) {
			case IClaimAttachmentsViewConstants.ATTACHMENT_ID:
				return "attachmentId";
			case IClaimAttachmentsViewConstants.ATTACHMENT_CONTROL_NUMBER:
				return "attachmentControlNumber";
			case IClaimAttachmentsViewConstants.CLAIM_IDENTIFIER:
				return "externalClaimId";
			case IClaimAttachmentsViewConstants.EXTERNAL_CLAIM_ID:
				return "externalClaimId";
			case IClaimAttachmentsViewConstants.STATUS:
				return "status";
			case IClaimAttachmentsViewConstants.PATIENT_FIRST_NAME:
				return "patientFirstName";
			case IClaimAttachmentsViewConstants.PATIENT_LAST_NAME:
				return "patientLastName";
			case IClaimAttachmentsViewConstants.PATIENT_CONTROL_NUMBER:
				return "patientControlNumber";
			case IClaimAttachmentsViewConstants.PATIENT_IDENTIFIER:
				return "patientIdentifierNumber";
			case IClaimAttachmentsViewConstants.PROVIDER_NAME:
				return "providerName";
			case IClaimAttachmentsViewConstants.PROVIDER_NPI:
				return "providerNpi";
			case IClaimAttachmentsViewConstants.MEDICAL_RECORD_NUMBER:
				return "medicalRecordNumber";
			case IClaimAttachmentsViewConstants.PAYER_CONTROL_NUMBER:
				return "payerClaimControlNumber";
			case IClaimAttachmentsViewConstants.REPORT_CODE:
				return "reportCode";
			case IClaimAttachmentsViewConstants.CLAIM_SERVICE_START_DATE:
				return "claimServiceStartDate";
			case IClaimAttachmentsViewConstants.CLAIM_SERVICE_END_DATE:
				return "claimServiceEndDate";
			case IClaimAttachmentsViewConstants.ATTACHMENT_TYPE:
				return "attachmentType";
			default:
				return "attachmentId";
			}
		} else {
			return "attachmentId";
		}

	}

	private boolean checkObjectNullOrEmpty(SearchClaimAttachments275Request searchClaimAttachments275Request) {
		return Stream.of(searchClaimAttachments275Request.getAttachId(),
				searchClaimAttachments275Request.getAttachCtrNumber(), searchClaimAttachments275Request.getClaimId(),
				searchClaimAttachments275Request.getStatus(), searchClaimAttachments275Request.getPatientLastName(),
				searchClaimAttachments275Request.getPatientFirstName(),
				searchClaimAttachments275Request.getPatientCtrNumber(),
				searchClaimAttachments275Request.getPatientIdentifier(),
				searchClaimAttachments275Request.getProviderName(), searchClaimAttachments275Request.getProviderNpi(),
				searchClaimAttachments275Request.getMedicalRecordNumber(),
				searchClaimAttachments275Request.getAttachType(),
				searchClaimAttachments275Request.getPayerControlNumber(),
				searchClaimAttachments275Request.getReportTypeCode(),
				searchClaimAttachments275Request.getClaimServiceStartDate(),
				searchClaimAttachments275Request.getClaimServiceEndDate()).filter(t -> t != null)
				.filter(t -> !((String) t).isEmpty()).findAny().isPresent();
	}

	private void getQueryData(SearchClaimAttachments275Request search275Request, Root<ClaimattachmentsView> root,
			CriteriaBuilder cb, List<Predicate> predicates) {

		DateFormat formatter = new SimpleDateFormat(CURRENT_DATE_FORMAT);

		if (search275Request.getAttachId() != null && !(search275Request.getAttachId().isEmpty())) {
			predicates.add(cb.equal(root.get("attachmentId"), search275Request.getAttachId()));
		}
		if (search275Request.getAttachCtrNumber() != null && !(search275Request.getAttachCtrNumber().isEmpty())) {
			predicates.add(cb.equal(cb.lower(root.get("attachmentControlNumber")),
					search275Request.getAttachCtrNumber().toLowerCase()));
		}
		if (search275Request.getClaimId() != null && !(search275Request.getClaimId().isEmpty())) {
			predicates
					.add(cb.equal(cb.lower(root.get("externalClaimId")), search275Request.getClaimId().toLowerCase()));
		}
		if (search275Request.getStatus() != null && !(search275Request.getStatus().isEmpty())) {
			predicates.add(cb.equal(cb.lower(root.get("status")), search275Request.getStatus().toLowerCase()));
		}
		if (search275Request.getPatientLastName() != null && !(search275Request.getPatientLastName().isEmpty())) {
			predicates.add(cb.equal(cb.lower(root.get("patientLastName")),
					search275Request.getPatientLastName().toLowerCase()));
		}
		if (search275Request.getPatientFirstName() != null && !(search275Request.getPatientFirstName().isEmpty())) {
			predicates.add(cb.equal(cb.lower(root.get("patientFirstName")),
					search275Request.getPatientFirstName().toLowerCase()));
		}
		if (search275Request.getPatientCtrNumber() != null && !(search275Request.getPatientCtrNumber().isEmpty())) {
			logger.info("getProviderNamecondition  added " + search275Request.getPatientCtrNumber().toLowerCase());
			predicates.add(cb.equal(cb.lower(root.get("patientControlNumber")),
					search275Request.getPatientCtrNumber().toLowerCase()));
		}
		if (search275Request.getPatientIdentifier() != null && !(search275Request.getPatientIdentifier().isEmpty())) {
			predicates.add(cb.equal(cb.lower(root.get("patientIdentifierNumber")),
					search275Request.getPatientIdentifier().toLowerCase()));
		}
		if (search275Request.getProviderName() != null && !(search275Request.getProviderName().isEmpty())) {
			predicates.add(
					cb.equal(cb.lower(root.get("providerName")), search275Request.getProviderName().toLowerCase()));
		}
		if (search275Request.getProviderNpi() != null && !(search275Request.getProviderNpi().isEmpty())) {
			predicates
					.add(cb.equal(cb.lower(root.get("providerNpi")), search275Request.getProviderNpi().toLowerCase()));
		}
		if (search275Request.getMedicalRecordNumber() != null
				&& !(search275Request.getMedicalRecordNumber().isEmpty())) {
			predicates.add(cb.equal(cb.lower(root.get("medicalRecordNumber")),
					search275Request.getMedicalRecordNumber().toLowerCase()));
		}
		if (search275Request.getPayerControlNumber() != null && !(search275Request.getPayerControlNumber().isEmpty())) {
			predicates.add(cb.equal(cb.lower(root.get("payerClaimControlNumber")),
					search275Request.getPayerControlNumber().toLowerCase()));
		}
		if (search275Request.getReportTypeCode() != null && !(search275Request.getReportTypeCode().isEmpty())) {
			logger.info("getReportTypeCode " + search275Request.getReportTypeCode());
		}
		if (search275Request.getReportTypeCode() != null && !(search275Request.getReportTypeCode().isEmpty())
				&& !(search275Request.getReportTypeCode().trim().equalsIgnoreCase("all"))) {

			predicates.add(
					cb.equal(cb.lower(root.get("reportCode")), search275Request.getReportTypeCode().toLowerCase()));
		}
		if (search275Request.getClaimServiceStartDate() != null
				&& !(search275Request.getClaimServiceStartDate().isEmpty())) {

			String beginServiceDateInDateType = null;
			try {
				Date startDate = dateFormat.parse(search275Request.getClaimServiceStartDate());
				beginServiceDateInDateType = formatter.format(startDate);
			} catch (ParseException e) {
				logger.info("getQueryData(): The error is at Start Date Parse , varaible beginServiceDateInDateType");
				e.printStackTrace();
			}
			predicates.add(cb.greaterThanOrEqualTo(root.get("claimServiceStartDate"), beginServiceDateInDateType));
		}
		if (search275Request.getClaimServiceEndDate() != null
				&& !(search275Request.getClaimServiceEndDate().isEmpty())) {

			String endServiceDateInDateType = null;
			try {
				Date endDate = dateFormat.parse(search275Request.getClaimServiceEndDate());
				endServiceDateInDateType = formatter.format(endDate);
			} catch (ParseException e) {
				logger.info("getQueryData(): The error is at End Date Parse , varaible endServiceDateInDateType");
				e.printStackTrace();
			}
			predicates.add(cb.lessThanOrEqualTo(root.get("claimServiceEndDate"), endServiceDateInDateType.toString()));
		}
	}

	private GenericResponse setGenericJsonResponse(SearchClaimAttachments275Request search275Request,
			PageRequest pageable, Long countElements, int pageNumber) {
		GenericResponse searchResponse;
		if (pageNumber == 0) {
			List<SearchClaimAttachments275Response> responseList = querySpecification(search275Request, pageable);
			searchResponse = new GenericResponse(search275Request.getPageNumber(), search275Request.getPageSize(),
					search275Request.getSortColumn(), countElements);
			searchResponse.setResponse(responseList);

		} else {
			List<SearchClaimAttachments275Response> responseList = querySpecification(search275Request, pageable);

			searchResponse = new GenericResponse(search275Request.getPageNumber().intValue(),
					search275Request.getPageSize().intValue(), search275Request.getSortColumn(), countElements);
			searchResponse.setResponse(responseList);

		}
		return searchResponse;
	}

	private List<SearchClaimAttachments275Response> querySpecification(
			SearchClaimAttachments275Request search275Request, PageRequest pageable) {
		Page<ClaimattachmentsView> cm = (Page<ClaimattachmentsView>) claimAttachmentsViewRepository
				.findAll(new Specification<ClaimattachmentsView>() {

					@Override
					public Predicate toPredicate(Root<ClaimattachmentsView> root, CriteriaQuery<?> query,
							CriteriaBuilder cb) {
						List<Predicate> predicates = new ArrayList<>();

						getQueryData(search275Request, root, cb, predicates);

						return cb.and(predicates.toArray(new Predicate[0]));
					}

				}, pageable);
		return assignData(cm);
	}

	private List<String> setPathAddess(String multiplePathAddress) {
		List<String> pathAddress = Arrays.asList(multiplePathAddress.split("<"));
		logger.info("SPLIT PATHS : " + pathAddress.toString());
		return pathAddress;
	}

	private List<SearchClaimAttachments275Response> assignData(Page<ClaimattachmentsView> cm) {
		DateFormat formatter = new SimpleDateFormat(CURRENT_DATE_FORMAT);
		List<SearchClaimAttachments275Response> responseList = new ArrayList<>();
		for (ClaimattachmentsView cs : cm) {
			SearchClaimAttachments275Response response = new SearchClaimAttachments275Response();
			response.setAttachId(cs.getAttachmentId());
			response.setAttachCtrNumber(cs.getAttachmentControlNumber());
			if (StringUtils.isNotBlank(cs.getExternalClaimId())) {
				response.setClaimId(cs.getExternalClaimId().replaceAll("\\s", ""));
				response.setExternalClaimId(cs.getExternalClaimId().replaceAll("\\s", ""));
			} else {
				response.setClaimId("");
				response.setExternalClaimId("");
			}

			if (StringUtils.isNotBlank(cs.getPatientIdentifierNumber())) {
				response.setPatientIdentifier(cs.getPatientIdentifierNumber().replaceAll("\\s", ""));
			} else {
				response.setPatientIdentifier("");
			}

			if (StringUtils.isNotBlank(cs.getProviderNpi())) {
				response.setProviderNpi(cs.getProviderNpi().replaceAll("\\s", ""));
			} else {
				response.setProviderNpi("");
			}
			response.setStatus(cs.getStatus());
			response.setPatientName(cs.getPatientLastName() + ", " + cs.getPatientFirstName());
			response.setPatientCtrNumber(cs.getPatientControlNumber());
			response.setProviderName(cs.getProviderName());
			response.setMedicalRecordNumber(cs.getMedicalRecordNumber());
			response.setPayerControlNumber(cs.getPayerClaimControlNumber());
			response.setReportTypeCode(cs.getReportCode());
			response.setAttachType(cs.getAttachmentType());
			response.setPathAddress(setPathAddess(cs.getAttachmentPath()));
			if (cs.getClaimServiceStartDate() != null) {
				try {
					response.setClaimServiceStartDate(
							dateFormat.format(formatter.parse(cs.getClaimServiceStartDate())));
				} catch (ParseException e) {
					logger.info("assignData(): The error is at Start Date Format");
					e.printStackTrace();
				}
			}
			if (cs.getClaimServiceEndDate() != null) {
				try {
					response.setClaimServiceEndDate(dateFormat.format(formatter.parse(cs.getClaimServiceEndDate())));
				} catch (ParseException e) {
					logger.info(" assignData(): The error is at EndDate Format");
					e.printStackTrace();
				}
			}
			responseList.add(response);

		}

		return responseList;
	}

	private int resetPageNumbersOnCountElements(SearchClaimAttachments275Request search275Request, Long countElements,
			int pageNumber) {
		if (countElements <= (search275Request.getPageSize() * search275Request.getPageNumber())) {
			Integer resultCountIntVal = countElements.intValue();
			float pageNumberFloat = (float) resultCountIntVal / (float) search275Request.getPageSize();
			pageNumber = (int) Math.ceil(pageNumberFloat);

			logger.debug("getAll275SearchResult() : pageNumber is :" + pageNumber + " and pageSize is :"
					+ search275Request.getPageSize());
		}
		return pageNumber;
	}

}
