/**
 * 
 */
package gov.va.med.ars.service.impl;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.stream.Stream;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import gov.va.med.ars.constants.ClaimSubmissionConstants;
import gov.va.med.ars.constants.ErrorMessages;
import gov.va.med.ars.dao.ars.IRfaiLookupRepository;
import gov.va.med.ars.dao.ars.RfaiLookupRepository;
import gov.va.med.ars.exceptions.GenericException;
import gov.va.med.ars.model.request.RfaiLookupRequest;
import gov.va.med.ars.model.response.GenericResponse;
import gov.va.med.ars.model.response.RfaiLookUpResponse;
import gov.va.med.ars.service.IRfaiLookupService;
import gov.va.med.domain.ars.ClaimSubmission;

/**
 * @author PII
 *
 */
@Service
public class RfaiLookupServiceImpl implements IRfaiLookupService {

	private static final Logger logger = LogManager.getLogger(RfaiLookupServiceImpl.class);

	DateFormat dateFormat = new SimpleDateFormat("MM/dd/yyyy");
	private static final String CURRENT_DATE_FORMAT = "MM/dd/yyyy";

	@Autowired
	RfaiLookupRepository rfaiRepository;

	@Autowired
	IRfaiLookupRepository iRfaiRepository;

	@Override
	public GenericResponse getRfaiSimpleSearchResults(RfaiLookupRequest rfaiLookupRequest) throws GenericException {
		GenericResponse searchResponse;
		List<ClaimSubmission> pdiClaim = new ArrayList<>();
		List<ClaimSubmission> submissionId = new ArrayList<>();
		List<ClaimSubmission> patientId;
		List<ClaimSubmission> providerNpi;

		if (rfaiLookupRequest.getNumber().matches("[0-9]+")) {
			pdiClaim = rfaiRepository.findBytrn02fppsClaimId(Long.valueOf(rfaiLookupRequest.getNumber()));
			submissionId = rfaiRepository.findBysubmissionid(Long.valueOf(rfaiLookupRequest.getNumber()));
			patientId = rfaiRepository.findByhl4nm109ptid(rfaiLookupRequest.getNumber());
			providerNpi = rfaiRepository.findByhl3nm109prvnpi(rfaiLookupRequest.getNumber());
		} else {
			patientId = rfaiRepository.findByhl4nm109ptid(rfaiLookupRequest.getNumber());
			providerNpi = rfaiRepository.findByhl3nm109prvnpi(rfaiLookupRequest.getNumber());
		}

		List<ClaimSubmission> cm = new ArrayList<>();
		cm.addAll(pdiClaim);
		cm.addAll(submissionId);
		cm.addAll(patientId);
		cm.addAll(providerNpi);

		Collections.sort(cm, new Comparator<ClaimSubmission>() {

			@Override
			public int compare(ClaimSubmission o1, ClaimSubmission o2) {
				try {
					switch (rfaiLookupRequest.getSortColumn()) {
					case ClaimSubmissionConstants.PROVIDER_NAME:
						return compareByAscDesc(rfaiLookupRequest.getDescending(), o1.getHl3nm103svcbillprvnm(),
								o2.getHl3nm103svcbillprvnm(), ClaimSubmission.class
										.getDeclaredField(ClaimSubmissionConstants.PROVIDER_NAME_DB_COLUMN).getType());
					case ClaimSubmissionConstants.PROVIDER_NPI:
						return compareByAscDesc(rfaiLookupRequest.getDescending(), o1.getHl3nm109prvnpi(),
								o2.getHl3nm109prvnpi(), ClaimSubmission.class
										.getDeclaredField(ClaimSubmissionConstants.PROVIDER_NPI_DB_COLUMN).getType());
					case ClaimSubmissionConstants.PROVIDER_TIN:
						return compareByAscDesc(rfaiLookupRequest.getDescending(), o1.getHl2nm109svcprvtin(),
								o2.getHl2nm109svcprvtin(), ClaimSubmission.class
										.getDeclaredField(ClaimSubmissionConstants.PROVIDER_TIN_DB_COLUMN).getType());
					case ClaimSubmissionConstants.PATIENT_LAST_NAME:
						return compareByAscDesc(rfaiLookupRequest.getDescending(), o1.getHl4nm103ptlstnm(),
								o2.getHl4nm103ptlstnm(),
								ClaimSubmission.class
										.getDeclaredField(ClaimSubmissionConstants.PATIENT_LAST_NAME_DB_COLUMN)
										.getType());
					case ClaimSubmissionConstants.PATIENT_FIRST_NAME:
						return compareByAscDesc(rfaiLookupRequest.getDescending(), o1.getHl4nm104ptftnm(),
								o2.getHl4nm104ptftnm(),
								ClaimSubmission.class
										.getDeclaredField(ClaimSubmissionConstants.PATIENT_FIRST_NAME_DB_COLUMN)
										.getType());
					case ClaimSubmissionConstants.PATIENT_ID:
						return compareByAscDesc(rfaiLookupRequest.getDescending(), o1.getHl4nm109ptid(),
								o2.getHl4nm109ptid(), ClaimSubmission.class
										.getDeclaredField(ClaimSubmissionConstants.PATIENT_ID_DB_COLUMN).getType());
					case ClaimSubmissionConstants.PATIENT_CONTROL_NUMBER:
						return compareByAscDesc(rfaiLookupRequest.getDescending(), o1.getRef102ptctrlnmbr(),
								o2.getRef102ptctrlnmbr(),
								ClaimSubmission.class
										.getDeclaredField(ClaimSubmissionConstants.PATIENT_CONTROL_NUMBER_DB_COLUMN)
										.getType());
					case ClaimSubmissionConstants.LINE_ITEM_CONTROL_NUMBER:
						return compareByAscDesc(rfaiLookupRequest.getDescending(), Long.valueOf(o1.getSubmissionid()),
								Long.valueOf(o2.getSubmissionid()), ClaimSubmission.class
										.getDeclaredField(ClaimSubmissionConstants.SUBMISSION_ID_DB_COLUMN).getType());
					case ClaimSubmissionConstants.MEDICAL_RECORD_NUMBER:
						return compareByAscDesc(rfaiLookupRequest.getDescending(), o1.getRef302medrecnmbr(),
								o2.getRef302medrecnmbr(),
								ClaimSubmission.class
										.getDeclaredField(ClaimSubmissionConstants.MEDICAL_RECORD_NUMBER_DB_COLUMN)
										.getType());
					case ClaimSubmissionConstants.DATE_SUBMITTED:
						return compareByAscDesc(rfaiLookupRequest.getDescending(), o1.getDatesubmitted(),
								o2.getDatesubmitted(), ClaimSubmission.class
										.getDeclaredField(ClaimSubmissionConstants.DATE_SUBMITTED_DB_COLUMN).getType());
					case ClaimSubmissionConstants.REQUIRED_RESPONSE_DATE:
						return compareByAscDesc(rfaiLookupRequest.getDescending(), o1.getDtp203responseduedate(),
								o2.getDtp203responseduedate(),
								ClaimSubmission.class
										.getDeclaredField(ClaimSubmissionConstants.RESPONSE_DUE_DATE_DB_COLUMN)
										.getType());
					case ClaimSubmissionConstants.PATIENT_NAME:
						return compareByAscDesc(rfaiLookupRequest.getDescending(), o1.getHl4nm103ptlstnm(),
								o2.getHl4nm103ptlstnm(),
								ClaimSubmission.class
										.getDeclaredField(ClaimSubmissionConstants.PATIENT_LAST_NAME_DB_COLUMN)
										.getType());
					case ClaimSubmissionConstants.PDI_CLAIM_ID:
						return compareByAscDesc(rfaiLookupRequest.getDescending(),
								Long.valueOf(o1.getTrn02fppsClaimId()), Long.valueOf(o2.getTrn02fppsClaimId()),
								ClaimSubmission.class
										.getDeclaredField(ClaimSubmissionConstants.Claim_PDI_DB_COLUMN_NAME).getType());
					case ClaimSubmissionConstants.SUBMISSION_ID:
						return compareByAscDesc(rfaiLookupRequest.getDescending(), Long.valueOf(o1.getSubmissionid()),
								Long.valueOf(o2.getSubmissionid()), ClaimSubmission.class
										.getDeclaredField(ClaimSubmissionConstants.SUBMISSION_ID_DB_COLUMN).getType());
					case ClaimSubmissionConstants.STATUS:
						return compareByAscDesc(rfaiLookupRequest.getDescending(), o1.getSubmissionStatus(),
								o2.getSubmissionStatus(),
								ClaimSubmission.class
										.getDeclaredField(ClaimSubmissionConstants.SUBMISSION_STATUS_DB_COLUMN)
										.getType());
					default:
						return compareByAscDesc(rfaiLookupRequest.getDescending(), Long.valueOf(o1.getSubmissionid()),
								Long.valueOf(o2.getSubmissionid()), ClaimSubmission.class
										.getDeclaredField(ClaimSubmissionConstants.SUBMISSION_ID_DB_COLUMN).getType());
					}
				} catch (NoSuchFieldException | SecurityException e) {
					logger.info("Exception occured in try/catch statements : " + e);
					return 0;
				}
			}

			int compareByAscDesc(boolean desc, Object o1, Object o2, Class<?> classObject) {
				Object pre;
				Object post;
				if (!desc) {
					post = o2;
					pre = o1;
				} else {
					post = o1;
					pre = o2;
				}
				if (pre == null) {
					return (post == null) ? 0 : -1;
				}
				if (post == null) {
					return 1;
				}
				switch (classObject.getSimpleName()) {
				case "Long":
				case "long":
					return ((Long) pre).compareTo((Long) post);
				case "Date":
					return ((Date) pre).compareTo((Date) post);
				default:
					return ((String) pre).compareTo((String) post);
				}
			}
		});

		List<RfaiLookUpResponse> responseList = assignClaimSubmissionListData(cm);

		searchResponse = new GenericResponse(rfaiLookupRequest.getPageNumber().intValue(),
				rfaiLookupRequest.getPageSize().intValue(), rfaiLookupRequest.getSortColumn(),
				new Long(responseList.size()));
		List<RfaiLookUpResponse> response = new ArrayList<>();
		int upperLimit = rfaiLookupRequest.getPageNumber() * rfaiLookupRequest.getPageSize();
		int lowerLimit = upperLimit - rfaiLookupRequest.getPageSize();
		for (int i = lowerLimit; i < upperLimit && i < responseList.size(); i++) {
			if (responseList.size() > i) {
				response.add(responseList.get(i));
			}
		}
		if (!response.isEmpty()) {
			searchResponse.setResponse(response);
		} else {
			throw new GenericException(ErrorMessages.INVALID_REQUEST, "Invalid request", HttpStatus.NOT_FOUND);
		}

		return searchResponse;
	}

	private List<RfaiLookUpResponse> assignClaimSubmissionListData(List<ClaimSubmission> cm) {
		List<RfaiLookUpResponse> responseList = new ArrayList<>();

		for (ClaimSubmission cs : cm) {
			RfaiLookUpResponse response = new RfaiLookUpResponse();
			response.setSubmissionId(cs.getSubmissionid());
			response.setPdiClaimId(cs.getTrn02fppsClaimId());
			response.setPatientName(cs.getHl4nm103ptlstnm() + ", " + cs.getHl4nm104ptftnm());
			response.setStatus(cs.getSubmissionStatus());
			response.setProviderName(cs.getHl3nm103svcbillprvnm());
			response.setProviderNpi(cs.getHl3nm109prvnpi());
			response.setProviderTin(cs.getHl2nm109svcprvtin());
			response.setPatientId(cs.getHl4nm109ptid());
			response.setPatientControlNumber(cs.getRef102ptctrlnmbr());
			response.setMedicalRecordNumber(cs.getRef302medrecnmbr());
			if (cs.getDatesubmitted() != null) {
				response.setDatesubmitted(dateFormat.format(cs.getDatesubmitted()));
			}
			if (cs.getDtp203responseduedate() != null) {
				response.setRequiredResponseDate(dateFormat.format(cs.getDtp203responseduedate()));
			}

			responseList.add(response);
		}
		return responseList;
	}

	private PageRequest generatePagination(RfaiLookupRequest rfaiLookupRequest) {
		PageRequest pageable;
		String sortColumn = validateSort(rfaiLookupRequest.getSortColumn());
		if (rfaiLookupRequest.getDescending()) {
			pageable = new PageRequest(rfaiLookupRequest.getPageNumber().intValue() - 1,
					rfaiLookupRequest.getPageSize().intValue(), Sort.Direction.DESC, sortColumn);
		} else {
			pageable = new PageRequest(rfaiLookupRequest.getPageNumber().intValue() - 1,
					rfaiLookupRequest.getPageSize().intValue(), Sort.Direction.ASC, sortColumn);
		}
		return pageable;
	}

	private String validateSort(String sortColumn) {
		if (sortColumn != null && !(sortColumn.isEmpty())) {
			switch (sortColumn) {
			case ClaimSubmissionConstants.PROVIDER_NAME:
				return "hl3nm103svcbillprvnm";
			case ClaimSubmissionConstants.PROVIDER_NPI:
				return "hl3nm109prvnpi";
			case ClaimSubmissionConstants.PROVIDER_TIN:
				return "hl2nm109svcprvtin";
			case ClaimSubmissionConstants.PATIENT_LAST_NAME:
				return "hl4nm103ptlstnm";
			case ClaimSubmissionConstants.PATIENT_FIRST_NAME:
				return "hl4nm104ptftnm";
			case ClaimSubmissionConstants.PATIENT_ID:
				return "hl4nm109ptid";
			case ClaimSubmissionConstants.PATIENT_CONTROL_NUMBER:
				return "ref102ptctrlnmbr";
			case ClaimSubmissionConstants.LINE_ITEM_CONTROL_NUMBER:
				return "submissionid";
			case ClaimSubmissionConstants.MEDICAL_RECORD_NUMBER:
				return "ref302medrecnmbr";
			case ClaimSubmissionConstants.DATE_SUBMITTED:
				return "datesubmitted";
			case ClaimSubmissionConstants.REQUIRED_RESPONSE_DATE:
				return "dtp203responseduedate";
			case ClaimSubmissionConstants.PATIENT_NAME:
				return "hl4nm103ptlstnm";
			case ClaimSubmissionConstants.PDI_CLAIM_ID:
				return "trn02fppsClaimId";
			case ClaimSubmissionConstants.SUBMISSION_ID:
				return ClaimSubmissionConstants.SUBMISSION_ID_DB_COLUMN;
			case ClaimSubmissionConstants.STATUS:
				return "submissionStatus";
			default:
				return ClaimSubmissionConstants.SUBMISSION_ID_DB_COLUMN;
			}
		} else {
			return "submissionid";
		}

	}

	@Override
	public GenericResponse getRfaiAdvResults(RfaiLookupRequest rfaiLookupRequest) throws GenericException {
		GenericResponse searchResponse = null;
		PageRequest pageable = generatePagination(rfaiLookupRequest);
		try {
			boolean value = checkObjectNullOrEmpty(rfaiLookupRequest);
			if (value) {
				Long countElements = iRfaiRepository.count(new Specification<ClaimSubmission>() {

					@Override
					public Predicate toPredicate(Root<ClaimSubmission> root, CriteriaQuery<?> query,
							CriteriaBuilder cb) {
						List<Predicate> predicates = new ArrayList<>();

						getQueryData(rfaiLookupRequest, root, cb, predicates);

						return cb.and(predicates.toArray(new Predicate[0]));
					}

				});
				int pageNumber = rfaiLookupRequest.getPageNumber();
				pageNumber = resetPageNumbersOnCountElements(rfaiLookupRequest, countElements, pageNumber);
				if (countElements > 0) {
					searchResponse = setGenericJsonResponse(rfaiLookupRequest, pageable, countElements, pageNumber);
				} else {
					// throw new GenericException(ErrorMessages.INVALID_REQUEST, "Invalid request",
					// HttpStatus.NOT_FOUND);
					GenericResponse emptyResponse = new GenericResponse(rfaiLookupRequest.getPageNumber(),
							rfaiLookupRequest.getPageSize(), rfaiLookupRequest.getSortColumn(), countElements);
					return emptyResponse;
				}
			} else {
				throw new GenericException(ErrorMessages.INVALID_REQUEST,
						"At least one field information needs to be added", HttpStatus.NOT_FOUND);
			}

		} catch (Exception e) {
			throw new GenericException(ErrorMessages.DATA_ACCESS_ERROR, e.getMessage(),
					HttpStatus.INTERNAL_SERVER_ERROR);
		}
		return searchResponse;
	}

	private GenericResponse setGenericJsonResponse(RfaiLookupRequest rfaiLookupRequest, PageRequest pageable,
			Long countElements, int pageNumber) {
		GenericResponse searchResponse;
		if (pageNumber == 0) {
			List<RfaiLookUpResponse> responseList = querySpecification(rfaiLookupRequest, pageable);
			searchResponse = new GenericResponse(rfaiLookupRequest.getPageNumber(), rfaiLookupRequest.getPageSize(),
					rfaiLookupRequest.getSortColumn(), countElements);
			searchResponse.setResponse(responseList);

		} else {
			List<RfaiLookUpResponse> responseList = querySpecification(rfaiLookupRequest, pageable);

			searchResponse = new GenericResponse(rfaiLookupRequest.getPageNumber().intValue(),
					rfaiLookupRequest.getPageSize().intValue(), rfaiLookupRequest.getSortColumn(), countElements);
			searchResponse.setResponse(responseList);
		}
		return searchResponse;
	}

	private int resetPageNumbersOnCountElements(RfaiLookupRequest rfaiLookupRequest, Long countElements,
			int pageNumber) {
		if (countElements <= (rfaiLookupRequest.getPageSize() * rfaiLookupRequest.getPageNumber())) {
			Integer resultCountIntVal = countElements.intValue();
			float pageNumberFloat = (float) resultCountIntVal / (float) rfaiLookupRequest.getPageSize();
			pageNumber = (int) Math.ceil(pageNumberFloat);
			/*
			 * logger.debug( "getClaimSearchResults() : pageNumber is :" + pageNumber +
			 * " and pageSize is :" + rfaiLookupRequest.getPageSize());
			 */
		}
		return pageNumber;
	}

	private boolean checkObjectNullOrEmpty(RfaiLookupRequest rfaiLookupRequest) {
		return Stream
				.of(rfaiLookupRequest.getNumber(), rfaiLookupRequest.getMedicalRecordNumber(),
						rfaiLookupRequest.getPatientControlNumber(), rfaiLookupRequest.getPatientfirstName(),
						rfaiLookupRequest.getPatientLastName(), rfaiLookupRequest.getProviderName(),
						rfaiLookupRequest.getProviderNpi(), rfaiLookupRequest.getProviderTin(),
						rfaiLookupRequest.getServiceEndDate(), rfaiLookupRequest.getServiceStartDate(),
						rfaiLookupRequest.getPatientIdentifier())
				.filter(t -> t != null).filter(t -> !((String) t).isEmpty()).findAny().isPresent();
	}

	private List<RfaiLookUpResponse> querySpecification(RfaiLookupRequest rfaiLookupRequest, PageRequest pageable) {
		Page<ClaimSubmission> cm = (Page<ClaimSubmission>) iRfaiRepository
				.findAll(new Specification<ClaimSubmission>() {

					@Override
					public Predicate toPredicate(Root<ClaimSubmission> root, CriteriaQuery<?> query,
							CriteriaBuilder cb) {
						List<Predicate> predicates = new ArrayList<>();

						getQueryData(rfaiLookupRequest, root, cb, predicates);

						return cb.and(predicates.toArray(new Predicate[0]));
					}

				}, pageable);

		return assignData(cm);
	}

	private List<RfaiLookUpResponse> assignData(Page<ClaimSubmission> cm) {
		List<RfaiLookUpResponse> responseList = new ArrayList<>();

		for (ClaimSubmission cs : cm) {
			RfaiLookUpResponse response = new RfaiLookUpResponse();
			response.setSubmissionId(cs.getSubmissionid());
			response.setPdiClaimId(cs.getTrn02fppsClaimId());
			response.setPatientName(cs.getHl4nm103ptlstnm() + ", " + cs.getHl4nm104ptftnm());
			response.setStatus(cs.getSubmissionStatus());
			response.setProviderName(cs.getHl3nm103svcbillprvnm());
			response.setProviderNpi(cs.getHl3nm109prvnpi());
			response.setProviderTin(cs.getHl2nm109svcprvtin());
			response.setPatientId(cs.getHl4nm109ptid());
			response.setPatientControlNumber(cs.getRef102ptctrlnmbr());
			response.setMedicalRecordNumber(cs.getRef302medrecnmbr());
			if (cs.getDatesubmitted() != null) {
				response.setDatesubmitted(dateFormat.format(cs.getDatesubmitted()));
			}
			if (cs.getDtp203responseduedate() != null) {
				response.setRequiredResponseDate(dateFormat.format(cs.getDtp203responseduedate()));
			}

			responseList.add(response);
		}
		return responseList;
	}

	private void getQueryData(RfaiLookupRequest rfaiLookupRequest, Root<ClaimSubmission> root, CriteriaBuilder cb,
			List<Predicate> predicates) {

		DateFormat formatter = new SimpleDateFormat(CURRENT_DATE_FORMAT);

		if (rfaiLookupRequest.getNumber() != null && !(rfaiLookupRequest.getNumber().isEmpty())) {
			predicates.add(cb.equal(root.get("trn02fppsClaimId"), rfaiLookupRequest.getNumber()));
		}

		if (rfaiLookupRequest.getProviderName() != null && !(rfaiLookupRequest.getProviderName().isEmpty())) {
			predicates.add(cb.equal(cb.lower(root.get("hl3nm103svcbillprvnm")),
					rfaiLookupRequest.getProviderName().toLowerCase()));
		}

		if (rfaiLookupRequest.getProviderNpi() != null && !(rfaiLookupRequest.getProviderNpi().isEmpty())) {
			predicates.add(
					cb.equal(cb.lower(root.get("hl3nm109prvnpi")), rfaiLookupRequest.getProviderNpi().toLowerCase()));
		}

		if (rfaiLookupRequest.getProviderTin() != null && !(rfaiLookupRequest.getProviderTin().isEmpty())) {
			predicates.add(cb.equal(cb.lower(root.get("hl2nm109svcprvtin")),
					rfaiLookupRequest.getProviderTin().toLowerCase()));
		}

		if (rfaiLookupRequest.getPatientLastName() != null && !(rfaiLookupRequest.getPatientLastName().isEmpty())) {
			predicates.add(cb.equal(cb.lower(root.get("hl4nm103ptlstnm")),
					rfaiLookupRequest.getPatientLastName().toLowerCase()));
		}

		if (rfaiLookupRequest.getPatientfirstName() != null && !(rfaiLookupRequest.getPatientfirstName().isEmpty())) {
			predicates.add(cb.equal(cb.lower(root.get("hl4nm104ptftnm")),
					rfaiLookupRequest.getPatientfirstName().toLowerCase()));
		}

		if (rfaiLookupRequest.getPatientIdentifier() != null && !(rfaiLookupRequest.getPatientIdentifier().isEmpty())) {
			predicates.add(cb.equal(cb.lower(root.get("hl4nm109ptid")),
					rfaiLookupRequest.getPatientIdentifier().toLowerCase()));
		}

		if (rfaiLookupRequest.getPatientControlNumber() != null
				&& !(rfaiLookupRequest.getPatientControlNumber().isEmpty())) {
			predicates.add(cb.equal(cb.lower(root.get("ref102ptctrlnmbr")),
					rfaiLookupRequest.getPatientControlNumber().toLowerCase()));
		}

		if (rfaiLookupRequest.getMedicalRecordNumber() != null
				&& !(rfaiLookupRequest.getMedicalRecordNumber().isEmpty())) {
			predicates.add(cb.equal(cb.lower(root.get("ref302medrecnmbr")),
					rfaiLookupRequest.getMedicalRecordNumber().toLowerCase()));
		}

		if (rfaiLookupRequest.getServiceStartDate() != null && !(rfaiLookupRequest.getServiceStartDate().isEmpty())) {

			Date beginServiceDateInDateType;
			try {
				beginServiceDateInDateType = formatter.parse(rfaiLookupRequest.getServiceStartDate());
				predicates.add(cb.greaterThanOrEqualTo(root.get("dtp103svcfrmdt"), beginServiceDateInDateType));
			} catch (ParseException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}

		if (rfaiLookupRequest.getServiceEndDate() != null && !(rfaiLookupRequest.getServiceEndDate().isEmpty())) {

			Date endServiceDateInDateType;
			try {
				endServiceDateInDateType = formatter.parse(rfaiLookupRequest.getServiceEndDate());
				Calendar cal = Calendar.getInstance();
				cal.setTime(endServiceDateInDateType);
				cal.add(Calendar.HOUR, 24);
				predicates.add(cb.lessThanOrEqualTo(root.get("dtp103svctodt"), cal.getTime()));
			} catch (ParseException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
	}
}