package gov.va.med.ars.service.impl;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import gov.va.med.ars.dao.ars.ClaimContactRepository;
import gov.va.med.ars.dao.ars.ClaimSubmissionRepository;
import gov.va.med.ars.dao.ars.ClaimSvcLinesRepository;
import gov.va.med.ars.dao.ars.HccsCdRepository;
import gov.va.med.ars.dao.ars.LoincCdRepository;
import gov.va.med.ars.dao.ars.LoincModCdRepository;
import gov.va.med.ars.dao.ars.RfaiLookupRepository;
import gov.va.med.ars.exceptions.GenericException;
import gov.va.med.ars.model.response.Rfai277SubmissionResponse;
import gov.va.med.ars.model.response.RfaiAttachmentResponse;
import gov.va.med.ars.model.response.RfaiLineItemResponse;
import gov.va.med.ars.model.response.RfaiRelatedItemResponse;
import gov.va.med.ars.service.IRfai277SubmissionService;
import gov.va.med.domain.ars.ClaimContact;
import gov.va.med.domain.ars.ClaimSubmission;
import gov.va.med.domain.ars.ClaimSvcLines;

@Service
public class Rfai277SubmissionServiceImpl implements IRfai277SubmissionService {
	
	@Autowired
	ClaimSubmissionRepository claimSubRepo;
	
	@Autowired
	HccsCdRepository hccsRepo;
	
	@Autowired
	LoincCdRepository loincRepo;
	
	@Autowired
	LoincModCdRepository loincModRepo;
	
	@Autowired
	RfaiLookupRepository rfaiRepository;
	
	@Autowired
	ClaimSvcLinesRepository claimSvcLinesRepository;
	
	@Autowired
	ClaimContactRepository claimContactRepository;
	
	@Override
	public Rfai277SubmissionResponse getRfaiSubmissionResults(Long submissionId) throws GenericException {
		
		final Logger logger = LogManager.getLogger(Rfai277SubmissionServiceImpl.class);
		
		Rfai277SubmissionResponse rfaiResponse = new Rfai277SubmissionResponse();
		ClaimSubmission claimSubmission = new ClaimSubmission();
		
		//get claim information by submission id and related submissions for claim id
		try {
			claimSubmission = rfaiRepository.getClaimSubInfo(submissionId);
			
			Long submissionLong = claimSubmission.getSubmissionid();
			//if submission id then populate response object and return data
			if(submissionLong != null) {

				//get the Service Line Items
				//added a new call since ServiceLines are not a mandatory fields to be filled in the db
				List<ClaimSvcLines> claimSvcLines = claimSvcLinesRepository.findAllByclaimSubmission(claimSubmission);
				
				//get related Contact Info
				//added a new call since contact details are not a mandatory fields to be filled in the db
				//List<ClaimContact> claimContact = claimContactRepository.findAllByclaimSubmissions(claimSubmission);
				
				//get related submissions for claim id
				List<RfaiRelatedItemResponse> rfaiRelatedItemResponses = getRfaiRelatedItemResponses(claimSubmission);
				
				//get attachment details for claim id 
				List<RfaiAttachmentResponse> rfaiAttachmentResponses = getRfaiAttachmentResponses(claimSubmission);
				
				//get service line items
				List<RfaiLineItemResponse> rfaiLineItemResponse = getRfaiServiceLineItemResponse(claimSvcLines);
				//populate rfaiResponse object
				rfaiResponse.setPayerName(claimSubmission.getHl1nm103pyrNm());
				rfaiResponse.setPayerId(claimSubmission.getHl1nm109pyrId());
				rfaiResponse.setInformationReceiver(claimSubmission.getHl2nm103infRcvr());
				if(claimSubmission.getDtp203responseduedate() != null) {
					rfaiResponse.setResponseDate(formatDate(claimSubmission.getDtp203responseduedate()));
				}
				rfaiResponse.setBillingProvider(claimSubmission.getHl3nm103svcbillprvnm());
				rfaiResponse.setProviderNpi(claimSubmission.getHl3nm109prvnpi());
				rfaiResponse.setProviderTin(claimSubmission.getHl2nm109svcprvtin());
				rfaiResponse.setPatientLastname(claimSubmission.getHl4nm103ptlstnm());
				rfaiResponse.setPatientFirstName(claimSubmission.getHl4nm104ptftnm());
				rfaiResponse.setPatientIdentifier(claimSubmission.getHl4nm109ptid());
				rfaiResponse.setPatientcontrolNumber(claimSubmission.getRef102ptctrlnmbr());
				rfaiResponse.setPayerClaimControlNumber(claimSubmission.getTrn02pyrclmctrlnmbr());	
				rfaiResponse.setBillType(claimSubmission.getRef202billtype());
				rfaiResponse.setClaimId(claimSubmission.getTrn02fppsClaimId());
				rfaiResponse.setContactInfo(claimSubmission.getClaimContact().getContactemail());
				rfaiResponse.setClearingHouseId(claimSubmission.getRef402clrnghsid());
				rfaiResponse.setMedicalRecordNumber(claimSubmission.getRef302medrecnmbr());
				rfaiResponse.setRfaiRelatedItemResponse(rfaiRelatedItemResponses);
				rfaiResponse.setRfaiAttachmentResponse(rfaiAttachmentResponses);
				if(claimSubmission.getDatesubmitted() !=  null) {
					rfaiResponse.setRfaiSubmissionDate(formatDate(claimSubmission.getDatesubmitted()));
				}
				rfaiResponse.setRfaiStatus(claimSubmission.getSubmissionStatus());
				rfaiResponse.setRfaiLineItemResponse(rfaiLineItemResponse);			
			}
		} catch(Exception e) {
			logger.error(
					"Rfai277SubmissionServiceImpl:getRfai277SubmissionResults() exception occured " + e.getMessage());
			throw e;
		}
		return rfaiResponse;
	}
		
	private List<RfaiLineItemResponse> getRfaiServiceLineItemResponse(List<ClaimSvcLines> claimSvcLines) {	
		List<RfaiLineItemResponse> rfaiSvcLines = new ArrayList<RfaiLineItemResponse>();
		//if (claimSvcLines.getClaimSvcLineses() != null && claimSvcLines.getClaimSvcLineses().size() > 0) {
		if (claimSvcLines != null && claimSvcLines.size() > 0) {
			for(ClaimSvcLines claimSvcLine : claimSvcLines) {
				RfaiLineItemResponse lineResponse = new RfaiLineItemResponse();
				lineResponse.setServiceLineId(Long.valueOf(claimSvcLine.getRef02svclnnmbr()));
				if(claimSvcLine.getDtp03svcfrmdt() != null) {
					lineResponse.setServiceFrom(formatDate(claimSvcLine.getDtp03svcfrmdt()));
				}
				if(claimSvcLine.getDtp03svctodt() != null) {
					lineResponse.setServiceTo(formatDate(claimSvcLine.getDtp03svctodt()));
				}
				lineResponse.setProcedureCode(claimSvcLine.getSvc01proccd());
				lineResponse.setRevenueCode(claimSvcLine.getSvc01revcd());
				lineResponse.setModifiers(claimSvcLine.getSvc01mod());
				lineResponse.setChargeAmount(claimSvcLine.getSvc02chrgamt());
				rfaiSvcLines.add(lineResponse);
			}		
		}
		return rfaiSvcLines;
	}

	private List<RfaiRelatedItemResponse> getRfaiRelatedItemResponses(ClaimSubmission claimSubmission) throws GenericException {
		
		final Logger logger = LogManager.getLogger(Rfai277SubmissionServiceImpl.class);
		
		List<RfaiRelatedItemResponse> rfaiRelatedItemResponses = new ArrayList<RfaiRelatedItemResponse>();
		List<ClaimSubmission> relatedSubmissions = new ArrayList<ClaimSubmission>();
		
		try {
			//get all related claims by claim id from submission
			relatedSubmissions = rfaiRepository.findBytrn02fppsClaimId(claimSubmission.getTrn02fppsClaimId());
			
			//loop through related submissions found to populate related item response object
			if(relatedSubmissions.size() > 0) {
				for(ClaimSubmission cs : relatedSubmissions) {
					if(cs.getSubmissionid() != claimSubmission.getSubmissionid()) {
						RfaiRelatedItemResponse rfaiRelatedItemResponse = new RfaiRelatedItemResponse();
						rfaiRelatedItemResponse.setClaimId(cs.getTrn02fppsClaimId());
						if(cs.getDtp103svctodt() != null) {
							rfaiRelatedItemResponse.setEndDate(formatDate(cs.getDtp103svctodt()));
						}
						rfaiRelatedItemResponse.setMedicalRecordNumber(cs.getRef302medrecnmbr());
						rfaiRelatedItemResponse.setPatientControlNumber(cs.getRef102ptctrlnmbr());
						rfaiRelatedItemResponse.setPatientId(cs.getHl4nm109ptid());
						rfaiRelatedItemResponse.setPatientName(cs.getHl4nm103ptlstnm() + "," + cs.getHl4nm104ptftnm());
						rfaiRelatedItemResponse.setProviderNPI(cs.getHl3nm109prvnpi());
						rfaiRelatedItemResponse.setProviderTin(cs.getHl2nm109svcprvtin());
						if(cs.getDtp103svcfrmdt() != null) {
							rfaiRelatedItemResponse.setStartDate(formatDate(cs.getDtp103svcfrmdt()));
						}
						rfaiRelatedItemResponse.setSubmissionId(cs.getSubmissionid());
						rfaiRelatedItemResponse.setProviderName(cs.getHl3nm103svcbillprvnm());
						rfaiRelatedItemResponses.add(rfaiRelatedItemResponse);
					}
				}
			}
		} catch(Exception e) {
			logger.error(
					"Rfai277SubmissionServiceImpl:getRfaiRelatedItemResponses() exception occured " + e.getMessage());
			throw e;
		}
		return rfaiRelatedItemResponses;
	}
	
	private List<RfaiAttachmentResponse> getRfaiAttachmentResponses(ClaimSubmission claimSubmission) throws GenericException {
		
		List<RfaiAttachmentResponse> rfaiAttachmentResponses = new ArrayList<RfaiAttachmentResponse>();	
		//Create the attachment response object
		RfaiAttachmentResponse rfaiAttachmentResponse = new RfaiAttachmentResponse();		
		rfaiAttachmentResponse.setLoincCodes(claimSubmission.getLoincCd().getStcloincCd());
		rfaiAttachmentResponse.setLoincModCodes(claimSubmission.getLoincModCd().getStcloincCdMod());
		rfaiAttachmentResponse.setStatusCode(claimSubmission.getHccsCd().getStchccstatuscatCd());
		rfaiAttachmentResponse.setClaimLevel("");
		rfaiAttachmentResponse.setAttachments("");
		rfaiAttachmentResponses.add(rfaiAttachmentResponse);
		return rfaiAttachmentResponses;
	}
	
	private String formatDate(Date date) {
		SimpleDateFormat dateFormat = new SimpleDateFormat("MM/dd/yyyy");
		return dateFormat.format(date);
	}
}