package gov.va.med.ars.service.impl;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.stream.Stream;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

import org.apache.commons.lang.StringUtils;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import gov.va.med.ars.constants.ErrorMessages;
import gov.va.med.ars.constants.ReportsViewConstants;
import gov.va.med.ars.dao.ars.IReportsViewRepository;
import gov.va.med.ars.exceptions.GenericException;
import gov.va.med.ars.model.request.SearchReportRequest;
import gov.va.med.ars.model.response.GenericResponse;
import gov.va.med.ars.model.response.ReportResponse;
import gov.va.med.ars.service.IReportService;
import gov.va.med.domain.ars.ReportsView;

@Service
public class ReportServiceImpl implements IReportService {

	private static final Logger logger = LogManager.getLogger(ReportServiceImpl.class);

	DateFormat formatDateOfRequest = new SimpleDateFormat("MM/dd/yyyy");
	DateFormat formatDateOfDB = new SimpleDateFormat("yyyyMMdd");

	@Autowired
	IReportsViewRepository reportsViewRepository;

	@Override
	public GenericResponse getReports(SearchReportRequest searchRequest, boolean getMatchedReport)
			throws GenericException {

		GenericResponse reportSearchResponse = null;
		PageRequest pageReq = generatePaginationWithSorting(searchRequest);
		logger.info("Inside ReportServiceImpl.getReports()");
		try {
			boolean requestIsNotEmpty = isRequestNotEmpty(searchRequest);
			if (requestIsNotEmpty) {
				logger.info("ReportServiceImpl.getReports(): calling repository for count...");
				Long countOfReports = reportsViewRepository.count(new Specification<ReportsView>() {
					@Override
					public Predicate toPredicate(Root<ReportsView> root, CriteriaQuery<?> query, CriteriaBuilder cb) {
						List<Predicate> predicates = new ArrayList<>();

						getQueryData(searchRequest, getMatchedReport, root, cb, predicates);

						return cb.and(predicates.toArray(new Predicate[0]));
					}
				});
				int pageNumber = resetPageNumbersOnCount(searchRequest, countOfReports);
				logger.info("countOfReports is " + countOfReports);
				if (countOfReports > 0) {
					logger.info("ReportServiceImpl.getReports(): COUNT(*) returned from DB is: " + countOfReports);
					reportSearchResponse = setGenericResponse(searchRequest, getMatchedReport, pageReq, countOfReports,
							pageNumber);
				} else {
					logger.info("ReportServiceImpl.getReports(): Empty count(*)! " + countOfReports);
					reportSearchResponse = new GenericResponse(searchRequest.getPageNumber(),
							searchRequest.getPageSize(), searchRequest.getSortColumn(), countOfReports);
					return reportSearchResponse;
				}
			} else {
				logger.info("Returning exception: at least 1 field information needs to be added");
				throw new GenericException(ErrorMessages.INVALID_REQUEST,
						"At least one field information needs to be added", HttpStatus.BAD_REQUEST);
			}
		} catch (Exception e) {
			throw new GenericException(ErrorMessages.DATA_ACCESS_ERROR, e.getMessage(),
					HttpStatus.INTERNAL_SERVER_ERROR);
		}
		return reportSearchResponse;
	}

	private PageRequest generatePaginationWithSorting(SearchReportRequest searchRequest) {
		PageRequest pageReq;
		String sortColumn = setSortColumn(searchRequest.getSortColumn());
		if (searchRequest.getDescending()) {
			pageReq = new PageRequest(searchRequest.getPageNumber().intValue() - 1,
					searchRequest.getPageSize().intValue(), Sort.Direction.DESC, sortColumn);
		} else {
			pageReq = new PageRequest(searchRequest.getPageNumber().intValue() - 1,
					searchRequest.getPageSize().intValue(), Sort.Direction.ASC, sortColumn);
		}
		return pageReq;
	}

	private String setSortColumn(String sortColumn) {
		if (StringUtils.isNotBlank(sortColumn)) {
			switch (sortColumn) {
			case ReportsViewConstants.ATTACHMENT_ID:
				return "attachmentId";
			case ReportsViewConstants.CLAIM_IDENTIFIER:
				return "externalClaimId";
			case ReportsViewConstants.FACILITY:
				return "facility";
			case ReportsViewConstants.PROVIDER_NAME:
				return "providerName";
			case ReportsViewConstants.PROVIDER_ZIP:
				return "providerZip";
			case ReportsViewConstants.PROVIDER_NPI:
				return "providerNpi";
			case ReportsViewConstants.PATIENT_FIRST_NAME:
				return "patientFirstName";
			case ReportsViewConstants.PATIENT_LAST_NAME:
				return "patientLastName";
			case ReportsViewConstants.PATIENT_IDENTIFIER:
				return "patientIdentifierNumber";
			case ReportsViewConstants.PATIENT_CONTROL_NUMBER:
				return "patientControlNumber";
			case ReportsViewConstants.CLAIM_SERVICE_START_DATE:
				return "claimServiceStartDate";
			case ReportsViewConstants.CLAIM_SERVICE_END_DATE:
				return "claimServiceEndDate";
			case ReportsViewConstants.ATTACHMENT_CONTROL_NUMBER:
				return "attachmentControlNumber";
			case ReportsViewConstants.REPORT_CODE:
				return "reportCode"; // fixed reportcode to reportCode
			case ReportsViewConstants.STATUS:
				return "status";
			case ReportsViewConstants.DATE_RECEIVED:
				return "dateReceived";
			case ReportsViewConstants.PATIENT_DOB:
				logger.info("========== INSIDE THE SWITCH FOR PATIENTDOB! MY SORT COLUMN WAS " + sortColumn);
				return "patientDob";
			default:
				return "attachmentId";
			}
		} else {
			return "attachmentId";
		}
	}

	private boolean isRequestNotEmpty(SearchReportRequest searchRequest) {
		// this stream filters out any null or empty
		return Stream.of(searchRequest.getPayerId(), searchRequest.getStartDate(), searchRequest.getEndDate())
				.filter(t -> t != null).filter(t -> !((String) t).isEmpty()).findAny().isPresent();

	}

	private void getQueryData(SearchReportRequest searchRequest, boolean getMatchedReport, Root<ReportsView> root,
			CriteriaBuilder cb, List<Predicate> predicates) {
		// evaluating whether to show matched or unmatched
		if (getMatchedReport) {
			logger.info("4a. inside the IF that evaluates ExternalClaimID, with getMatched value being "
					+ getMatchedReport);
			predicates.add(cb.isNotNull(root.get("externalClaimId")));
		} else {
			logger.info("4b. inside the IF that evaluates ExternalClaimID, with getMatched value being "
					+ getMatchedReport);
			predicates.add(cb.isNull(root.get("externalClaimId")));
		}

		// evaluating payerId in request
		if (StringUtils.isNotBlank(searchRequest.getPayerId())) {
			logger.info("5. Payer Id " + searchRequest.getPayerId());
			predicates.add(cb.equal(root.get("payerId"), searchRequest.getPayerId()));
		}

		if (StringUtils.isNotBlank(searchRequest.getStartDate())
				&& StringUtils.isNotBlank(searchRequest.getEndDate())) {
			// evaluating startDate and endDate in request
			String startDateInString = "";
			String endDateInString = "";
			try {
				Date startDate = formatDateOfRequest.parse(searchRequest.getStartDate());
				startDateInString = formatDateOfDB.format(startDate);

				Date endDate = formatDateOfRequest.parse(searchRequest.getEndDate());
				endDateInString = formatDateOfDB.format(endDate);

				logger.info("Start Date " + startDateInString);
				logger.info("End Date " + endDateInString);
			} catch (ParseException e) {
				logger.info("unable to parse current Date in search request");
				e.printStackTrace();
			}
			predicates.add(cb.greaterThanOrEqualTo(root.get("claimServiceStartDate"), startDateInString));
			predicates.add(cb.lessThanOrEqualTo(root.get("claimServiceStartDate"), endDateInString));

		}

		logger.info("end of getQueryData method");

	}

	private int resetPageNumbersOnCount(SearchReportRequest searchRequest, Long countOfReports) {
		int pageNumber = searchRequest.getPageNumber();

		if (countOfReports <= (searchRequest.getPageSize() * searchRequest.getPageNumber())) {
			Integer resultCountIntVal = countOfReports.intValue();
			float pageNumberFloat = (float) resultCountIntVal / (float) searchRequest.getPageSize();
			pageNumber = (int) Math.ceil(pageNumberFloat);

			logger.info("resetPageNumbersOnCount() : pageNumber is :" + pageNumber + " and pageSize is :"
					+ searchRequest.getPageSize());
		}
		return pageNumber;
	}

	private GenericResponse setGenericResponse(SearchReportRequest searchRequest, boolean getMatchedReport,
			PageRequest pageReq, Long resultCount, int pageNumber) {
		logger.info("8. in setGenericResponse method");
		GenericResponse searchResponse;
		String sortColumn = StringUtils.isBlank(searchRequest.getSortColumn()) ? "attachmentId"
				: searchRequest.getSortColumn();
		searchResponse = new GenericResponse(searchRequest.getPageNumber(), searchRequest.getPageSize(), sortColumn,
				resultCount);

		List<ReportResponse> responseList = querySpecification(searchRequest, getMatchedReport, pageReq);
		searchResponse.setResponse(responseList);

		return searchResponse;
	}

	private List<ReportResponse> querySpecification(SearchReportRequest searchRequest, boolean getMatchedReport,
			PageRequest pageReq) {
		logger.info("9. in querySpecification method");
		Page<ReportsView> rvPage = reportsViewRepository.findAll(new Specification<ReportsView>() {
			@Override
			public Predicate toPredicate(Root<ReportsView> root, CriteriaQuery<?> query, CriteriaBuilder cb) {
				List<Predicate> predicates = new ArrayList<>();
				logger.info("about to call getQueryData again");
				getQueryData(searchRequest, getMatchedReport, root, cb, predicates);

				return cb.and(predicates.toArray(new Predicate[0]));
			}
		}, pageReq);
		logger.info("finished querySpecification's repository.findAll()");

		return assignData(rvPage);
	}

	private List<ReportResponse> assignData(Page<ReportsView> reportsViewPage) {
		logger.info("in assignData");

		List<ReportResponse> responseList = new ArrayList<>();
		for (ReportsView rv : reportsViewPage) {
			String claimStartString = "";
			String claimEndString = "";
			String dateReceivedStr = "";
			String patientDobStr = "";
			try {
				if (StringUtils.isNotBlank(rv.getClaimServiceStartDate())) {
					Date claimStartDate = formatDateOfDB.parse(rv.getClaimServiceStartDate());
					claimStartString = formatDateOfRequest.format(claimStartDate);
				}

				if (StringUtils.isNotBlank(rv.getClaimServiceEndDate())) {
					Date claimEndDate = formatDateOfDB.parse(rv.getClaimServiceEndDate());
					claimEndString = formatDateOfRequest.format(claimEndDate);
				}

				if (StringUtils.isNotBlank(rv.getDateReceived())) {
					Date dateReceived = formatDateOfDB.parse(rv.getDateReceived());
					dateReceivedStr = formatDateOfRequest.format(dateReceived);
				}

				if (StringUtils.isNotBlank(rv.getPatientDob())) {
					Date patientDob = formatDateOfDB.parse(rv.getPatientDob());
					patientDobStr = formatDateOfRequest.format(patientDob);
				}
			} catch (ParseException e) {
				logger.info("unable to parse current Date in search request");
				e.printStackTrace();
			}

			ReportResponse response = new ReportResponse();

			response.setAttachmentId(rv.getAttachmentId());
			response.setProviderName(rv.getProviderName());
			response.setProviderZip(rv.getProviderZip());
			response.setProviderNpi(rv.getProviderNpi());
			response.setPatientFirstName(rv.getPatientFirstName());
			response.setPatientLastName(rv.getPatientLastName());
			response.setPatientName(rv.getPatientLastName() + ", " + rv.getPatientFirstName());
			response.setPatientIdentifierNumber(rv.getPatientIdentifierNumber());
			response.setPatientControlNumber(rv.getPatientControlNumber());
			response.setClaimServiceStartDate(claimStartString);
			response.setClaimServiceEndDate(claimEndString);
			response.setExternalClaimId(rv.getExternalClaimId());
			response.setAttachmentControlNumber(rv.getAttachmentControlNumber());
			response.setStatus(rv.getStatus());
			response.setPayerId(rv.getPayerId());
			response.setDateReceived(dateReceivedStr);
			response.setFacility(rv.getFacility());
			response.setReportCode(rv.getReportCode());
			response.setPatientDob(patientDobStr);
			responseList.add(response);
		}
		return responseList;
	}

}
