package gov.va.med.ars.service.impl;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.dao.DataAccessException;
import org.springframework.dao.DataAccessResourceFailureException;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import gov.va.med.ars.dao.ars.IArsUserRepository;
import gov.va.med.ars.dao.ars.IArsUserRolePermissionsRepository;
import gov.va.med.ars.dao.ars.IAuthenticatorRepository;
import gov.va.med.ars.exceptions.GenericException;
import gov.va.med.ars.model.response.UserDetailsResponse;
import gov.va.med.ars.model.response.UserInfoResponse;
import gov.va.med.ars.model.response.UserRolePermissions;
import gov.va.med.ars.model.response.UserRoles;
import gov.va.med.ars.service.IAuthenticatorService;
import gov.va.med.domain.ars.ArsUser;
import gov.va.med.domain.ars.ArsUserRoleUsages;
import gov.va.med.domain.ars.SsoAuthentication;
import net.minidev.json.JSONObject;

@Service
public  class AuthenticatorServiceImpl implements IAuthenticatorService {
	
	private static final Logger logger = LogManager.getLogger(AuthenticatorServiceImpl.class);
	
	@Autowired 
	IAuthenticatorRepository authenticatorRepository;
	
	@Autowired
	IArsUserRepository arsUserRepository;
	
	@Autowired
	IArsUserRolePermissionsRepository userRolePermissionRepository;
	
	@Value("${postBack.url}")
    private String postBackUrl;
    
    @Value("${captured.url}")
    private String capturedUrl;
    
    @Value("${aspCheckPage.url}")
    private String aspCheckPageUrl;
    
	private boolean isAuthenticated = false;
	
	private List<ArsUser> arsUsers;
	private List<UserRoles> userRoles;
	private ArsUser arsUser;
	
	/**
	 * 
	 */
	public AuthenticatorServiceImpl() {
		arsUser = new ArsUser();
		arsUsers = new ArrayList<>();
		userRoles = new ArrayList<>();
	}
	
	/**
	 * @param authenticatorRepository
	 */
	public void setAuthenticatorDAO(IAuthenticatorRepository authenticatorRepository) {
		this.authenticatorRepository = authenticatorRepository;
	}
	
	/**
	 * @param appUserRepository
	 */
	public void setAppUserRoleRepository(IArsUserRepository appUserRepository) {
		this.arsUserRepository = appUserRepository;
	}
	
	/**
	 * The method accept GUID and query from SsoAuthentication table
	 * and return the UserName. I used JDBCTemplate and manually mapped the result to the object
	 * @param guid
	 * @return String
	 */
	@Override
	public String getUsernameFromAspAuth(String guid) throws GenericException {
		//For testing only
		try {
			String userName = null;
			List<SsoAuthentication> results = authenticatorRepository.findByguid(guid);
			
			if(!results.isEmpty()) {
				SsoAuthentication s = results.get(0);
				userName = s.getNtname();
				logger.info("getUsernameFromAspAuth() Retrieved UserName: "+userName +" from Asp Auth");
			}
			return userName;
		} catch(IllegalArgumentException e) {
			// this exception is thrown by the JPA find method when the claimIndex is invalid format or null
			logger.info("authenticate : invalid_input_error for "+guid+" "+e);
			throw new GenericException("invalid_input_error",e.getMessage(), HttpStatus.BAD_REQUEST);	
		} catch(DataAccessResourceFailureException e) {
			// this exception is thrown by the database connection failure
			logger.info("authenticate  : database_connection_error "+e);
			throw new GenericException("database_connection_error",e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);		
		} catch(DataAccessException e){
			// this is root for all the data access exceptions by spring framework
			logger.info("authenticate : data_access_error "+e);
			throw new GenericException("data_access_error", e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);	
		}  catch(Exception e) {
			logger.info("authenticate : internal_server_error  "+e);
			throw new GenericException("internal_server_error", e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);
		}
	}
	/**
	 * This method will authenticate GUID of User and get AppUser
	 * @param guid
	 * @return boolean
	 */
	@Override
	public final boolean authenticate(String guid) throws GenericException {
		logger.info("autoLogin : Performing AutoLogin");
		try {
			logger.info(" guid from the request is :"+guid);
			//String decryptedGuid = IDCheckAESEncryption.decrypt(guid);
			String decryptedGuid = guid;
			logger.info("decryptedGuid is :"+decryptedGuid);
			String userName = getUsernameFromAspAuth(decryptedGuid);
			if(userName != null) {
				// Get AppUser using userName
				 arsUsers = arsUserRepository.findByArsUserName(userName);
				
				 
				 isAuthenticated = true;
			} else {
				logger.error("authenticate(): could not obtain userName from ASP. Authentication module error ");
				isAuthenticated = false;
				throw new GenericException("Asp.Authentication Failed","Could not obtain userName for guid:"+guid+" from ASP Authentication module",HttpStatus.BAD_REQUEST);
			}	
			return isAuthenticated;
		}  catch(IllegalArgumentException e) {
			// this exception is thrown by the JPA find method when the claimIndex is invalid format or null
			logger.info("authenticate : invalid_input_error for "+guid+" "+e);
			throw new GenericException("invalid_input_error",e.getMessage(), HttpStatus.BAD_REQUEST);	
		} catch(DataAccessResourceFailureException e) {
			// this exception is thrown by the database connection failure
			logger.info("authenticate  : database_connection_error "+e);
			throw new GenericException("database_connection_error",e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);		
		} catch(DataAccessException e){
			// this is root for all the data access exceptions by spring framework
			logger.info("authenticate : data_access_error "+e);
			throw new GenericException("data_access_error", e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);	
		} catch(GenericException e ) {
			throw e;
		} catch(Exception e) {
			logger.info("authenticate : internal_server_error  "+e);
			throw new GenericException("internal_server_error", e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);
		}
		
	}
	
	/**
	 * This method will authenticate GUID of User and get AppUser
	 * @param guid
	 * @return boolean
	 */
	@Override
	public final boolean authenticateUser(String userName) throws GenericException {
		logger.info("autoLogin : Performing AutoLogin");
		try {
			logger.info(" userName from the request is :"+userName);
			if(userName != null) {
				// Get AppUser using userName
				 logger.info("Searching for userName "+userName);
				 arsUsers = arsUserRepository.findByArsUserName(userName);
				 logger.info("List of users");
				 				 
/*				 for(int i=0;i<arsUsers.size();i++){
					    System.out.println(arsUsers.get(i).getArsUserId());
					}*/
				 isAuthenticated = true;
			} else {
				logger.error("authenticate(): could not obtain userName from ASP. Authentication module error ");
				isAuthenticated = false;
				throw new GenericException("Asp.Authentication Failed","Could not obtain userName "+userName+" from ASP Authentication module",HttpStatus.BAD_REQUEST);
			}	
			return isAuthenticated;
		}  catch(IllegalArgumentException e) {
			// this exception is thrown by the JPA find method when the claimIndex is invalid format or null
			logger.info("authenticate : invalid_input_error for "+userName+" "+e);
			throw new GenericException("invalid_input_error",e.getMessage(), HttpStatus.BAD_REQUEST);	
		} catch(DataAccessResourceFailureException e) {
			// this exception is thrown by the database connection failure
			logger.info("authenticate  : database_connection_error "+e);
			throw new GenericException("database_connection_error",e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);		
		} catch(DataAccessException e){
			// this is root for all the data access exceptions by spring framework
			logger.info("authenticate : data_access_error "+e);
			throw new GenericException("data_access_error", e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);	
		} catch(GenericException e ) {
			throw e;
		} catch(Exception e) {
			logger.info("authenticate : internal_server_error  "+e);
			throw new GenericException("internal_server_error", e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);
		}
		
	}
	
	/**
	 * Update the lastlogin of AppUser
	 * @param arsuser
	 */
	private void updateLastLogin(ArsUser arsuser) throws GenericException {
		try {
			logger.info("UpdateLastLogin() : Updating userLastLogin for "+arsuser.getUserName());
			arsuser.setLastLoginDate(new Date());
			arsUserRepository.save(arsuser);
			
		} catch(DataAccessResourceFailureException e) {
			// this exception is thrown by the database connection failure
			logger.info("updateLastLogin  : database_connection_error "+e);
			throw new GenericException("database_connection_error",e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);		
		} catch(DataAccessException e){
			// this is root for all the data access exceptions by spring framework
			logger.info("updateLastLogin : data_access_error "+e);
			throw new GenericException("data_access_error", "Unable to update user lastLoginDate", HttpStatus.INTERNAL_SERVER_ERROR);	
		} catch(Exception e) {
			logger.info("updateLastLogin : internal_server_error  "+e);
			throw new GenericException("internal_server_error", e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);
		}
	}
	
	/**
	 * This method return list of Roles assigned 
	 * @param guid
	 * @return List<UserRoles>
	 */
	@Override
	public UserInfoResponse getUserRoles() throws GenericException {
		logger.info( "Getting user role details ");
		UserInfoResponse userInfoResponse = null;
		Set<String> roles = new HashSet<>();
		Set<Long> assignedRoles = null;
		try {
			if(isAuthenticated) {
				//AppUser appUser = new AppUser();
				this.userRoles = new ArrayList<>();
				if(!arsUsers.isEmpty()) {
					arsUser = arsUsers.get(0);
					
					//Get user who is enabled
					if(arsUser.getEnabled().equalsIgnoreCase("Y")) {
						Set<ArsUserRoleUsages> userRoleUsages = arsUser.getArsUserRoleUsageses();
						// Update App_User LastLoginDate
						updateLastLogin(arsUser);
						
						if(!userRoleUsages.isEmpty()) {
							assignedRoles = new HashSet<>();
							// Add the role name description and send it to the controller
							for(ArsUserRoleUsages u : userRoleUsages) {
								UserRoles userRole = new UserRoles();
								userRole.setRoleName(u.getArsRole().getRoleName());
								userRole.setDescription(u.getArsRole().getDescription());
								userRoles.add(userRole);
								
								// add the user roles to the set
								assignedRoles.add(u.getArsRole().getArsRoleId());
								roles.add(u.getArsRole().getRoleName());
							}
						} else {
							logger.error("getUserRoles(): No Role found for User "+arsUser.getUserName());
							throw new GenericException("UserRole Error","No UserRole found for "+arsUser.getUserName(),HttpStatus.BAD_REQUEST);
						}
						
						userInfoResponse = new UserInfoResponse();
						userInfoResponse.setUserName(arsUser.getUserName());
					//	userInfoResponse.setUserName(arsUser.getUserName());
						
						SimpleDateFormat dateFormat = new SimpleDateFormat("MM/dd/yyyy HH:mm:ss");
						
						userInfoResponse.setLastLogin(dateFormat.format(arsUser.getLastLoginDate()));
						userInfoResponse.setUserRoles(roles);
						
						// set permissions of the user
						logger.debug("Setting permissions of the user");
						JSONObject permissionAssociatedWithRoles=getPermissions(assignedRoles);
						// adding edit and export permissions
						if(arsUser.getEnableEdit()){
							permissionAssociatedWithRoles.put("editPermission", true);
						}							
						else{
							permissionAssociatedWithRoles.put("editPermission", false);
						}
						
						if(arsUser.getEnableExport()){
							permissionAssociatedWithRoles.put("exportPermission", true);
						}							
						else{
							permissionAssociatedWithRoles.put("exportPermission", false);
						}
							
						userInfoResponse.setPermissions(permissionAssociatedWithRoles);
						logger.debug("The roles of the user are :"+userInfoResponse.toString());
					}
				}
				
				return userInfoResponse;
			}
			else {
				logger.error("getUserRoles(): GUID is not Authenticated ");
			}
			return userInfoResponse;
		}  catch(Exception e) {
			logger.info("updateLastLogin : internal_server_error  "+e);
			throw new GenericException("internal_server_error", e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);
		}

	}
		
/*	 (non-Javadoc)
	 * @see gov.va.med.fee.service.IAuthenticatorService#getAuthParams(java.lang.String)
	 
	@Override
	public AuthenticationParamResponse getAuthParams(String sessionId) throws GenericException{
		logger.info("getAuthParams() :"+sessionId);
	
		String aspChkPageUrl= this.aspCheckPageUrl;
		String encryptedPostBackUrl = null;
		String encryptedCapturedUrl = null;
		try {
			encryptedPostBackUrl = URLEncoder.encode(IDCheckAESEncryption.encrypt(this.postBackUrl), "UTF-8");
			encryptedCapturedUrl = URLEncoder.encode(IDCheckAESEncryption.encrypt(this.capturedUrl), "UTF-8");
		} catch (Exception e) {
			logger.error("Exception while encripting urls"+e.getMessage());
			throw new GenericException("getAuthParamsFailed","Retrieve Authentication Params Failed",HttpStatus.INTERNAL_SERVER_ERROR);
		}
			
		AuthenticationParamResponse response = new AuthenticationParamResponse();
		response.setEncryptedCapturedUrl(encryptedCapturedUrl);
		response.setEncryptedPostBackUrl(encryptedPostBackUrl);
		response.setSessionId(sessionId);
		response.setAspCheckPageUrl(aspChkPageUrl);
		
		return response;
	}*/

	/**
	 * @return the appUsers
	 */
	@Override
	public List<ArsUser> getArsUsers() {
		return arsUsers;
	}

	/**
	 * @return the appUser
	 * @throws GenericException 
	 */
	@Override
	public ArsUser getArsUser(String decryptedGuid) throws GenericException {
		String userName = getUsernameFromAspAuth(decryptedGuid);
		if (userName != null) {
			// Get ArsUser using userName
			arsUsers = arsUserRepository.findByArsUserName(userName);
			if (!arsUsers.isEmpty()) {
				arsUser = arsUsers.get(0);
			}
		}

		return arsUser;
	}
	
	/**
	 * @param assignedRolesSet
	 * @return
	 */
	public JSONObject getPermissions(Set<Long> assignedRolesSet) {
		
		Set<String> assignedRolesString = new HashSet<>();
		if ( assignedRolesSet != null ) {
			assignedRolesSet.forEach(role -> {
				if( role != null ) {
					assignedRolesString.add(role.toString());
				}
			});
		}
		
		logger.debug(" assignedRolesString is :"+assignedRolesString);
		String roleIds = String.join(",", assignedRolesString);
		logger.debug(" roleIds is :"+roleIds);
		List<UserRolePermissions> userRolePermissions = userRolePermissionRepository.getUserRolePermissions(roleIds);
		
		logger.debug(" userRolePermissions from the repository are : ", userRolePermissions.toString());
		
		HashSet<String> pages = new HashSet<>();
		Map<String, Object> permissions = new HashMap<>();
		Map<String,Boolean> pagePermissions = new HashMap<>();
		
		for( UserRolePermissions userRolePermission : userRolePermissions ) {
			pages.add(userRolePermission.getPageName());
		}
		
		logger.debug(" List of the pages from the repository are : "+pages.toString());
		
		for( String page : pages ) {
			logger.debug(" Getting the permissions for the page : "+page);
			List<UserRolePermissions> filteredPermissions = userRolePermissions.stream().filter( permission -> permission.getPageName().equals(page)).collect(Collectors.toList());
			logger.debug(" filteredPermissions are : "+filteredPermissions);
			pagePermissions = new HashMap<>();
			
			for( UserRolePermissions filtered : filteredPermissions ) {	
				// when multiple roles has the same permissions,the below condition will set the permission as enabled if one of the role has it as enabled
				if( !(pagePermissions.containsKey(filtered.getPermissionName())) ||
						(pagePermissions.containsKey(filtered.getPermissionName()) && !(pagePermissions.get(filtered.getPermissionName()))))
				pagePermissions.put(filtered.getPermissionName(), filtered.getEnabled());
				else
				pagePermissions.put(filtered.getPermissionName(), Boolean.TRUE);
			}
			
			logger.debug(" Permissions for the page : "+page+" are :"+pagePermissions.toString());
			permissions.put( page, pagePermissions);
		}
	    
	    JSONObject permissionsJson = new JSONObject();
	    permissionsJson.putAll( permissions );
	    
	    logger.debug("Permissions of the user are : "+permissionsJson.toString());
	    return permissionsJson;
	}
	
	public List<String> getUsers(){
		return arsUserRepository.getAllUsers();
	}
	
}

