package gov.va.med.ars.security;

import java.io.UnsupportedEncodingException;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.KeySpec;

import javax.crypto.Cipher;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.SecretKeySpec;

import org.apache.log4j.Logger;

import java.util.Base64;

/**
 * 
 * This class provides AES level encryption. It was developed with the intention
 * assist in the prevention of sensitive data from being passed plain text
 * 
 * @author Ian Meinert
 * @version 1.0 February 23, 2018
 * 
 */
public class AesEncryption {

    /**
     * The logger for the class
     */
    private static final Logger LOGGER = Logger
            .getLogger(AesEncryption.class.getName());

    /**
     * Encrypts a given String
     * 
     * @param raw
     *            The raw string to encrypt
     * @return an encrypted String
     * @throws InvalidKeyException
     * @throws InvalidAlgorithmParameterException
     * @throws UnsupportedEncodingException
     * @throws Exception
     */
    public static String encrypt(String raw)
            throws InvalidKeyException, InvalidAlgorithmParameterException,
            UnsupportedEncodingException, Exception {
        LOGGER.debug("raw: " + raw);

        Cipher cipher = getCipher(Cipher.ENCRYPT_MODE);

        byte[] encryptedVal = cipher.doFinal(raw.getBytes("UTF-8"));
        LOGGER.debug("encryptedVal: " + new String(encryptedVal));

        byte[] encodedBytes = Base64.getEncoder().encode(encryptedVal);
        LOGGER.debug("encodedBytes: " + new String(encodedBytes));

        return new String(encodedBytes);
    }

    /**
     * Decrypts a given String
     * 
     * @param encrypted
     *            The encrypted string to decrypt
     * @return
     * @throws InvalidKeyException
     * @throws InvalidAlgorithmParameterException
     * @throws UnsupportedEncodingException
     * @throws Exception
     */
    public static String decrypt(String encrypted)
            throws InvalidKeyException, InvalidAlgorithmParameterException,
            UnsupportedEncodingException, Exception {
        LOGGER.debug("encrypted: " + new String(encrypted));

        Cipher cipher = getCipher(Cipher.DECRYPT_MODE);

        byte[] decodedValue = Base64.getDecoder().decode(encrypted);
        LOGGER.debug("decodedValue: " + new String(decodedValue));

        byte[] decValue = cipher.doFinal(decodedValue);
        LOGGER.debug("decValue: " + new String(decValue));

        return new String(decValue);
    }

    /**
     * Initializes the cipher for either encryption or decryption
     * 
     * @param mode
     *            The mode of the encryption
     * @return
     * @throws InvalidKeyException
     * @throws InvalidAlgorithmParameterException
     * @throws UnsupportedEncodingException
     * @throws Exception
     */
    private static Cipher getCipher(int mode)
            throws InvalidKeyException, InvalidAlgorithmParameterException,
            UnsupportedEncodingException, Exception {
        Cipher cipher = Cipher.getInstance("AES/CBC/PKCS5Padding");

        cipher.init(mode, generateKey(),
                new IvParameterSpec(EncryptionConstants.IV.getBytes("UTF-8")));
        return cipher;
    }

    /**
     * Generates a key using EncryptionConstants
     * 
     * @return The cipher Key
     * @throws NoSuchAlgorithmException
     * @throws UnsupportedEncodingException
     * @throws InvalidKeySpecException
     */
    private static Key generateKey() throws NoSuchAlgorithmException,
            UnsupportedEncodingException, InvalidKeySpecException {
        SecretKeyFactory factory = SecretKeyFactory
                .getInstance("PBKDF2WithHmacSHA1");

        KeySpec spec = new PBEKeySpec(
                EncryptionConstants.PASSWORD.toCharArray(),
                EncryptionConstants.SALT.getBytes("UTF-8"),
                EncryptionConstants.ITERATION_COUNT,
                EncryptionConstants.KEY_LENGTH);

        SecretKey tmp = factory.generateSecret(spec);
        byte[] encoded = tmp.getEncoded();

        LOGGER.debug("encoded: " + new String(encoded));

        return new SecretKeySpec(encoded, "AES");
    }
}
