import { Component, OnInit, ViewChild, ElementRef } from '@angular/core';
import { FormBuilder, Validators, FormGroup } from '@angular/forms';
import { Router } from '@angular/router';

import { ScriptService } from './../shared/scriptLoader/script.service';
import {
  PrimarySearchModel,
  AdvancedSearchModel,
  LookupResponseModel
} from './rfai-lookup.model';
import { RfaiLookupService } from './rfai-lookup.service';
import { TableService } from './../shared/table/table.service';
import {
  TableColumnModel,
  TableSettings,
  PaginationSettings
} from './../shared/table/table.component.model';

@Component({
  selector: 'app-rfai-lookup',
  templateUrl: './rfai-lookup.component.html',
  styleUrls: [
    './../rfai-form/rfai-form.component.scss',
    './rfai-lookup.component.scss'
  ],
  providers: [ScriptService]
})
export class RfaiLookupComponent implements OnInit {
  // table setup
  tableColumns: TableColumnModel[] = [
    new TableColumnModel('Submission ID', 'submissionId'),
    new TableColumnModel('PDI / Claim ID', 'pdiClaimId'),
    new TableColumnModel('Billing Provider Name', 'providerName'),
    new TableColumnModel('Status', 'status'),
    // new TableColumnModel('Provider NPI', 'providerNpi'),
    // new TableColumnModel('Provider TIN', 'providerTin'),
    new TableColumnModel('Patient Name', 'patientName'),
    new TableColumnModel('Date Submitted', 'datesubmitted'),
    new TableColumnModel('Required Response Date', 'requiredResponseDate')
  ];
  tableSettings: TableSettings = new TableSettings();
  paginationSettings: PaginationSettings = new PaginationSettings();

  // end of table setup

  primaryForm: FormGroup;
  advancedForm: FormGroup;
  primaryModel: PrimarySearchModel = new PrimarySearchModel();
  advancedModel: AdvancedSearchModel = new AdvancedSearchModel();
  typeOfSearch = 'P';
  validDateRange = true;
  errorMsg = '';
  errorStatus = false;
  successStatus = false;
  successMsg = '';
  row = {};
  searchResults: LookupResponseModel[];
  showResults = false;

  @ViewChild('serviceStartDate')
  serviceStartDate: ElementRef = new ElementRef('');
  @ViewChild('serviceEndDate') serviceEndDate: ElementRef = new ElementRef('');
  @ViewChild('primaryDiv') primaryDiv: ElementRef = new ElementRef('');
  @ViewChild('advancedDiv') advancedDiv: ElementRef = new ElementRef('');
  @ViewChild('primaryRadioButton')
  primaryRadioButton: ElementRef = new ElementRef('');
  @ViewChild('advancedRadioButton')
  advancedRadioButton: ElementRef = new ElementRef('');
  constructor(
    private fb: FormBuilder,
    private script: ScriptService,
    private rfaiLookupService: RfaiLookupService,
    private tableService: TableService,
    private router: Router
  ) {
    this.script
      .load('datepicker')
      .then(data => {})
      .catch(error => console.log(error));
  }

  ngOnInit() {
    this.formInit();
    this.showPrimary();
    this.primaryRadioButton.nativeElement.focus = true;
    // table Settings
    this.tableSettings.rowNumberDisplay = false;
    this.tableSettings.checkBoxesShow = false;
    this.tableSettings.buttonShow = false;
    this.tableSettings.linkColumn = false;
    this.tableSettings.imgColumn = true;
    this.tableSettings.imageUrls = ['./../../assets/images/edit.png'];
    this.tableSettings.imageColumnNames = [];
  }

  // Initialize form and set up the Validations
  formInit() {
    this.primaryForm = this.fb.group({
      primarySearchParam: [this.primaryModel.number]
    });

    this.advancedForm = this.fb.group({
      pdiClaimId: [this.advancedModel.number, [Validators.pattern('[0-9]*')]],
      billingProvider: [this.advancedModel.providerName],
      providerNpi: [
        this.advancedModel.providerNpi,
        [Validators.pattern('[0-9]*')]
      ],
      providerTin: [
        this.advancedModel.providerTin,
        [Validators.pattern('[0-9]*')]
      ],
      patientLastname: [this.advancedModel.patientLastName],
      patientFirstName: [this.advancedModel.patientfirstName],
      patientIdentifier: [
        this.advancedModel.patientIdentifier,
        [Validators.pattern('[0-9]*')]
      ],
      patientcontrolNumber: [
        this.advancedModel.patientControlNumber,
        [Validators.pattern('[0-9]*')]
      ],
      medicalRecordNumber: [
        this.advancedModel.medicalRecordNumber,
        [Validators.pattern('[0-9]*')]
      ],
      // lineItemControlNumber: [
      //   this.advancedModel.lineItemControlNumber,
      //   [Validators.pattern('[0-9]*')]
      // ],
      serviceStartDate: [
        this.serviceStartDate.nativeElement.value,
        [
          Validators.pattern(
            '(0[1-9]|1[012])[/](0[1-9]|[12][0-9]|3[01])[/](19|20)\\d\\d'
          )
        ]
      ],
      serviceEndDate: [
        this.serviceEndDate.nativeElement.value,
        [
          Validators.pattern(
            '(0[1-9]|1[012])[/](0[1-9]|[12][0-9]|3[01])[/](19|20)\\d\\d'
          )
        ]
      ]
    });
  }

  primarySearch() {
    // console.log('searchParam', this.primaryModel.number);
    // console.log('paginationSettings before rest', this.paginationSettings);
    // this.primaryModel.number = +this.primaryModel.number;
    // update table settings
    this.primaryModel.pageSize = this.paginationSettings.pageSize;
    this.primaryModel.pageNumber = this.paginationSettings.currentPage;
    this.primaryModel.descending = this.paginationSettings.descending;
    this.primaryModel.sortColumn = this.paginationSettings.sortColumn;
    this.rfaiLookupService.quickSearch(this.primaryModel).subscribe(data => {
      // console.log('data from quick search', data);
      if (data.errorCode) {
        this.paginationSettings = new PaginationSettings();
        // console.log('data error pagination settings ', this.paginationSettings);
        this.errorStatus = true;
        this.errorMsg = 'Error in processing search request.';
        this.searchResults = [];
        this.showResults = false;
      } else {
        this.searchResults = data.response;
        this.showResults = true;
        this.errorStatus = false;
        this.errorMsg = '';
        // service to setup table
        // this.tableService.setData([this.tableColumns, this.searchResults]);
        this.paginationSettings = {
          currentPage: data.pageNumber,
          pageSize: data.pageSize,
          totalPages: Math.ceil(data.totalNumberOfResults / data.pageSize),
          totalResults: data.totalNumberOfResults,
          sortColumn: data.sortColumn,
          descending: this.primaryModel.descending
        };
      }
    });
  }
  clearPrimary() {
    this.primaryModel = new PrimarySearchModel();
    this.formInit();
    this.showResults = false;
    this.errorStatus = false;
    this.errorMsg = '';
  }
  advancedSearch() {
    // this.advancedModel.number = +this.advancedModel.number;
    this.advancedModel.serviceStartDate = this.serviceStartDate.nativeElement.value;
    this.advancedModel.serviceEndDate = this.serviceEndDate.nativeElement.value;
    // update table settings
    this.advancedModel.pageSize = this.paginationSettings.pageSize;
    this.advancedModel.pageNumber = this.paginationSettings.currentPage;
    this.advancedModel.descending = this.paginationSettings.descending;
    this.advancedModel.sortColumn = this.paginationSettings.sortColumn;
    // console.log('advancedSearch', this.advancedModel);
    this.rfaiLookupService
      .advancedSearch(this.advancedModel)
      .subscribe(data => {
        if (data.errorCode) {
          this.errorStatus = true;
          this.errorMsg = 'Error in processing search request.';
          this.searchResults = [];
          this.showResults = false;
        } else {
          this.searchResults = data.response;
          // console.log('advancedSearch Results', this.searchResults);
          this.showResults = true;
          this.errorStatus = false;
          this.errorMsg = '';

          // service to setup table
          // this.tableService.setData([this.tableColumns, this.searchResults]);
          this.paginationSettings = {
            currentPage: data.pageNumber,
            pageSize: data.pageSize,
            totalPages: Math.ceil(data.totalNumberOfResults / data.pageSize),
            totalResults: data.totalNumberOfResults,
            sortColumn: data.sortColumn,
            descending: this.advancedModel.descending
          };
        }
      });
  }
  clearAdvancedForm() {
    this.advancedModel = new AdvancedSearchModel();
    this.showResults = false;
    this.errorStatus = false;
    this.errorMsg = '';
    this.formInit();
  }

  // validate the date range between the two date fields. fromDate should be less than toDate.
  datesValidator() {
    if (
      this.serviceStartDate.nativeElement.value.length === 10 &&
      this.serviceEndDate.nativeElement.value.length === 10
    ) {
      this.advancedForm
        .get('serviceStartDate')
        .setValue(this.serviceStartDate.nativeElement.value);
      this.advancedForm
        .get('serviceEndDate')
        .setValue(this.serviceEndDate.nativeElement.value);
      const fromDate = new Date(
        this.advancedForm.get('serviceStartDate').value
      );
      const toDate = new Date(this.advancedForm.get('serviceEndDate').value);
      if (toDate.valueOf() < fromDate.valueOf()) {
        this.validDateRange = false;
        this.advancedForm.get('serviceStartDate').setValue('');
        this.advancedForm.get('serviceEndDate').setValue('');
      }
    } else {
      this.validDateRange = true;
    }
  }

  // Methods to toggle Display between Quick Search and Advanced Search.
  // Using display instead of ngIf as calendar not functioning when ngIf is used.
  showAdvanced() {
    this.primaryDiv.nativeElement.style.display = 'none';
    this.advancedDiv.nativeElement.style.display = 'block';
    this.clearPrimary();
    this.searchResults = [];
    this.showResults = false;
    this.errorStatus = false;
    this.errorMsg = '';
  }
  showPrimary() {
    this.advancedDiv.nativeElement.style.display = 'none';
    this.primaryDiv.nativeElement.style.display = 'block';
    this.clearAdvancedForm();
    this.searchResults = [];
    this.showResults = false;
    this.errorStatus = false;
    this.errorMsg = '';
  }
  updateTable() {
    if (this.typeOfSearch === 'P') {
      this.primarySearch();
    } else {
      this.advancedSearch();
    }
  }
  viewSubmissionDetails(row) {
    this.setSubmissionId(row.data.submissionId);
    this.router.navigate(['/viewSubmission']);
  }
  setSubmissionId(id) {
    this.rfaiLookupService.setSubmissionId(id);
  }
}
