import { Component, OnInit, ViewChild, ElementRef } from '@angular/core';
import { FormBuilder, Validators, FormGroup } from '@angular/forms';

import { ScriptService } from './../shared/scriptLoader/script.service';
import { RfaiFormService } from './rfai-form.service';
import {
  AdditionalInformationRequestModel,
  PDIInfoModel,
  LineModel,
  RfaiInfoModel
} from './rfai-form.model';
@Component({
  selector: 'app-rfai-form',
  templateUrl: './rfai-form.component.html',
  styleUrls: ['./rfai-form.component.scss'],
  providers: [ScriptService]
})
export class RfaiFormComponent implements OnInit {
  form: FormGroup;
  infoForm: FormGroup;
  lines: LineModel[] = [];
  modalClosed = true;
  selectedLines = [];
  enableAddLine = false;
  errorMsg = '';
  errorStatus = false;
  successStatus = false;
  successMsg = '';
  populateInfoClicked = false;
  prepopulatedInfo: PDIInfoModel = new PDIInfoModel();

  // Initial Values for the dropdown
  healthCareClaimStatusCodeDisplay = [
    'R0   General Requests-Requests that do not fall into other R-type categories',
    'R1   Entity Requests-Requests for information about specific entities',
    'R3   Claim/Line-Requests for information that could normally be submitted on a claim',
    'R4   Documentation-Requests for additional supporting documentation',
    'R5   More specific/detailed information is requested',
    'R6   Regulatory requirements',
    'R7   Confirm care is consistent with Health Plan policy coverage',
    'R8   Confirm care is consistent with health plan coverage exceptions',
    'R9   Determination of medical necessity',
    'R10  Support a filed grievance or appeal',
    'R11  Pre-payment review of claims',
    'R12  Clarification or justification of use for specified procedure code',
    'R13  Original documents submitted are not readable',
    'R14  Original documents received are not what was requested',
    'R15  Workers Compensation coverage determination',
    'R16  Eligibility determination',
    'R17  Replacement of a Prior Request'
  ];
  healthCareClaimStatusCodeValues = [
    'R0',
    'R1',
    'R3',
    'R4',
    'R5',
    'R6',
    'R7',
    'R8',
    'R9',
    'R10',
    'R11',
    'R12',
    'R13',
    'R14',
    'R15',
    'R16',
    'R17'
  ];
  loincCodeDisplay = [
    '34133-9	Continuity of Care Document',
    '11488-4	Consultation Note',
    '18748-4	Diagnostic Imaging Study',
    '18842-5	Discharge Summary',
    '34117-2	History and Physical Note',
    '11504-8	Provider-unspecified Operative Note',
    '28570-0	Provider-unspecified Procedure Note',
    '11506-3	Provider-unspecified Progress Note'
  ];
  loincCodeValues = [
    '34133-9',
    '11488-4',
    '18748-4',
    '18842-5',
    '34117-2',
    '11504-8',
    '28570-0',
    '11506-3'
  ];
  loincCodeModifierDisplay = [
    '18789-8	Include data of selected type within the date window associated with the service.',
    '18790-6	Include data of selected type on or before the date of service.',
    '18791-4	Include data of selected type within or aligned to a service.',
    '18792-2	Include data of selected type on or after the date of service.',
    '18803-7	Include data of selected type that represents observations made 30 days or fewer before the starting date of service.',
    '18804-5	Include data of selected type that represents observations made three months or fewer before the starting date of service.',
    '18805-2	Include data of selected type that represents observations made six months or fewer before the starting date of service.',
    '18806-0	Include data of selected type that represents observations made nine months or fewer before the starting date of service.',
    '18807-8	Include data of selected type that represents observations made one year or less before the starting date of service.',
    '53033-7	Include data of selected type that represents observations made two years or less before the starting date of service.',
    '18793-0	Use no fixed time limit on data—any of the selected type are relevant no matter when obtained.',
    '18794-8	Send all items of the specified type within the time window.',
    '18795-5	Send all items of the specified type within the time window relevant to the service.',
    '18796-3	Send all abnormals within the time window.',
    '18797-1	Send the first abnormals within the time window.',
    '18798-9	Send the last abnormals within the time window.',
    '18800-3	Send the worst abnormal result for each kind of observation in the time window.',
    '18799-7	Send the first (oldest) result for each kind of observation in the time window.',
    '18802-9	Send the last (most recent) within the time window.'
  ];
  loincCodeModifierValues = [
    '18789-8',
    '18790-6',
    '18791-4',
    '18792-2',
    '18803-7',
    '18804-5',
    '18805-2',
    '18806-0',
    '18807-8',
    '53033-7',
    '18793-0',
    '18794-8',
    '18795-5',
    '18796-3',
    '18797-1',
    '18798-9',
    '18800-3',
    '18799-7',
    '18802-9'
  ];
  model: AdditionalInformationRequestModel = new AdditionalInformationRequestModel();
  pdiInfoModel: PDIInfoModel = new PDIInfoModel();
  line: LineModel = new LineModel();
  // viewChild is required here due to the implementation of the calendar.
  @ViewChild('responseDueDate')
  responseDueDate: ElementRef = new ElementRef('');
  @ViewChild('serviceFromDate')
  serviceFromDate: ElementRef = new ElementRef('');
  @ViewChild('serviceToDate') serviceToDate: ElementRef = new ElementRef('');

  @ViewChild('selectAllLines') selectAllLines: ElementRef = new ElementRef('');

  constructor(
    private fb: FormBuilder,
    private rfaiAdditionalInformationService: RfaiFormService,
    private script: ScriptService
  ) {
    this.script
      .load('datepicker')
      .then(data => {})
      .catch(error => console.log(error));
  }

  ngOnInit() {
    this.formInit();
  }

  formInit() {
    this.form = this.fb.group({
      claimId: [this.model.claimId, Validators.pattern('[0-9]*')],
      claimStatus: [this.model.claimStatus, Validators.required],
      loincCode: [this.model.loincCode, Validators.required],
      loincCodeModifier: [this.model.loincCodeModifier, Validators.required],
      payerName: [this.pdiInfoModel.payerName, Validators.required],
      payerID: [
        this.pdiInfoModel.payerId,
        [Validators.required, Validators.pattern('[0-9]{0,15}')]
      ],
      informationReceiver: [
        this.pdiInfoModel.informationReceiver,
        Validators.required
      ],
      responseDueDate: [
        this.responseDueDate.nativeElement.value,
        Validators.required
      ],
      contactInfo: [this.pdiInfoModel.contactInformation],
      billingProvider: [
        this.pdiInfoModel.providerInformation,
        Validators.required
      ],
      providerNpi: [
        this.pdiInfoModel.providerNpi,
        [Validators.pattern('[0-9]*'), Validators.required]
      ],
      serviceProviderTIN: [
        this.pdiInfoModel.providerTin,
        [Validators.pattern('[0-9]*'), Validators.required]
      ],
      lastName: [this.pdiInfoModel.patientLastname, Validators.required],
      firstName: [this.pdiInfoModel.patientFirstName, Validators.required],
      patientID: [
        this.pdiInfoModel.patientIdentifier,
        [Validators.pattern('[0-9]*'), Validators.required]
      ],
      patientControlNumber: [
        this.pdiInfoModel.patientcontrolNumber,
        [Validators.pattern('[0-9]*'), Validators.required]
      ],
      payerClaimControlNumber: [
        this.pdiInfoModel.payerClaimControlNumber,
        [Validators.pattern('[0-9]*'), Validators.required]
      ],
      billType: [this.pdiInfoModel.billType, Validators.required],
      clearingHouseID: [
        this.pdiInfoModel.clearingHouseId,
        [Validators.pattern('[0-9]*'), Validators.required]
      ],
      medicalRecordNumber: [
        this.pdiInfoModel.medicalRecordNumber,
        Validators.pattern('[0-9]*')
      ],
      level: [this.pdiInfoModel.level, Validators.required],
      lines: this.lines,
      serviceFromDate: [
        this.serviceFromDate.nativeElement.value,
        Validators.required
      ],
      serviceToDate: [
        this.serviceToDate.nativeElement.value,
        Validators.required
      ]
    });
  }

  onSubmit() {
    if (this.populateInfoClicked) {
      this.lines = this.prepopulatedInfo.rfaiLineItemResponse;
      this.prepopulatedInfo.contactInformation = this.pdiInfoModel.contactInformation;
      this.pdiInfoModel = this.prepopulatedInfo;
    } else if (this.pdiInfoModel.level === 'C') {
      this.pdiInfoModel.serviceFromDate = this.serviceFromDate.nativeElement.value;
      this.pdiInfoModel.serviceToDate = this.serviceToDate.nativeElement.value;
      this.pdiInfoModel.responseDate = this.responseDueDate.nativeElement.value;
    } else if (this.pdiInfoModel.level === 'L') {
      this.pdiInfoModel.responseDate = this.responseDueDate.nativeElement.value;
    }

    this.selectedLines = this.lines.filter(line => line.selected);
    this.pdiInfoModel.rfaiLineItemResponse = this.selectedLines;
    this.model.claimId = this.model.claimId ? +this.model.claimId : null;
    let rfaiInfoModel: RfaiInfoModel = new RfaiInfoModel();
    rfaiInfoModel = Object.assign({}, this.model, this.pdiInfoModel);
    rfaiInfoModel.claimId = this.model.claimId
      ? +this.model.claimId
      : +rfaiInfoModel.payerClaimControlNumber;
    this.rfaiAdditionalInformationService
      .submitRfai(rfaiInfoModel)
      .subscribe(data => {
        if (data.result) {
          this.successMsg = '277 RFAI successfully submitted.';
          this.successStatus = true;
          this.errorStatus = false;
          setTimeout(() => {
            this.successStatus = false;
          }, 8000);
        } else {
          this.errorStatus = true;
          this.successStatus = false;
          this.errorMsg = 'An error occured during 277 RFAI submission.';
          setTimeout(() => {
            this.errorStatus = false;
          }, 8000);
        }
      });
  }
  onReset() {
    this.populateInfoClicked = false;
    this.errorStatus = false;
    this.successStatus = false;
    this.errorMsg = '';
    this.successMsg = '';
    this.pdiInfoModel = new PDIInfoModel();
    this.model = new AdditionalInformationRequestModel();
    this.lines = [new LineModel()];
    this.pdiInfoModel.level = 'C';
    this.formInit();
  }
  addNewInitialLine() {
    if (this.lines.length === 0) {
      this.addNewLine();
    }
  }
  addNewLine() {
    const line: LineModel = new LineModel();
    this.lines.push(line);
    this.enableAddLine = false;
  }
  populateInfo() {
    if (this.model.claimId) {
      this.populateInfoClicked = true;
      this.errorStatus = false;
      this.errorMsg = '';
      this.rfaiAdditionalInformationService
        .populateInfo(this.model.claimId)
        .subscribe(data => {
          if (data.errorCode) {
            this.errorMsg = data.message;
            this.errorStatus = true;
            this.prepopulatedInfo = new PDIInfoModel();
          } else {
            this.prepopulatedInfo = data;
            this.errorStatus = false;
            this.errorMsg = '';
          }
          this.selectAllLines.nativeElement.checked = false;
        });
    } else {
      this.errorMsg = 'Enter PDI or Claim ID.';
      this.errorStatus = true;
    }
  }

  selectAll(bool: boolean) {
    this.prepopulatedInfo.rfaiLineItemResponse.map(
      line => (line.selected = bool)
    );
  }

  saveLine() {
    this.lines.forEach(line => {
      if (line.serviceLineId) {
        line.selected = true;
        line.serviceLineId = +line.serviceLineId;
        line.chargeAmount = +line.chargeAmount;
        this.enableAddLine = true;
      } else {
        this.enableAddLine = false;
      }
    });
  }

  enableAddLineButton() {
    this.enableAddLine = true;
  }
}
