import {
  Component,
  Input,
  AfterContentChecked,
  Output,
  EventEmitter,
  OnInit
} from '@angular/core';
import { Router } from '@angular/router';
import { Subscription } from 'rxjs/Subscription';
import { Observable } from 'rxjs/Observable';
import { RestService } from '../../shared/service/rest.service';

import { TableService } from './table.service';
import { SortingFunctions } from './sortingFunctions.model';
import { ColumnMetadata } from './cols.model';
import { TableOptions } from './table-options.model';
import { ReconcileService } from './../../claims/claim-detail/reconcile/reconcile.service';

@Component({
  selector: 'table-component',
  templateUrl: './table.component.html',
  styleUrls: ['./table.component.scss'],
  providers: [TableService, RestService, ReconcileService]
})
export class TableComponent implements AfterContentChecked, OnInit {
  //  each col should contain 'id' (name of corresponding item property), (display) 'name', 'selectBox' (bool), 'button' (bool)
  @Input() cols: ColumnMetadata[] = [];
  @Output() onButtonClick: EventEmitter<{}> = new EventEmitter<{}>();
  //  items, aka records/rows, are the datas in the table. Can be injected/controlled by the parent, or indirectly filled in by table rest functionality
  //  Optionally, items can have 'Link' appended to regular property names to have corresponding links on the fields (i.e. claim.claimIdLink='/institutionalClaims')
  //  Button items can also have 'Event' appended to the property name for an onClick() function (i.e. claim.claimEditEvent=function(){window.alert('It Works!')})
  @Input() items: any[] = [];
  @Output() notify = new EventEmitter<any[]>();
  //  Global table options
  @Input() tableOptions = new TableOptions();

  @Input() numPerPageOptions: any[] = [10, 25, 50, 100];

  // These inputs are for the REST functionality of the table, which takes the REST responsibilities off of the parent component.
  @Input() url: String;
  @Input() method = 'GET';
  @Input() restParams = {};
  @Input() loading = true;

  // Internal variables for keeping track of table stats
  numPerPage: number = this.numPerPageOptions[0];
  pageNum = 0;
  sort = '';
  asc = false;
  numPages = 0;
  numResults = 0;

  constructor(
    private service: TableService,
    private router: Router,
    private reconcileService: ReconcileService
  ) { }
  ngOnInit() { }

  //  If using injected data, updated the numPages once after data is loaded
  ngAfterContentChecked() {
    if (!this.url && !!this.items) {
      this.numPages = Math.ceil(this.items.length / this.numPerPage);
    }
  }

  //  Sorts items by the property - reverses order if repeated property
  sortCol(id) {
    if (this.sort !== id) {
      this.sort = id;
      this.asc = false;
    } else {
      this.asc = !this.asc;
    }
    if (!!this.url) {
      this.update();
    } else {
      this.items = SortingFunctions.sortBy(id, this.asc, this.items);
    }
  }

  //  Used for the header checkmarks
  onClickColCheckMark(event) {
    this.items.forEach(item => (item.selected = event.target.checked));
  }

  //  updates with automatic subscription (if available)
  update() {
    const sub = this.restUpdate();
    if (!!sub) {
      sub.subscribe();
    }
  }
  //  updates rest, returning Observable. Can be linked (ex. update() table, .then() manipulate the new data or other functionality)
  //  TODO: add error handling, preferably where the 'No data available' is replaced by an error message
  restUpdate(): Observable<any[]> {
    if (!!this.url) {
      this.loading = true;
      return this.service
        .update(
        this.url,
        this.method,
        // This line merges two sets of parameters - the standard slice parameters taken by the standardized rest call, and the custom parameters from the parent
        Object.assign(
          {
            pageSize: this.numPerPage,
            pageNumber: this.pageNum + 1,
            sortColumn: this.sort,
            descending: this.asc
          },
          this.restParams
        )
        )
        .map(data => this.setData(data));
    } else {
      return;
    }
  }

  // updates data when rest update is made
  // TODO: merge with restUpdate()
  setData(data) {
    // The standardized rest response contains this property
    this.items = !!data['response'] ? data['response'] : [];
    // TODO: get rid of notify
    this.notify.emit(this.items);
    if (data['totalNumberOfResults'] !== undefined) {
      this.numResults = data['totalNumberOfResults'];
      this.numPages = Math.ceil(data['totalNumberOfResults'] / this.numPerPage);
      //  Note that the server doesn't follow array index numbering
      if (this.pageNum !== data['pageNumber'] - 1) {
        this.pageNum = data['pageNumber'] - 1;
      }
    }
    // Run afterLoad(), which is a function injected by the parent. This is used to manipulate the data retrieved before it is shown to the user.
    // Usually this is used to populate button names
    if (!!this.tableOptions['afterLoad']) {
      this.tableOptions['afterLoad'](this.items);
    }
    this.loading = false;
    return this.items;
  }
  //  This is used to convey clicked buttons up to the parent.  Emits the item by default
  onClick(item, colId) {
    const event = { colId: colId, item: item };
    this.onButtonClick.emit(event);
  }
}
