import { async, ComponentFixture, TestBed } from '@angular/core/testing';
import { DebugElement } from '@angular/core';
import { By } from '@angular/platform-browser';

import { FormsModule } from '@angular/forms';
import { RouterTestingModule } from '@angular/router/testing';
import { HttpModule } from '@angular/http';
import { BrowserModule } from '@angular/platform-browser';

import { RestService } from '../../shared/service/rest.service';
import { SharedDataService } from '../../shared/service/shared-data.service';
import { TableService } from '../table/table.service';
import { ReconcileService } from '../../claims/claim-detail/reconcile/reconcile.service';

import { TableComponent } from '../../shared/table/table.component';
import { RerouteComponent } from './reroute.component';
import { RerouteService } from './reroute.service';
import { MenuComponent } from '../../shared/menu/menu.component';
import { PaginationComponent } from '../../shared/table/pagination/pagination.component';
import { NumPerPageComponent } from '../../shared/table/numPerPage/numPerPage.component';
import { SpinnerComponent } from '../../shared/spinner/spinner.component';
import { WindowRefService } from '../../window-ref.service';
import { NumResultsComponent } from '../../shared/table/numResults/numResults.component';

describe('RerouteComponent', () => {
  let component: RerouteComponent;
  let fixture: ComponentFixture<RerouteComponent>;

  beforeEach(
    async(() => {
      TestBed.configureTestingModule({
        declarations: [
          NumPerPageComponent,
          PaginationComponent,
          RerouteComponent,
          MenuComponent,
          TableComponent,
          SpinnerComponent,
          NumResultsComponent
        ],
        imports: [
          /* RouterTestingModule, */
          RouterTestingModule.withRoutes([
            {
              path: 'login',
              component: RerouteComponent /* faking this for "async" test cases to work */
            }
          ]),
          HttpModule,
          FormsModule,
          BrowserModule
        ],
        providers: [
          RestService,
          SharedDataService,
          TableService,
          ReconcileService,
          RerouteService,
          WindowRefService
        ]
      }).compileComponents();
    })
  );

  beforeEach(() => {
    fixture = TestBed.createComponent(RerouteComponent);
    component = fixture.componentInstance;

    component.karmaUnitTesingBeingDone = true;

    const userInfo = {
      permissions: {
        claimsMenu: {
          paymentError: 'true',
          unmatchedPayments: 'true',
          inprocess: 'true',
          aged: 'true',
          searchClaim: 'true',
          rejectClaims: 'true',
          awaitingProcessing: 'true'
        },
        reportsMenu: {
          claimsAwaitingProcessing: 'true',
          reroutedClaims: 'true',
          claimsInprocessSummary: 'true',
          vistAError: 'true',
          claimsCompleted: 'true',
          CPEReport: 'true',
          claimsCompletedSummary: 'true',
          claimsInprocess: 'true',
          outOfSystemPaymentsSummary: 'true',
          BPPR16: 'true',
          outOfSystemPayments: 'true',
          claimsAwaitingProcessingSummary: 'true',
          feeReport: 'true'
        }
      }
    };

    sessionStorage.setItem('userInfo', JSON.stringify(userInfo));

    fixture.detectChanges();
  });

  function setClaimsSummaryDataWithFourClaims() {
    component.claimsThatCanBeRerouted = [
      {
        claim_id: 1009,
        veteran_name: 'CASSANDRA ALVARADO',
        provider_name: 'GWENDOLYN'
      },
      {
        claim_id: 530,
        veteran_name: 'GETTYS CHARLES',
        provider_name: 'KIMBERELY'
      },
      { claim_id: 400, veteran_name: 'HILTON SMITH', provider_name: 'VERNA' },
      {
        claim_id: 620,
        veteran_name: 'SINGAPORE HOSPITAL',
        provider_name: 'SINGAPORE'
      }
    ];

    fixture.detectChanges();
  }

  function setClaimsHistoryDataForSomeClaims() {
    component.historyClaimIds = [1009, 530, 400, 620];
    component.claimsRerouteHistory = {
      '1009': [
        {
          date: '10/06/2017',
          reason: 'Transfer Code 1',
          explanation: 'Hi Tester!',
          transferredTo: 'VISN:8/FAC:123/USER:',
          transferredFrom: 'VISN:1/FAC:1/USER:',
          transferredBy: 'VHAISMBAIAHBBB'
        },
        {
          date: '10/07/2017',
          reason: 'Transfer Code 2',
          explanation: 'The cat is in the hat!',
          transferredTo: 'VISN:5/FAC:123/USER:',
          transferredFrom: 'VISN:12/FAC:1/USER:',
          transferredBy: 'VHAISMBAIAHCCC'
        }
      ],
      '400': [
        {
          date: '09/06/2017',
          reason: 'Transfer Code 3',
          explanation: 'Where am I?',
          transferredTo: 'VISN:8/FAC:333/USER:',
          transferredFrom: 'VISN:1/FAC:7/USER:',
          transferredBy: 'VHAISMBAIAHDDD'
        },
        {
          date: '11/10/2017',
          reason: 'Transfer Code 4',
          explanation: '',
          transferredTo: 'VISN:333/FAC:123/USER:',
          transferredFrom: 'VISN:1255/FAC:1/USER:',
          transferredBy: 'VHAISMBAIAHEEE'
        }
      ]
    };

    fixture.detectChanges();
  }

  function setValidVisnValues() {
    component.selectedVisns = [10, 20, 30, 40, 50, 60, 70, 80, 90, 100];
    fixture.detectChanges();
  }

  function setValidFacilityValues() {
    component.selectedFacilities = ['21', '563', '564', '580', '586', '598'];
    fixture.detectChanges();
  }

  it('should be created', () => {
    expect(component).toBeTruthy();
  });

  it('verify claims summary data table should exist if there are claims to be routed', () => {
    setClaimsSummaryDataWithFourClaims();

    let debugElement: DebugElement = fixture.debugElement.query(
      By.css('#claimsSummaryTable')
    );
    let htmlElement: HTMLElement = debugElement.nativeElement;
    expect(htmlElement).toBeTruthy();
  });

  it('verify claims summary data table should not exist if there are no claims to be routed', () => {
    component.claimsThatCanBeRerouted = [];
    fixture.detectChanges();

    let debugElement: DebugElement = fixture.debugElement.query(
      By.css('#claimsSummaryTable')
    );
    expect(debugElement).toBeNull();
  });

  it('verify claims summary data table exists with "claimId" column header and correct column header string if there are claims to be routed', () => {
    setClaimsSummaryDataWithFourClaims();

    let tableComponentDebugElement: DebugElement = fixture.debugElement.query(
      By.css('#claimsSummaryTable')
    );
    let tableHeaderDebugElement: DebugElement = tableComponentDebugElement.query(
      By.css('thead')
    );
    let claimIdTableHeaderDebugElement: DebugElement = tableHeaderDebugElement.query(
      By.css('#claimId')
    );

    let claimIdTableHeaderHtmlElement: HTMLElement =
      claimIdTableHeaderDebugElement.nativeElement;
    expect(claimIdTableHeaderHtmlElement).toBeTruthy();

    expect(claimIdTableHeaderHtmlElement.innerText).toContain('Claim ID');
  });

  it('verify claims summary data table exists with "veteranName" column header and correct column header string if there are claims to be routed', () => {
    setClaimsSummaryDataWithFourClaims();

    let tableComponentDebugElement: DebugElement = fixture.debugElement.query(
      By.css('#claimsSummaryTable')
    );
    let tableHeaderDebugElement: DebugElement = tableComponentDebugElement.query(
      By.css('thead')
    );
    let veteranNameTableHeaderDebugElement: DebugElement = tableHeaderDebugElement.query(
      By.css('#veteranName')
    );

    let veteranNameTableHeaderHtmlElement: HTMLElement =
      veteranNameTableHeaderDebugElement.nativeElement;
    expect(veteranNameTableHeaderHtmlElement).toBeTruthy();

    expect(veteranNameTableHeaderHtmlElement.innerText).toContain(
      'Veteran Name'
    );
  });

  it('verify claims summary data table exists with "providerName" column header and correct column header string if there are claims to be routed', () => {
    setClaimsSummaryDataWithFourClaims();

    let tableComponentDebugElement: DebugElement = fixture.debugElement.query(
      By.css('#claimsSummaryTable')
    );
    let tableHeaderDebugElement: DebugElement = tableComponentDebugElement.query(
      By.css('thead')
    );
    let providerNameTableHeaderDebugElement: DebugElement = tableHeaderDebugElement.query(
      By.css('#providerName')
    );

    let providerNameTableHeaderHtmlElement: HTMLElement =
      providerNameTableHeaderDebugElement.nativeElement;
    expect(providerNameTableHeaderHtmlElement).toBeTruthy();

    expect(providerNameTableHeaderHtmlElement.innerText).toContain(
      'Provider Name'
    );
  });

  it('verify claims summary data table exists with correct number of rows if there are claims to be routed', () => {
    setClaimsSummaryDataWithFourClaims();

    let tableComponentDebugElement: DebugElement = fixture.debugElement.query(
      By.css('#claimsSummaryTable')
    );
    let tbodyDebugElement: DebugElement = tableComponentDebugElement.query(
      By.css('tbody')
    );

    let trDebugElements: DebugElement[] = tbodyDebugElement.queryAll(
      By.css('tr')
    );
    expect(trDebugElements.length).toEqual(
      component.claimsThatCanBeRerouted.length
    );
  });

  it('verify claims summary data table exists with correct data in each row if there are claims to be routed', () => {
    setClaimsSummaryDataWithFourClaims();

    let tableComponentDebugElement: DebugElement = fixture.debugElement.query(
      By.css('#claimsSummaryTable')
    );

    for (let i = 0; i < 3; i++) {
      let claimIdRowDebugElement: DebugElement = tableComponentDebugElement.query(
        By.css('#claimId' + i)
      );
      let claimIdRowHtmlElement: HTMLElement =
        claimIdRowDebugElement.nativeElement;
      expect(claimIdRowHtmlElement).toBeTruthy();
      expect(claimIdRowHtmlElement.innerText.trim()).toEqual(
        String(component.claimsThatCanBeRerouted[i].claim_id)
      );

      let veteranNameRowDebugElement: DebugElement = tableComponentDebugElement.query(
        By.css('#veteranName' + i)
      );
      let veteranNameRowHtmlElement: HTMLElement =
        veteranNameRowDebugElement.nativeElement;
      expect(veteranNameRowHtmlElement).toBeTruthy();
      expect(veteranNameRowHtmlElement.innerText.trim()).toEqual(
        String(component.claimsThatCanBeRerouted[i].veteran_name)
      );

      let providerNameRowDebugElement: DebugElement = tableComponentDebugElement.query(
        By.css('#providerName' + i)
      );
      let providerNameRowHtmlElement: HTMLElement =
        providerNameRowDebugElement.nativeElement;
      expect(providerNameRowHtmlElement).toBeTruthy();
      expect(providerNameRowHtmlElement.innerText.trim()).toEqual(
        String(component.claimsThatCanBeRerouted[i].provider_name)
      );
    }
  });

  it('verify visn select box should exist if there are claims to be routed', () => {
    setClaimsSummaryDataWithFourClaims();

    let debugElement: DebugElement = fixture.debugElement.query(
      By.css('#visnSelect')
    );
    let htmlElement: HTMLElement = debugElement.nativeElement;
    expect(htmlElement).toBeTruthy();
  });

  it('verify visn select box initially has the "Select VISN" value if there are claims to be routed', () => {
    setClaimsSummaryDataWithFourClaims();
    setValidVisnValues();

    let visnSelectBoxDebugElement: DebugElement = fixture.debugElement.query(
      By.css('#visnSelect')
    );
    let visnSelectBoxHtmlElement: HTMLElement =
      visnSelectBoxDebugElement.nativeElement;
    expect(visnSelectBoxHtmlElement).toBeTruthy();

    expect(visnSelectBoxHtmlElement.getAttribute('ng-reflect-model')).toEqual(
      ''
    );

    let selectVisnOptionDebugElement: DebugElement = visnSelectBoxDebugElement.query(
      By.css("[value='']")
    );
    let selectVisnOptionHtmlElement: HTMLElement =
      selectVisnOptionDebugElement.nativeElement;
    expect(selectVisnOptionHtmlElement).toBeTruthy();

    expect(selectVisnOptionHtmlElement.innerText).toEqual('Select VISN');
  });

  it('verify visn select box has all the option values it is suppose to have if there are claims to be routed', () => {
    setClaimsSummaryDataWithFourClaims();
    setValidVisnValues();

    let visnSelectBoxDebugElement: DebugElement = fixture.debugElement.query(
      By.css('#visnSelect')
    );
    let visnSelectBoxHtmlElement: HTMLElement =
      visnSelectBoxDebugElement.nativeElement;
    expect(visnSelectBoxHtmlElement).toBeTruthy();

    expect(visnSelectBoxHtmlElement.getAttribute('ng-reflect-model')).toEqual(
      ''
    );

    let selectVisnOptionDebugElement: DebugElement = visnSelectBoxDebugElement.query(
      By.css("[value='']")
    );
    let selectVisnOptionHtmlElement: HTMLElement =
      selectVisnOptionDebugElement.nativeElement;
    expect(selectVisnOptionHtmlElement).toBeTruthy();

    expect(selectVisnOptionHtmlElement.innerText).toEqual('Select VISN');

    component.selectedVisns.forEach(aVisnOptionValue => {
      let visnOptionDebugElement: DebugElement = visnSelectBoxDebugElement.query(
        By.css("[value='" + aVisnOptionValue + "']")
      );
      let visnOptionHtmlElement: HTMLElement =
        visnOptionDebugElement.nativeElement;
      expect(visnOptionHtmlElement).toBeTruthy();

      expect(visnOptionHtmlElement.innerText).toEqual(
        'VISN - ' + aVisnOptionValue
      );
    });
  });

  it(
    'verify that the "onSelectVisn" method gets called when there is a change in value of the visn select box if there are claims to be routed',
    async(() => {
      setClaimsSummaryDataWithFourClaims();
      setValidVisnValues();

      let visnSelectBoxDebugElement: DebugElement = fixture.debugElement.query(
        By.css('#visnSelect')
      );
      let visnSelectBoxHtmlElement: HTMLSelectElement =
        visnSelectBoxDebugElement.nativeElement;

      spyOn(component, 'onSelectVisn');

      // visnSelectBoxHtmlElement.value = "6";
      visnSelectBoxHtmlElement.selectedIndex = 6;
      visnSelectBoxHtmlElement.dispatchEvent(new Event('change'));

      fixture.whenStable().then(() => {
        expect(component.onSelectVisn).toHaveBeenCalled();
      });
    })
  );

  it('verify visn select box value can be changed and have new value show up if there are claims to be routed', () => {
    setClaimsSummaryDataWithFourClaims();
    setValidVisnValues();

    let visnSelectBoxDebugElement: DebugElement = fixture.debugElement.query(
      By.css('#visnSelect')
    );
    let visnSelectBoxHtmlElement: HTMLSelectElement =
      visnSelectBoxDebugElement.nativeElement;
    expect(visnSelectBoxHtmlElement).toBeTruthy();

    expect(visnSelectBoxHtmlElement.getAttribute('ng-reflect-model')).toEqual(
      ''
    );

    // The below for some strange reason did not cause the "component.selectedVisn" nor "mg-reflect-model" to change value. I do not
    //   know the reason for this.
    // visnSelectBoxHtmlElement.value = "60";
    // visnSelectBoxHtmlElement.selectedIndex = 6;
    // visnSelectBoxHtmlElement.dispatchEvent(new Event('change'));

    // The below was the only method that caused the "ng-reflect-model" to change value.
    component.selectedVisn = '60';

    fixture.detectChanges();

    console.log(
      '******* ng-reflect-model = ' +
        visnSelectBoxHtmlElement.getAttribute('ng-reflect-model')
    );
    console.log(
      '******* visnSelectBoxHtmlElement.value = ' +
        visnSelectBoxHtmlElement.value
    );
    console.log(
      '******* visnSelectBoxHtmlElement.selectedIndex = ' +
        visnSelectBoxHtmlElement.selectedIndex
    );
    console.log('******* component.selectedVisn = ' + component.selectedVisn);
    expect(visnSelectBoxHtmlElement.getAttribute('ng-reflect-model')).toEqual(
      '60'
    );
  });

  it('verify reason select box should exist if there are claims to be routed', () => {
    setClaimsSummaryDataWithFourClaims();

    let debugElement: DebugElement = fixture.debugElement.query(
      By.css('#reason')
    );
    let htmlElement: HTMLElement = debugElement.nativeElement;
    expect(htmlElement).toBeTruthy();
  });

  it('verify reason select box initially has the "Select Reason" value if there are claims to be routed', () => {
    setClaimsSummaryDataWithFourClaims();
    setValidVisnValues();

    let reasonSelectBoxDebugElement: DebugElement = fixture.debugElement.query(
      By.css('#reason')
    );
    let reasonSelectBoxHtmlElement: HTMLElement =
      reasonSelectBoxDebugElement.nativeElement;
    expect(reasonSelectBoxHtmlElement).toBeTruthy();

    expect(reasonSelectBoxHtmlElement.getAttribute('ng-reflect-model')).toEqual(
      ''
    );

    let selectReasonOptionDebugElement: DebugElement = reasonSelectBoxDebugElement.query(
      By.css("[value='']")
    );
    let selectReasonOptionHtmlElement: HTMLElement =
      selectReasonOptionDebugElement.nativeElement;
    expect(selectReasonOptionHtmlElement).toBeTruthy();

    expect(selectReasonOptionHtmlElement.innerText).toEqual('Select Reason');
  });

  it('verify reason select box value can be changed and have new value show up if there are claims to be routed', () => {
    setClaimsSummaryDataWithFourClaims();
    setValidVisnValues();

    let reasonSelectBoxDebugElement: DebugElement = fixture.debugElement.query(
      By.css('#reason')
    );
    let reasonSelectBoxHtmlElement: HTMLSelectElement =
      reasonSelectBoxDebugElement.nativeElement;
    expect(reasonSelectBoxHtmlElement).toBeTruthy();

    expect(reasonSelectBoxHtmlElement.getAttribute('ng-reflect-model')).toEqual(
      ''
    );

    // The below for some strange reason did not cause the "component.selectedVisn" nor "mg-reflect-model" to change value. I do not
    //   know the reason for this.
    // reasonSelectBoxHtmlElement.value = "1";
    // reasonSelectBoxHtmlElement.selectedIndex = 6;
    // reasonSelectBoxHtmlElement.dispatchEvent(new Event('change'));

    // The below was the only method that caused the "ng-reflect-model" to change value.
    component.selectedReason = '3';

    fixture.detectChanges();

    console.log(
      '******* ng-reflect-model = ' +
        reasonSelectBoxHtmlElement.getAttribute('ng-reflect-model')
    );
    console.log(
      '******* reasonSelectBoxHtmlElement.value = ' +
        reasonSelectBoxHtmlElement.value
    );
    console.log(
      '******* reasonSelectBoxHtmlElement.selectedIndex = ' +
        reasonSelectBoxHtmlElement.selectedIndex
    );
    console.log(
      '******* component.selectedReason = ' + component.selectedReason
    );
    expect(reasonSelectBoxHtmlElement.getAttribute('ng-reflect-model')).toEqual(
      '3'
    );
  });

  it('verify facility select box should exist if there are claims to be routed', () => {
    setClaimsSummaryDataWithFourClaims();

    let debugElement: DebugElement = fixture.debugElement.query(
      By.css('#facilitySelect')
    );
    let htmlElement: HTMLElement = debugElement.nativeElement;
    expect(htmlElement).toBeTruthy();
  });

  it('verify facility select box initially has the "Select Facility" value if there are claims to be routed', () => {
    setClaimsSummaryDataWithFourClaims();

    let facilitySelectBoxDebugElement: DebugElement = fixture.debugElement.query(
      By.css('#facilitySelect')
    );
    let facilitySelectBoxHtmlElement: HTMLElement =
      facilitySelectBoxDebugElement.nativeElement;
    expect(facilitySelectBoxHtmlElement).toBeTruthy();

    expect(
      facilitySelectBoxHtmlElement.getAttribute('ng-reflect-model')
    ).toEqual('');

    let selectFacilityOptionDebugElement: DebugElement = facilitySelectBoxDebugElement.query(
      By.css("[value='']")
    );
    let selectFacilityOptionHtmlElement: HTMLElement =
      selectFacilityOptionDebugElement.nativeElement;
    expect(selectFacilityOptionHtmlElement).toBeTruthy();

    expect(selectFacilityOptionHtmlElement.innerText).toEqual(
      'Select Facility'
    );
  });

  it('verify facility select box has all the option values it is suppose to have if there are claims to be routed', () => {
    setClaimsSummaryDataWithFourClaims();
    setValidFacilityValues();

    let facilitySelectBoxDebugElement: DebugElement = fixture.debugElement.query(
      By.css('#facilitySelect')
    );
    let facilitySelectBoxHtmlElement: HTMLElement =
      facilitySelectBoxDebugElement.nativeElement;
    expect(facilitySelectBoxHtmlElement).toBeTruthy();

    expect(
      facilitySelectBoxHtmlElement.getAttribute('ng-reflect-model')
    ).toEqual('');

    let selectFacilityOptionDebugElement: DebugElement = facilitySelectBoxDebugElement.query(
      By.css("[value='']")
    );
    let selectFacilityOptionHtmlElement: HTMLElement =
      selectFacilityOptionDebugElement.nativeElement;
    expect(selectFacilityOptionHtmlElement).toBeTruthy();

    expect(selectFacilityOptionHtmlElement.innerText).toEqual(
      'Select Facility'
    );

    component.selectedFacilities.forEach(aFacilityOptionValue => {
      let facilityOptionDebugElement: DebugElement = facilitySelectBoxDebugElement.query(
        By.css("[value='" + aFacilityOptionValue + "']")
      );
      let facilityOptionHtmlElement: HTMLElement =
        facilityOptionDebugElement.nativeElement;
      expect(facilityOptionHtmlElement).toBeTruthy();

      expect(facilityOptionHtmlElement.innerText).toEqual(aFacilityOptionValue);
    });
  });

  it('verify facility select box value can be changed and have new value show up if there are claims to be routed', () => {
    setClaimsSummaryDataWithFourClaims();
    setValidFacilityValues();

    let facilitySelectBoxDebugElement: DebugElement = fixture.debugElement.query(
      By.css('#facilitySelect')
    );
    let facilitySelectBoxHtmlElement: HTMLSelectElement =
      facilitySelectBoxDebugElement.nativeElement;
    expect(facilitySelectBoxHtmlElement).toBeTruthy();

    expect(
      facilitySelectBoxHtmlElement.getAttribute('ng-reflect-model')
    ).toEqual('');

    // The below for some strange reason did not cause the "component.selectedVisn" nor "mg-reflect-model" to change value. I do not
    //   know the reason for this.
    // facilitySelectBoxHtmlElement.value = "60";
    // facilitySelectBoxHtmlElement.selectedIndex = 6;
    // facilitySelectBoxHtmlElement.dispatchEvent(new Event('change'));

    // The below was the only method that caused the "ng-reflect-model" to change value.
    component.selectedFacility = '580';

    fixture.detectChanges();

    console.log(
      '******* ng-reflect-model = ' +
        facilitySelectBoxHtmlElement.getAttribute('ng-reflect-model')
    );
    console.log(
      '******* facilitySelectBoxHtmlElement.value = ' +
        facilitySelectBoxHtmlElement.value
    );
    console.log(
      '******* facilitySelectBoxHtmlElement.selectedIndex = ' +
        facilitySelectBoxHtmlElement.selectedIndex
    );
    console.log(
      '******* component.selectedVisn = ' + component.selectedFacility
    );
    expect(
      facilitySelectBoxHtmlElement.getAttribute('ng-reflect-model')
    ).toEqual('580');
  });

  it('verify reroute button should not exist if there are no claims to be routed', () => {
    let debugElement: DebugElement = fixture.debugElement.query(
      By.css('#submit')
    );
    expect(debugElement).toBeNull();
  });

  it('verify reroute button with correct label string should exist if there are claims to be routed', () => {
    setClaimsSummaryDataWithFourClaims();

    let debugElement: DebugElement = fixture.debugElement.query(
      By.css('#submit')
    );
    let htmlElement: HTMLElement = debugElement.nativeElement;
    expect(htmlElement).toBeTruthy();

    expect(htmlElement.innerText).toEqual('Reroute');
  });

  // Does not work!! This is because for Template Driven Forms, the Form Validation logic cannot be
  // Unit Tested as per the https://blog.angular-university.io/introduction-to-angular-2-forms-template-driven-vs-model-driven
  // web page. The only way to test this logic is to run an end to end test with a browser, for example using a headless browser
  // like PhantomJs.
  /*
	it('verify reroute button should exist and initially be in the disabled state if there are claims to be routed', () => {
		setClaimsSummaryDataWithThreeClaims();

		let debugElement: DebugElement = fixture.debugElement.query(By.css('#submit'));
		let htmlElement: HTMLElement = debugElement.nativeElement;
		expect(htmlElement).toBeTruthy();

		expect(htmlElement.getAttribute("disabled")).not.toBeNull();
		expect(htmlElement.getAttribute("class")).toContain(" disabled");
	});
	*/

  it(
    'should call "onSubmit" method when reroute button is clicked',
    async(() => {
      setClaimsSummaryDataWithFourClaims();

      let debugElement: DebugElement = fixture.debugElement.query(
        By.css('#submit')
      );
      let htmlElement: HTMLElement = debugElement.nativeElement;

      spyOn(component, 'onSubmit');

      htmlElement.click();

      fixture.whenStable().then(() => {
        expect(component.onSubmit).toHaveBeenCalled();
      });
    })
  );

  it('verify comments text area should not exist if there are no claims to be routed', () => {
    let debugElement: DebugElement = fixture.debugElement.query(
      By.css('#comments')
    );
    expect(debugElement).toBeNull();
  });

  it('verify comments text area should exist if there are claims to be routed', () => {
    setClaimsSummaryDataWithFourClaims();

    let debugElement: DebugElement = fixture.debugElement.query(
      By.css('#comments')
    );
    let htmlElement: HTMLElement = debugElement.nativeElement;
    expect(htmlElement).toBeTruthy();
  });

  it('verify comments text area initially has an empty string value if there are claims to be routed', () => {
    setClaimsSummaryDataWithFourClaims();

    let debugElement: DebugElement = fixture.debugElement.query(
      By.css('#comments')
    );
    let htmlElement: HTMLElement = debugElement.nativeElement;
    expect(htmlElement).toBeTruthy();

    expect(htmlElement.getAttribute('ng-reflect-model')).toEqual('');
    expect(htmlElement.innerText).toEqual('');
  });

  // Below did not work. Had to use subsequent Unit Test using technique described in the
  // https://stackoverflow.com/questions/41063005/angular2-unit-test-input-value web page.
  /*
	it('verify comments text area value can be changed and have new value if there are claims to be routed', () => {
		setClaimsSummaryDataWithThreeClaims();

		let debugElement: DebugElement = fixture.debugElement.query(By.css('#comments'));
		let htmlElement: HTMLInputElement = debugElement.nativeElement;
		expect(htmlElement).toBeTruthy();

		let newCommentsText = "The cat is in the hat!"

		// Below does not work!
		component.selectedComments = newCommentsText;

		htmlElement.value = newCommentsText;
		htmlElement.dispatchEvent(new Event('input'));

		fixture.detectChanges();

		console.log("htmlElement.value = " + htmlElement.value);

		expect(htmlElement.getAttribute("ng-reflect-model")).toEqual(newCommentsText);
		expect(htmlElement.innerText).toEqual(newCommentsText);
	});
	*/

  // Above Unit Test case did not work. Had to use below Unit Test using technique described in the
  // https://stackoverflow.com/questions/41063005/angular2-unit-test-input-value web page.
  it(
    'verify comments text area value can be changed and have new value if there are claims to be routed',
    async(() => {
      setClaimsSummaryDataWithFourClaims();

      fixture.whenStable().then(() => {
        let debugElement: DebugElement = fixture.debugElement.query(
          By.css('#comments')
        );
        let htmlElement: HTMLInputElement = debugElement.nativeElement;
        expect(htmlElement).toBeTruthy();

        expect(htmlElement.value).toBe('');

        let newCommentsText = 'The cat is in the hat!';
        htmlElement.value = newCommentsText;
        htmlElement.dispatchEvent(new Event('input'));

        // Make sure that when user enters text, it is reflected back in the NgModel variable.
        expect(component.selectedComments).toBe(newCommentsText);
      });
    })
  );

  it('verify appropriate claims history data exists for those claims that can be rerouted and that have claim history', () => {
    setClaimsSummaryDataWithFourClaims();
    setClaimsHistoryDataForSomeClaims();

    for (let i = 0; i < component.historyClaimIds.length; i++) {
      let claimId = String(component.historyClaimIds[i]);

      let claimRerouteHistoryTitleDebugElement: DebugElement = fixture.debugElement.query(
        By.css('#claimRerouteHistoryTitle' + claimId)
      );
      expect(claimRerouteHistoryTitleDebugElement).toBeTruthy();
      let claimRerouteHistoryTitleHtmlElement: HTMLElement =
        claimRerouteHistoryTitleDebugElement.nativeElement;
      expect(claimRerouteHistoryTitleHtmlElement).toBeTruthy();
      expect(claimRerouteHistoryTitleHtmlElement.innerText).toEqual(
        'Claim ' + claimId
      );

      if (component.claimsRerouteHistory[claimId]) {
        let claimRerouteHistory = component.claimsRerouteHistory[claimId];

        let claimRerouteHistoryTableDebugElement: DebugElement = fixture.debugElement.query(
          By.css('#claimRerouteHistoryTable' + claimId)
        );
        expect(claimRerouteHistoryTableDebugElement).toBeTruthy();

        let claimRerouteHistoryTableRowDataDebugElements: DebugElement[] = claimRerouteHistoryTableDebugElement.queryAll(
          By.css('#claimRerouteHistoryTableRowData')
        );
        for (
          let j = 0;
          j < claimRerouteHistoryTableRowDataDebugElements.length;
          j++
        ) {
          let claimRerouteHistoryTableRowDataDebugElement: DebugElement =
            claimRerouteHistoryTableRowDataDebugElements[j];

          let dateClaimRerouteHistoryTableDataDebugElement: DebugElement = claimRerouteHistoryTableRowDataDebugElement.query(
            By.css('#dateClaimRerouteHistoryTableData')
          );
          expect(dateClaimRerouteHistoryTableDataDebugElement).toBeTruthy();
          let dateClaimRerouteHistoryTableDataHtmlElement: HTMLElement =
            dateClaimRerouteHistoryTableDataDebugElement.nativeElement;
          expect(dateClaimRerouteHistoryTableDataHtmlElement).toBeTruthy();
          expect(dateClaimRerouteHistoryTableDataHtmlElement.innerText).toEqual(
            claimRerouteHistory[j].date
          );

          let reasonClaimRerouteHistoryTableDataDebugElement: DebugElement = claimRerouteHistoryTableRowDataDebugElement.query(
            By.css('#reasonClaimRerouteHistoryTableData')
          );
          expect(reasonClaimRerouteHistoryTableDataDebugElement).toBeTruthy();
          let reasonClaimRerouteHistoryTableDataHtmlElement: HTMLElement =
            reasonClaimRerouteHistoryTableDataDebugElement.nativeElement;
          expect(reasonClaimRerouteHistoryTableDataHtmlElement).toBeTruthy();
          expect(
            reasonClaimRerouteHistoryTableDataHtmlElement.innerText
          ).toEqual(claimRerouteHistory[j].reason);

          let explanationClaimRerouteHistoryTableDataDebugElement: DebugElement = claimRerouteHistoryTableRowDataDebugElement.query(
            By.css('#explanationClaimRerouteHistoryTableData')
          );
          expect(
            explanationClaimRerouteHistoryTableDataDebugElement
          ).toBeTruthy();
          let explanationClaimRerouteHistoryTableDataHtmlElement: HTMLElement =
            explanationClaimRerouteHistoryTableDataDebugElement.nativeElement;
          expect(
            explanationClaimRerouteHistoryTableDataHtmlElement
          ).toBeTruthy();
          expect(
            explanationClaimRerouteHistoryTableDataHtmlElement.innerText
          ).toEqual(claimRerouteHistory[j].explanation);

          let transferredToClaimRerouteHistoryTableDataDebugElement: DebugElement = claimRerouteHistoryTableRowDataDebugElement.query(
            By.css('#transferredToClaimRerouteHistoryTableData')
          );
          expect(
            transferredToClaimRerouteHistoryTableDataDebugElement
          ).toBeTruthy();
          let transferredToClaimRerouteHistoryTableDataHtmlElement: HTMLElement =
            transferredToClaimRerouteHistoryTableDataDebugElement.nativeElement;
          expect(
            transferredToClaimRerouteHistoryTableDataHtmlElement
          ).toBeTruthy();
          expect(
            transferredToClaimRerouteHistoryTableDataHtmlElement.innerText
          ).toEqual(claimRerouteHistory[j].transferredTo);

          let transferredFromClaimRerouteHistoryTableDataDebugElement: DebugElement = claimRerouteHistoryTableRowDataDebugElement.query(
            By.css('#transferredFromClaimRerouteHistoryTableData')
          );
          expect(
            transferredFromClaimRerouteHistoryTableDataDebugElement
          ).toBeTruthy();
          let transferredFromClaimRerouteHistoryTableDataHtmlElement: HTMLElement =
            transferredFromClaimRerouteHistoryTableDataDebugElement.nativeElement;
          expect(
            transferredFromClaimRerouteHistoryTableDataHtmlElement
          ).toBeTruthy();
          expect(
            transferredFromClaimRerouteHistoryTableDataHtmlElement.innerText
          ).toEqual(claimRerouteHistory[j].transferredFrom);

          let transferredByClaimRerouteHistoryTableDataDebugElement: DebugElement = claimRerouteHistoryTableRowDataDebugElement.query(
            By.css('#transferredByClaimRerouteHistoryTableData')
          );
          expect(
            transferredByClaimRerouteHistoryTableDataDebugElement
          ).toBeTruthy();
          let transferredByClaimRerouteHistoryTableDataHtmlElement: HTMLElement =
            transferredByClaimRerouteHistoryTableDataDebugElement.nativeElement;
          expect(
            transferredByClaimRerouteHistoryTableDataHtmlElement
          ).toBeTruthy();
          expect(
            transferredByClaimRerouteHistoryTableDataHtmlElement.innerText
          ).toEqual(claimRerouteHistory[j].transferredBy);
        }
      }
    }
  });

  it('verify that "No reroute history" mesage is dipslayed for those claims that can be rerouted and that have no claim history', () => {
    setClaimsSummaryDataWithFourClaims();
    setClaimsHistoryDataForSomeClaims();

    for (let i = 0; i < component.historyClaimIds.length; i++) {
      let claimId = String(component.historyClaimIds[i]);

      let claimRerouteHistoryTitleDebugElement: DebugElement = fixture.debugElement.query(
        By.css('#claimRerouteHistoryTitle' + claimId)
      );
      expect(claimRerouteHistoryTitleDebugElement).toBeTruthy();
      let claimRerouteHistoryTitleHtmlElement: HTMLElement =
        claimRerouteHistoryTitleDebugElement.nativeElement;
      expect(claimRerouteHistoryTitleHtmlElement).toBeTruthy();
      expect(claimRerouteHistoryTitleHtmlElement.innerText).toEqual(
        'Claim ' + claimId
      );

      if (!component.claimsRerouteHistory[claimId]) {
        let noRerouteHistoryDebugElement: DebugElement = fixture.debugElement.query(
          By.css('#noRerouteHistory' + claimId)
        );
        expect(noRerouteHistoryDebugElement).toBeTruthy();

        let noRerouteHistoryHtmlElement: HTMLElement =
          noRerouteHistoryDebugElement.nativeElement;
        expect(noRerouteHistoryHtmlElement).toBeTruthy();

        expect(noRerouteHistoryHtmlElement.innerText).toEqual(
          'No reroute history'
        );
      }
    }
  });
});
