import {
  async,
  fakeAsync,
  getTestBed,
  inject,
  tick,
  ComponentFixture,
  TestBed
} from '@angular/core/testing';
import { MockBackend, MockConnection } from '@angular/http/testing';
import { RouterTestingModule } from '@angular/router/testing';
import { FormsModule } from '@angular/forms';
import {
  Headers,
  BaseRequestOptions,
  Http,
  HttpModule,
  Response,
  ResponseOptions,
  RequestMethod,
  XHRBackend
} from '@angular/http';
import { SharedDataService } from '../../shared/service/shared-data.service';
import { DisapproveService } from './service/disapprove.service';
import { DebugElement, SimpleChanges } from '@angular/core';
import { By } from '@angular/platform-browser';
import { DisapproveComponent } from './disapprove.component';
import { MenuComponent } from '../../shared/menu/menu.component';
import { WindowRefService } from './../../../app/window-ref.service';
describe('DisapproveComponent', () => {
  let component: DisapproveComponent;
  let fixture: ComponentFixture<DisapproveComponent>;

  let mockBackend: MockBackend;

  function findOptionHtmlElementHavingCodeValue(
    optionDebugElements: DebugElement[],
    codeValue: string
  ): HTMLElement {
    for (let i = 0; i < optionDebugElements.length; i++) {
      const optionHtmlElement: HTMLElement =
        optionDebugElements[i].nativeElement;
      const optionValue = optionHtmlElement.getAttribute('value');
      if (codeValue === optionValue) {
        return optionHtmlElement;
      }
    }
    return null;
  }

  function setClaimsDataToDisplay() {
    component.claimsToDisplay = [
      {
        claimId: 3615179,
        veteranName: 'Kirk, Brittney',
        serviceProvider: 'Memorial Hospital',
        billingProvider: 'Memorial Hospital'
      },
      {
        claimId: 4128800,
        veteranName: 'Wilkinson, Pope',
        serviceProvider: 'General Hospital',
        billingProvider: 'General Hospital'
      },
      {
        claimId: 9514989,
        veteranName: 'Vasquez, Cervantes',
        serviceProvider: 'Memorial Hospital',
        billingProvider: 'Memorial Hospital'
      }
    ];

    component.claimIds2.length = 3;
    fixture.detectChanges();
  }

  beforeEach(
    async(() => {
      TestBed.configureTestingModule({
        imports: [
          RouterTestingModule.withRoutes([
            {
              path: 'login',
              component: DisapproveComponent /* faking this for "async" test cases to work */
            }
          ]),
          FormsModule,
          HttpModule
        ],
        declarations: [DisapproveComponent, MenuComponent],
        providers: [
          WindowRefService,
          SharedDataService,
          DisapproveService,
          MockBackend,
          BaseRequestOptions,
          {
            provide: Http,
            deps: [MockBackend, BaseRequestOptions],
            useFactory: (
              backend: XHRBackend,
              defaultOptions: BaseRequestOptions
            ) => {
              return new Http(backend, defaultOptions);
            }
          }
        ]
      }).compileComponents();

      mockBackend = getTestBed().get(MockBackend);
    })
  );

  beforeEach(() => {
    fixture = TestBed.createComponent(DisapproveComponent);
    component = fixture.componentInstance;

    component.karmaUnitTesingBeingDone = true;

    component.validReasons = [
      {
        code: '65/159/177',
        description: '65/159/177 : Duplicate claim - Previously processed.'
      },
      {
        code: '78',
        description:
          '78 : EOB from other insurance required - CBOPC secondary payer.'
      },
      {
        code: '124',
        description:
          '124 : Claim not timely filed. (See applicable CBOPC Program Guide.)'
      },
      {
        code: '278',
        description:
          '278 : Multiple primary insurance coverage. Please resubmit EOBs from each payer.'
      },
      {
        code: '148',
        description: '148 : Claim denied - Chiropractic services not covered.'
      }
    ];

    fixture.detectChanges();
  });

  it('should be created', () => {
    expect(component).toBeTruthy();
  });

  it('should have "disapprove" button not exist when no claims data exists to display', () => {
    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('#disapproveButton')
    );
    expect(debugElement).toBeNull();

    // let htmlElement: HTMLElement = debugElement.nativeElement;
    // expect(htmlElement).toBeTruthy()
  });

  it('should have "disapprove" button exist and be visible when claims data exists to display', () => {
    setClaimsDataToDisplay();

    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('#disapproveButton')
    );
    const htmlElement: HTMLElement = debugElement.nativeElement;

    expect(htmlElement).toBeTruthy();
    expect(htmlElement.hidden).toEqual(false);
    expect(htmlElement.hidden).toBe(false);
  });

  it('should have "disapprove" button initially disabled when claims data exists to display', () => {
    setClaimsDataToDisplay();

    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('#disapproveButton')
    );
    const htmlElement: HTMLElement = debugElement.nativeElement;

    expect(htmlElement).toBeTruthy();
  });

  it('should have "cancel" button not exist if no claims data exists to display', () => {
    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('#cancelButton')
    );
    expect(debugElement).toBeNull();
  });

  it('should have "cancel" button exists, enabled and visible when claims data exists to display', () => {
    setClaimsDataToDisplay();

    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('#cancelButton')
    );
    const htmlElement: HTMLElement = debugElement.nativeElement;

    expect(htmlElement).toBeTruthy();
    expect(htmlElement.getAttribute('disabled')).toBeNull();
    expect(htmlElement.hidden).toEqual(false);
    expect(htmlElement.hidden).toBe(false);
  });

  it(
    'should call "onClickCancel" method when "cancel" button is clicked and claims data exists to display',
    async(() => {
      setClaimsDataToDisplay();

      const debugElement: DebugElement = fixture.debugElement.query(
        By.css('#cancelButton')
      );
      const htmlElement: HTMLElement = debugElement.nativeElement;

      spyOn(component, 'onClickCancel');

      htmlElement.click();

      fixture.whenStable().then(() => {
        expect(component.onClickCancel).toHaveBeenCalled();
      });
    })
  );

  it(
    'should route to "claimsHomePage" url when "cancel" button is clicked and claims data exists to display',
    async(() => {
      setClaimsDataToDisplay();

      const debugElement: DebugElement = fixture.debugElement.query(
        By.css('#cancelButton')
      );
      const htmlElement: HTMLElement = debugElement.nativeElement;

      const navigateSpy = spyOn((<any>component).router, 'navigate');

      htmlElement.click();

      fixture.whenStable().then(() => {
        expect(navigateSpy).toHaveBeenCalledWith(['claimsHomePage']);
      });
    })
  );

  it('should have "comments" input text area not exist when no claims data exists to display', () => {
    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('#comments')
    );
    expect(debugElement).toBeNull();
  });

  it('should have "comments" input text area exist, enabled, and visible when claims data exists to display', () => {
    setClaimsDataToDisplay();

    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('#comments')
    );
    const htmlElement: HTMLElement = debugElement.nativeElement;

    expect(htmlElement).toBeTruthy();
    expect(htmlElement.getAttribute('disabled')).toBeNull();
    expect(htmlElement.hidden).toEqual(false);
    expect(htmlElement.hidden).toBe(false);
  });

  it('should have "comments" input text area value be an empty string initially when claims data exists to display', () => {
    setClaimsDataToDisplay();

    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('#comments')
    );
    const htmlElement: HTMLElement = debugElement.nativeElement;

    expect(htmlElement).toBeTruthy();
    expect(htmlElement.getAttribute('ng-reflect-model')).toEqual('');
  });

  it('should have "comments" input text area value be same value that is in "disapproveClaimParam.comment" variable when claims data exists to display', () => {
    setClaimsDataToDisplay();

    const commentStringToSet = 'The cat is in the hat!';
    component.disapproveClaimParam.comment = commentStringToSet;

    fixture.detectChanges();

    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('#comments')
    );
    const htmlElement: HTMLElement = debugElement.nativeElement;

    expect(htmlElement).toBeTruthy();
    expect(htmlElement.getAttribute('ng-reflect-model')).toEqual(
      commentStringToSet
    );
  });

  it('should have "reason" select box not exist when no claims data exists to display', () => {
    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('#reason')
    );
    expect(debugElement).toBeNull();

    /*
		let htmlElement: HTMLElement = debugElement.nativeElement;
		expect(htmlElement).toBeTruthy();
		*/
  });

  it('should have "reason" select box exits, enabled, and visible when claims data exists to display', () => {
    setClaimsDataToDisplay();

    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('#reason')
    );
    const htmlElement: HTMLElement = debugElement.nativeElement;

    expect(htmlElement).toBeTruthy();
    expect(htmlElement.getAttribute('disabled')).toBeNull();
    expect(htmlElement.hidden).toEqual(false);
    expect(htmlElement.hidden).toBe(false);
  });

  it('should have "reason" select box value initially be the "Select Reason" empty string value when claims data exists to display', () => {
    setClaimsDataToDisplay();

    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('#reason')
    );
    const htmlElement: HTMLElement = debugElement.nativeElement;

    expect(htmlElement).toBeTruthy();
    expect(htmlElement.getAttribute('ng-reflect-model')).toEqual('');

    const optionDebugElements: DebugElement[] = debugElement.queryAll(
      By.css('option')
    );
    let optionEmptyStringValueFound = false;
    for (let i = 0; i < optionDebugElements.length; i++) {
      const optionHtmlElement: HTMLElement =
        optionDebugElements[i].nativeElement;
      const optionValue = optionHtmlElement.getAttribute('value');
      if (optionValue === '') {
        optionEmptyStringValueFound = true;
        const innerText = optionHtmlElement.innerText;
        if (innerText === 'Select Reason') {
          break;
        } else {
          fail(
            'Expected "Select Reason" text for empty string value, but instead got "' +
              innerText +
              "' text!"
          );
        }
      }
    }

    if (!optionEmptyStringValueFound) {
      fail('No "Select Reason" empty string option value found!');
    }
  });

  it('should have "reason" select box value and description match what is in the "disapproveClaimParam.reason" variable when claims data exists to display', () => {
    setClaimsDataToDisplay();

    const chosenCodeValue = component.validReasons[0].code;
    const chosenCodeDescription = component.validReasons[0].description;
    component.disapproveClaimParam.reason = chosenCodeValue;

    fixture.detectChanges();

    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('#reason')
    );
    const htmlElement: HTMLElement = debugElement.nativeElement;

    expect(htmlElement).toBeTruthy();
    expect(htmlElement.getAttribute('ng-reflect-model')).toEqual(
      chosenCodeValue
    );

    const optionDebugElements: DebugElement[] = debugElement.queryAll(
      By.css('option')
    );
    let optionStringValueFound = false;
    for (let i = 0; i < optionDebugElements.length; i++) {
      const optionHtmlElement: HTMLElement =
        optionDebugElements[i].nativeElement;
      const optionValue = optionHtmlElement.getAttribute('value');
      if (optionValue === chosenCodeValue) {
        optionStringValueFound = true;
        const innerText = optionHtmlElement.innerText;
        if (innerText === chosenCodeDescription) {
          break;
        } else {
          fail(
            'Expected "' +
              chosenCodeDescription +
              '" description text for "' +
              chosenCodeValue +
              '" string value, but instead got "' +
              innerText +
              "' description text!"
          );
        }
      }
    }

    if (!optionStringValueFound) {
      fail('No "' + chosenCodeValue + '" string option value found!');
    }
  });

  it('should initially have "reason" select box have "Select Reason" empty string value and "Select Reason" text when claims data exists to display', () => {
    setClaimsDataToDisplay();

    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('#reason')
    );
    const htmlElement: HTMLElement = debugElement.nativeElement;

    expect(htmlElement).toBeTruthy();

    const optionDebugElements: DebugElement[] = debugElement.queryAll(
      By.css('option')
    );
    let optionEmptyStringValueFound = false;
    for (let i = 0; i < optionDebugElements.length; i++) {
      const optionHtmlElement: HTMLElement =
        optionDebugElements[i].nativeElement;
      const optionValue = optionHtmlElement.getAttribute('value');
      if (optionValue === '') {
        optionEmptyStringValueFound = true;
        const innerText = optionHtmlElement.innerText;
        if (innerText === 'Select Reason') {
          break;
        } else {
          fail(
            'Expected "Select Reason" text for empty string value, but instead got "' +
              innerText +
              "' text!"
          );
        }
      }
    }

    if (!optionEmptyStringValueFound) {
      fail('No "Select Reason" empty string option value found!');
    }
  });

  it('should have "reason" select box have appropriate number of option elements when claims data exists to display', () => {
    setClaimsDataToDisplay();

    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('#reason')
    );
    const htmlElement: HTMLElement = debugElement.nativeElement;

    expect(htmlElement).toBeTruthy();

    const optionDebugElements: DebugElement[] = debugElement.queryAll(
      By.css('option')
    );
    const expectedNumberOfOptionElements = component.validReasons.length + 1;
    const actualNumberOfOptionElements = optionDebugElements.length;
    expect(expectedNumberOfOptionElements).toEqual(
      actualNumberOfOptionElements
    );
  });

  it('should have "reason" select box have all the appropriate option values and descriptions based on the "validReasons" data when claims data exists to display', () => {
    setClaimsDataToDisplay();

    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('#reason')
    );
    const htmlElement: HTMLElement = debugElement.nativeElement;

    expect(htmlElement).toBeTruthy();

    const optionDebugElements: DebugElement[] = debugElement.queryAll(
      By.css('option')
    );
    for (let i = 0; i < component.validReasons.length; i++) {
      const codeValue: string = component.validReasons[i].code;
      const optionHtmlElement: HTMLElement = findOptionHtmlElementHavingCodeValue(
        optionDebugElements,
        codeValue
      );
      if (optionHtmlElement === null) {
        fail(
          'Expected to find "' +
            codeValue +
            '" option value in "reason" select box, but it was not present!'
        );
      } else {
        const innerText = optionHtmlElement.innerText;
        const description = component.validReasons[i].description;
        if (!(innerText === description)) {
          fail(
            'Expected to have the "' +
              description +
              '" description for the "' +
              codeValue +
              '" value in the "reason" select box, but instead its description was: "' +
              innerText +
              '"!'
          );
        }
      }
    }
  });

  it('should have "table" element not exist when no claims data exists to display', () => {
    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('table')
    );
    expect(debugElement).toBeNull();

    /*
		let htmlElement: HTMLElement = debugElement.nativeElement;
		expect(htmlElement).toBeTruthy();
		*/
  });

  it('should have "table" element exist when claims data exists to display', () => {
    setClaimsDataToDisplay();

    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('table')
    );
    const htmlElement: HTMLElement = debugElement.nativeElement;

    expect(htmlElement).toBeTruthy();
  });

  it('should have "table" element that has the correct number of columns when claims data exists to display', () => {
    setClaimsDataToDisplay();

    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('table')
    );
    const htmlElement: HTMLElement = debugElement.nativeElement;

    expect(htmlElement).toBeTruthy();

    const tableHeaderDebugElements: DebugElement[] = debugElement.queryAll(
      By.css('th')
    );
    expect(component.cols.length).toEqual(tableHeaderDebugElements.length);
  });

  it('should have "table" element that has the correct number of rows when "claims" data exists to display', () => {
    setClaimsDataToDisplay();

    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('table')
    );
    const htmlElement: HTMLElement = debugElement.nativeElement;
    expect(htmlElement).toBeTruthy();

    const tableBodyDebugElement: DebugElement = debugElement.query(
      By.css('tbody')
    );
    expect(tableBodyDebugElement).toBeTruthy();

    const tableRowDebugElements: DebugElement[] = tableBodyDebugElement.queryAll(
      By.css('tr')
    );
    expect(tableRowDebugElements).toBeTruthy();
    expect(component.claimsToDisplay.length).toEqual(
      tableRowDebugElements.length
    );
  });

  it('should have "table" element that has the correct number of "td" data cell elements when "claims" data exists to display', () => {
    setClaimsDataToDisplay();

    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('table')
    );
    const htmlElement: HTMLElement = debugElement.nativeElement;
    expect(htmlElement).toBeTruthy();

    const tableBodyDebugElement: DebugElement = debugElement.query(
      By.css('tbody')
    );
    expect(tableBodyDebugElement).toBeTruthy();

    const tableDataCellDebugElements: DebugElement[] = tableBodyDebugElement.queryAll(
      By.css('td')
    );
    expect(tableDataCellDebugElements).toBeTruthy();
    expect(component.claimsToDisplay.length * component.cols.length).toEqual(
      tableDataCellDebugElements.length
    );
  });

  it('should have "table" element that has the correct data when "claims" data exists to display', () => {
    setClaimsDataToDisplay();

    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('table')
    );
    const htmlElement: HTMLElement = debugElement.nativeElement;
    expect(htmlElement).toBeTruthy();

    const tableBodyDebugElement: DebugElement = debugElement.query(
      By.css('tbody')
    );
    expect(tableBodyDebugElement).toBeTruthy();

    const tableRowDebugElements: DebugElement[] = tableBodyDebugElement.queryAll(
      By.css('tr')
    );
    expect(tableRowDebugElements).toBeTruthy();
    expect(component.claimsToDisplay.length).toEqual(
      tableRowDebugElements.length
    );

    for (let row = 0; row < tableRowDebugElements.length; row++) {
      const tableRowDebugElement: DebugElement = tableRowDebugElements[row];
      expect(tableRowDebugElement).toBeTruthy();

      const tableDataCellColumnDebugElements: DebugElement[] = tableRowDebugElement.queryAll(
        By.css('td')
      );
      expect(tableDataCellColumnDebugElements.length).toEqual(
        component.cols.length
      );

      for (
        let column = 0;
        column < tableDataCellColumnDebugElements.length;
        column++
      ) {
        const tableDataCellColumnDebugElement: DebugElement =
          tableDataCellColumnDebugElements[column];
        const tableDataCellColumnHtmlElement: HTMLElement =
          tableDataCellColumnDebugElement.nativeElement;
        // console.log("   tableDataCellColumnHtmlElement.innerText = " + tableDataCellColumnHtmlElement.innerText);
        const fieldName = component.cols[column].id;
        const fieldValue = component.claimsToDisplay[row][fieldName];
        // console.log("    field value = " + fieldValue);
        expect(tableDataCellColumnHtmlElement.innerText).toEqual(
          String(fieldValue)
        );
      }
    }
  });

  it('should have "disapprove" button enabled when "reason" and "comments" are entered and "claims" data exists to display', () => {
    setClaimsDataToDisplay();

    const commentStringToSet = 'The cat is in the hat!';
    component.disapproveClaimParam.reason = component.validReasons[0].code;
    component.disapproveClaimParam.comment = commentStringToSet;

    fixture.detectChanges();

    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('#disapproveButton')
    );
    const htmlElement: HTMLElement = debugElement.nativeElement;

    expect(htmlElement).toBeTruthy();
    expect(htmlElement.getAttribute('disabled')).toBeNull();
  });

  it(
    'should call "onClickDisapproveClaims" method when "disapprove" button is clicked after "reason" and "comments" are entered and "claims" data exists to display',
    async(() => {
      setClaimsDataToDisplay();

      const commentStringToSet = 'The cat is in the hat!';
      component.disapproveClaimParam.reason = component.validReasons[0].code;
      component.disapproveClaimParam.comment = commentStringToSet;

      fixture.detectChanges();

      const debugElement: DebugElement = fixture.debugElement.query(
        By.css('#disapproveButton')
      );
      const htmlElement: HTMLElement = debugElement.nativeElement;

      spyOn(component, 'onClickDisapproveClaims');

      htmlElement.click();

      fixture.whenStable().then(() => {
        expect(component.onClickDisapproveClaims).toHaveBeenCalled();
      });
    })
  );

  it('should have "disapprove" button disabled when "reason" value is entered, "claims" data exists to display, but no "comments" value entered', () => {
    setClaimsDataToDisplay();

    component.disapproveClaimParam.reason = component.validReasons[0].code;

    fixture.detectChanges();

    const debugElement: DebugElement = fixture.debugElement.query(
      By.css('#disapproveButton')
    );
    const htmlElement: HTMLElement = debugElement.nativeElement;

    expect(htmlElement).toBeTruthy();
    //	expect(htmlElement.getAttribute('disabled')).not.toBeNull();
  });

  // tslint:disable-next-line:max-line-length
  it(
    'clicking on "disapprove" button should only display successful confirmation message when (a) "reason" and "comments" are entered, (b) "claims" data exists to display, and (c) all claims in table were successfully disapproved',
    async(() => {
      setClaimsDataToDisplay();

      const commentStringToSet = 'The cat is in the hat!';
      component.disapproveClaimParam.reason = component.validReasons[0].code;
      component.disapproveClaimParam.comment = commentStringToSet;

      fixture.detectChanges();

      const disapproveClaimsStatuses = {
        disapproveClaims: [
          { status: 200, message: 123 },
          { status: 200, message: 124 },
          { status: 200, message: 125 },
          { status: 200, message: 126 }
        ]
      };

      mockBackend.connections.subscribe((connection: MockConnection) => {
        connection.mockRespond(
          new Response(
            new ResponseOptions({
              body: disapproveClaimsStatuses
            })
          )
        );
      });

      let debugElement: DebugElement = fixture.debugElement.query(
        By.css('#disapproveButton')
      );
      let htmlElement: HTMLElement = debugElement.nativeElement;

      htmlElement.click();

      fixture.detectChanges();

      expect(component.btnClicked).toBe(true);

      debugElement = fixture.debugElement.query(
        By.css('#successfulClaimsDisapprovalMessage')
      );
      htmlElement = debugElement.nativeElement;

      expect(htmlElement).toBeTruthy();

      const expectedMessage =
        'Success: Disapproval Completed. The following claim(s) have been disapproved: 123, 124, 125, 126';
      expect(htmlElement.innerText).toBe(expectedMessage);

      debugElement = fixture.debugElement.query(
        By.css('#failedClaimsDisapprovalMessage')
      );
      expect(debugElement).toBeNull();
    })
  );

  // tslint:disable-next-line:max-line-length
  it(
    'clicking on "disapprove" button should only display failure confirmation message when (a) "reason" and "comments" are entered, (b) "claims" data exists to display, and (c) all claims in table failed to be disapproved',
    async(() => {
      setClaimsDataToDisplay();

      const commentStringToSet = 'The cat is in the hat!';
      component.disapproveClaimParam.reason = component.validReasons[0].code;
      component.disapproveClaimParam.comment = commentStringToSet;

      fixture.detectChanges();

      const disapproveClaimsStatuses = {
        disapproveClaims: [
          { status: 300, message: 123 },
          { status: 201, message: 124 },
          { status: 205, message: 125 },
          { status: 500, message: 126 }
        ]
      };

      mockBackend.connections.subscribe((connection: MockConnection) => {
        connection.mockRespond(
          new Response(
            new ResponseOptions({
              body: disapproveClaimsStatuses
            })
          )
        );
      });

      let debugElement: DebugElement = fixture.debugElement.query(
        By.css('#disapproveButton')
      );
      let htmlElement: HTMLElement = debugElement.nativeElement;

      htmlElement.click();

      fixture.detectChanges();

      expect(component.btnClicked).toBe(true);

      debugElement = fixture.debugElement.query(
        By.css('#failedClaimsDisapprovalMessage')
      );
      htmlElement = debugElement.nativeElement;

      expect(htmlElement).toBeTruthy();

      expect(htmlElement.innerText).toContain(
        'Error while disapproving the following claim(s):'
      );
      expect(htmlElement.innerText).toContain('123, 124, 125, 126');

      debugElement = fixture.debugElement.query(
        By.css('#successfulClaimsDisapprovalMessage')
      );
      expect(debugElement).toBeNull();
    })
  );

  // tslint:disable-next-line:max-line-length
  it(
    'clicking on "disapprove" button should display both failue and failure confirmation messages when (a) "reason" and "comments" are entered, (b) "claims" data exists to display, (c) some claims in table failed to be disapproved, and (d) some claims were successfully disapproved',
    async(() => {
      setClaimsDataToDisplay();

      const commentStringToSet = 'The cat is in the hat!';
      component.disapproveClaimParam.reason = component.validReasons[0].code;
      component.disapproveClaimParam.comment = commentStringToSet;

      fixture.detectChanges();

      const disapproveClaimsStatuses = {
        disapproveClaims: [
          { status: 200, message: 123 },
          { status: 200, message: 124 },
          { status: 205, message: 125 },
          { status: 500, message: 126 }
        ]
      };

      mockBackend.connections.subscribe((connection: MockConnection) => {
        connection.mockRespond(
          new Response(
            new ResponseOptions({
              body: disapproveClaimsStatuses
            })
          )
        );
      });

      let debugElement: DebugElement = fixture.debugElement.query(
        By.css('#disapproveButton')
      );
      let htmlElement: HTMLElement = debugElement.nativeElement;

      htmlElement.click();

      fixture.detectChanges();

      expect(component.btnClicked).toBe(true);

      debugElement = fixture.debugElement.query(
        By.css('#failedClaimsDisapprovalMessage')
      );
      htmlElement = debugElement.nativeElement;
      expect(htmlElement).toBeTruthy();

      expect(htmlElement.innerText).toContain(
        'Error while disapproving the following claim(s):'
      );
      expect(htmlElement.innerText).toContain('125, 126');

      debugElement = fixture.debugElement.query(
        By.css('#successfulClaimsDisapprovalMessage')
      );
      htmlElement = debugElement.nativeElement;
      expect(htmlElement).toBeTruthy();

      const expectedMessage =
        'Success: Disapproval Completed. The following claim(s) have been disapproved: 123, 124';
      expect(htmlElement.innerText).toBe(expectedMessage);
    })
  );
});
