package gov.va.med.fee.service.impl;

import static org.junit.Assert.assertEquals;
import static org.mockito.Mockito.when;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.runners.MockitoJUnitRunner;

import gov.va.med.domain.fee.AppRole;
import gov.va.med.domain.fee.AppUser;
import gov.va.med.domain.fee.ContactInfo;
import gov.va.med.domain.fee.UserRoleUsages;
import gov.va.med.domain.fee.UserRoleUsagesId;
import gov.va.med.domain.fee.VaFacility;
import gov.va.med.domain.fee.VaFacilityType;
import gov.va.med.domain.fee.Visn;
import gov.va.med.domain.fee.ZipCode;
import gov.va.med.fee.dao.IAppUserRepository;
import gov.va.med.fee.dao.IContactInfoRepository;
import gov.va.med.fee.dao.IVaFacilityRepository;
import gov.va.med.fee.dao.IVaFacilityRepositoryTemplate;
import gov.va.med.fee.dao.IVaFacilityTypeRepository;
import gov.va.med.fee.dao.IVisnRepository;
import gov.va.med.fee.dao.IZipCodeRepository;
import gov.va.med.fee.exceptions.GenericException;
import gov.va.med.fee.model.request.ContactInfoRequest;
import gov.va.med.fee.model.request.VaFacilityRequest;
import gov.va.med.fee.model.request.VaFacilitySearchRequest;
import gov.va.med.fee.model.request.ZipCodeRequest;
import gov.va.med.fee.model.response.GenericResponse;
import gov.va.med.fee.model.response.VaFacilityResponse;
import gov.va.med.fee.model.response.VaFacilityTypeResponse;
import gov.va.med.fee.model.response.VaFacilityVisnResponse;

@RunWith(MockitoJUnitRunner.class)
public class VaFacilityServiceImplTest {
	
	private static final Logger logger = LogManager.getLogger(VaFacilityServiceImplTest.class);
	
	@InjectMocks
	VaFacilityServiceImpl vaFacilityServiceImpl;
	
	@Mock
	IVaFacilityRepositoryTemplate vaFacilityRepoTemplate;
	
	@Mock
	IZipCodeRepository zipCodeRepository;
	
	@Mock
	IVaFacilityRepository iVaFacilityRepository;
	
	@Mock
	IVisnRepository iVisnRepository;
	
	@Mock
	IVaFacilityTypeRepository iVaFacilityTypeRepository;
	
	@Mock
	IAppUserRepository appUserRepository;
	
	@Mock
	IContactInfoRepository iContactInfoRepository;
	
	List<AppUser> list = new ArrayList<>();
	
	@Test
	public void test_getAllVisns() {
		VaFacilityVisnResponse response = new VaFacilityVisnResponse();
		List<VaFacilityVisnResponse> res = new ArrayList<VaFacilityVisnResponse>();
		
		response.setVisnIdCd(1234);
		response.setDescription("This is a test visn");
		res.add(response);
		
		Mockito.when(vaFacilityRepoTemplate.getAllVisnsForFormLoad()).thenReturn(res);
		
		List<VaFacilityVisnResponse> result = new ArrayList<VaFacilityVisnResponse>();
		try {
			result = vaFacilityServiceImpl.getAllVisns();
		} catch (GenericException e) {
			e.printStackTrace();
		}		
		assertEquals(result.get(0).getDescription(), "This is a test visn");
	}
	
	@Test
	public void test_getAllVaFacilityType() {
		VaFacilityTypeResponse response = new VaFacilityTypeResponse();
		List<VaFacilityTypeResponse> res = new ArrayList<VaFacilityTypeResponse>();
		
		response.setVaTypeCd("Test");
		response.setDescription("This is a test type");
		res.add(response);
		
		Mockito.when(vaFacilityRepoTemplate.getAllVaFacilityTypeForFormLoad()).thenReturn(res);
		
		List<VaFacilityTypeResponse> result = new ArrayList<VaFacilityTypeResponse>();
		try {
			result = vaFacilityServiceImpl.getAllVaFacilityType();
		} catch (GenericException e) {
			
			e.printStackTrace();
		}
		
		assertEquals(result.get(0).getVaTypeCd(), "Test");
	}
	
	@Test
	public void test_addVaFacility_success() {
		VaFacilityRequest request = new VaFacilityRequest();
		
		request.setAgedDefinition(new Long(1234));
		request.setAppUserName("test user");
		request.setParentVaFacilityCd("parent cd");
		request.setReroutingAllowed(true);
		request.setShortName("test short name");
		request.setVaFacilityCd("test facility cd");
		request.setVaFacilityName("test va facility name");
		request.setVaTypeCd("test type cd");
		request.setVisnIdCd(1234L);
		
		setUpforZipCode();
		when(appUserRepository.findByAppUserName(request.getAppUserName())).thenReturn(list);
		
		Mockito.when(vaFacilityRepoTemplate.insertNewVaFacility(request)).thenReturn(1);
		
		int rowCount = 0;
		try {
			rowCount = vaFacilityServiceImpl.addVaFacility(request);
		} catch (GenericException e) {
			assertEquals(e.getMessage(), "Unable to add a new facility");
		}
		assertEquals(rowCount, 1);
	}
	
	@Test
	public void test_addVaFacility_error() {
		VaFacilityRequest request = new VaFacilityRequest();
		
		request.setAgedDefinition(new Long(1234));
		request.setAppUserName("test user");
		request.setParentVaFacilityCd("parent cd");
		request.setReroutingAllowed(true);
		request.setShortName("test short name");
		request.setVaFacilityCd("test facility cd");
		request.setVaFacilityName("test va facility name");
		request.setVaTypeCd("test type cd");
		request.setVisnIdCd(1234L);
		
		setUpforZipCode();
		when(appUserRepository.findByAppUserName(request.getAppUserName())).thenReturn(list);
		
		Mockito.when(vaFacilityRepoTemplate.insertNewVaFacility(request)).thenReturn(0);
		
		int rowCount = 0;
		try {
			rowCount = vaFacilityServiceImpl.addVaFacility(request);
		} catch (GenericException e) {
			assertEquals(e.getMessage(), "Unable to add a new facility");
		}
		assertEquals(rowCount, 0);		
	}
	
	@Test
	public void test_modifyVaFacility_success() throws Exception {
		VaFacilityRequest request = new VaFacilityRequest();
		
		request.setAgedDefinition(new Long(1234));
		request.setAppUserName("Test");
		request.setParentVaFacilityCd("parent cd");
		request.setReroutingAllowed(true);
		request.setShortName("test short name");
		request.setVaFacilityCd("test facility cd");
		request.setVaFacilityName("test va facility name");
		request.setVaTypeCd("test type cd");
		request.setVisnIdCd(1234L);
		Date date = new Date();
		boolean result = false;
		VaFacility vaFacilityTest = new VaFacility();
		Visn visn = new Visn(1234L, "Test User", date);
		VaFacilityType vaFacilityType = new VaFacilityType("test type cd", "Test User", date);
		
		VaFacility vaFacility = new VaFacility();
		vaFacility.setVisn(visn);
		vaFacility.setVaFacilityType(vaFacilityType);
		vaFacility.setVaFacilityName(request.getVaFacilityName());
		vaFacility.setShortName(request.getShortName());
		vaFacility.setAgedDefinition(request.getAgedDefinition());
		vaFacility.setReroutingAllowed(request.isReroutingAllowed());
		vaFacility.setParentVaFacilityCd(request.getParentVaFacilityCd());
		vaFacility.setModifiedBy(request.getAppUserName());

		Mockito.when(iVaFacilityRepository.findVaFacilityByVaFacilityCd(request.getVaFacilityCd())).thenReturn(vaFacility);
		Mockito.when(iVisnRepository.findOne(request.getVisnIdCd())).thenReturn(visn);
		Mockito.when(iVaFacilityTypeRepository.findOne(request.getVaTypeCd())).thenReturn(vaFacilityType);
		Mockito.when(iVaFacilityRepository.save(vaFacility)).thenReturn(vaFacilityTest);
		
		setUpforZipCode();
		when(appUserRepository.findByAppUserName(request.getAppUserName())).thenReturn(list);
		
		try {
			result = vaFacilityServiceImpl.modifyStation(request);
			assertEquals(result, true);
		} catch (GenericException e) {
			assertEquals(e.getMessage(), "Unable to add a new facility");
		}
	}
	
	@Test
	public void test_modifyVaFacility_error() {
		VaFacilityRequest request = new VaFacilityRequest();
		
		request.setAgedDefinition(new Long(1234));
		request.setAppUserName("test user");
		request.setParentVaFacilityCd("parent cd");
		request.setReroutingAllowed(true);
		request.setShortName("test short name");
		request.setVaFacilityCd("test facility cd");
		request.setVaFacilityName("test va facility name");
		request.setVaTypeCd("test type cd");
		request.setVisnIdCd(1234L);
		
		Date date = new Date();
		boolean result = false;
		VaFacility vaFacilityTest = null;
		Visn visn = new Visn(1234L, "Test User", date);
		VaFacilityType vaFacilityType = new VaFacilityType("test type cd", "Test User", date);
		
		VaFacility vaFacility = new VaFacility();
		vaFacility.setVisn(visn);
		vaFacility.setVaFacilityType(vaFacilityType);
		vaFacility.setVaFacilityName(request.getVaFacilityName());
		vaFacility.setShortName(request.getShortName());
		vaFacility.setAgedDefinition(request.getAgedDefinition());
		vaFacility.setReroutingAllowed(request.isReroutingAllowed());
		vaFacility.setParentVaFacilityCd(request.getParentVaFacilityCd());
		vaFacility.setModifiedBy(request.getAppUserName());
		
		setUpforZipCode();
		when(appUserRepository.findByAppUserName(request.getAppUserName())).thenReturn(list);
		
		Mockito.when(iVaFacilityRepository.findVaFacilityByVaFacilityCd(request.getVaFacilityCd())).thenReturn(vaFacility);
		Mockito.when(iVisnRepository.findOne(request.getVisnIdCd())).thenReturn(visn);
		Mockito.when(iVaFacilityTypeRepository.findOne(request.getVaTypeCd())).thenReturn(vaFacilityType);
		Mockito.when(iVaFacilityRepository.save(vaFacility)).thenReturn(vaFacilityTest);

		try {
			result = vaFacilityServiceImpl.modifyStation(request);
		} catch (GenericException e) {
			assertEquals(e.getMessage(), "Unable to add a new facility");
		}
		assertEquals(result, false);		
	}
	
	/*@Test
	public void test_removeVaFacility() throws Exception {
		String appUserName = "ABC";
		String zip_code= "20855";
		String vaFacilityCd = "10";
		
		ZipCode zipcode = new ZipCode();			
		zipcode.setCreatedBy(appUserName);
		zipcode.setDateCreated(new Date());
		
		VaFacilityRequest request = new VaFacilityRequest();
		request.setAgedDefinition(new Long(1234));
		request.setAppUserName("test user");
		request.setParentVaFacilityCd("parent cd");
		request.setReroutingAllowed(true);
		request.setShortName("test short name");
		request.setVaFacilityCd("test facility cd");
		request.setVaFacilityName("test va facility name");
		request.setVaTypeCd("test type cd");
		request.setVisnIdCd(1234L);
		
		Date date = new Date();
		boolean result = false;
		VaFacility vaFacilityTest = null;
		Visn visn = new Visn(1234L, "Test User", date);
		VaFacilityType vaFacilityType = new VaFacilityType("test type cd", "Test User", date);
		
		VaFacility vaFacility = new VaFacility();
		vaFacility.setVisn(visn);
		vaFacility.setVaFacilityType(vaFacilityType);
		vaFacility.setVaFacilityName(request.getVaFacilityName());
		vaFacility.setShortName(request.getShortName());
		vaFacility.setAgedDefinition(request.getAgedDefinition());
		vaFacility.setReroutingAllowed(request.isReroutingAllowed());
		vaFacility.setParentVaFacilityCd(request.getParentVaFacilityCd());
		vaFacility.setModifiedBy(request.getAppUserName());
		vaFacility.setContactInfos(null);
		vaFacility.setZipCodes(null);
		
		zipcode.setZipCode(zip_code);	
		when(zipCodeRepository.findZipCodeByZipcode(zip_code)).thenReturn(zipcode);
		when(iVaFacilityRepository.findVaFacilityByContactInfo(vaFacilityCd.toUpperCase())).thenReturn(vaFacility);
		
		setUpforZipCode();
		when(appUserRepository.findByAppUserName(appUserName)).thenReturn(list);
		
		Mockito.doNothing().when(zipCodeRepository).delete(zipcode);
		
		ZipCodeRequest request = new ZipCodeRequest();
		request.setZip_code(zip_code);
		request.setAppUserName(appUserName);
		request.setVaFacilityCd(vaFacilityCd);
		
		Boolean response = vaFacilityServiceImpl.removeZipCode(request);
		assertEquals(response, Boolean.TRUE);	
	}*/
	
	@Test
	public void test_searchStation() throws Exception {
		List<VaFacilityResponse> vaFacilityResponseList = new ArrayList<VaFacilityResponse>();
		Long resultsCount = 2L;
		BigDecimal value = new BigDecimal(30);
		VaFacilitySearchRequest vaFacilityRequest = new VaFacilitySearchRequest(100, null, false, 1, null, "TOGUS",
				null, "ADMIN", null);
		VaFacilityResponse vaFacilityResponse = new VaFacilityResponse("119", "ToGUs ME", "TOGUS VA MED/REGIONAL OFFICE CENTER", "FAC", "7", true, value, null, "CPE_OWNER", "06/01/2017 00:00:00 AM");
		vaFacilityResponseList.add(vaFacilityResponse);
		when(vaFacilityRepoTemplate.getSearchResultsCount(vaFacilityRequest)).thenReturn(resultsCount);
		when(vaFacilityRepoTemplate.getSearchResults(vaFacilityRequest)).thenReturn(vaFacilityResponseList);
		GenericResponse visnSearchResponse = new GenericResponse(1, 100,
				"station", 2L);
		visnSearchResponse.setResponse(vaFacilityResponseList);
		try {
			GenericResponse response = vaFacilityServiceImpl.getSearchVisn(vaFacilityRequest);
			assertEquals(visnSearchResponse.getResponse(), response.getResponse());
			logger.info("station searched successfully");
		} catch (GenericException e) {
			logger.info("Exception occured in searching the station");
			assertEquals(e.getMessage(), "Unable to search for a facility");
		}
		
	}
	
	@Test
	public void test_addContactInfo() throws Exception {
		String appUserName = "ABC";
		String vaFacilityCd = "123";
					
		ContactInfoRequest request = new ContactInfoRequest();
		request.setAppUserName(appUserName);
		request.setVaFacilityCd(vaFacilityCd);
		request.setAddress1("1 Oracle Dr.");
		request.setCity("Palo Alto");
		request.setContactName("Mr. ABC");
		request.setContactType("Admin");
		request.setEmail("PII       ");
		request.setZip("99988");
		request.setPhone("1234567890");
	
		Visn toVisn = new Visn(Long.valueOf("16"), "CCSE", new Date());
		VaFacility facility = new VaFacility(vaFacilityCd, toVisn,
				new VaFacilityType("FAC", "CCSE", new Date()), true,"CCSE", new Date());
		when(iVaFacilityRepository.findVaFacilityByVaFacilityCd(vaFacilityCd)).thenReturn(facility);
		
		ContactInfo contactInfoModel = new ContactInfo();
		contactInfoModel.setContactName(request.getContactName());
		contactInfoModel.setContactType(request.getContactType());
		contactInfoModel.setDateCreated(new Date());
		contactInfoModel.setAddress1(request.getAddress1());
		contactInfoModel.setAddress2(request.getAddress2());
		contactInfoModel.setCity(request.getCity());
		contactInfoModel.setZip(request.getZip());
		contactInfoModel.setEmail(request.getEmail());
		contactInfoModel.setPhone(request.getPhone());
		contactInfoModel.setFax(request.getFax());
		contactInfoModel.setCreatedBy(request.getAppUserName());
		Boolean response = Boolean.TRUE;
		when(iContactInfoRepository.saveAndFlush(contactInfoModel)).thenReturn(contactInfoModel);
		
		setUpforZipCode();
		when(appUserRepository.findByAppUserName(appUserName)).thenReturn(list);
		
		response = vaFacilityServiceImpl.postContactInfo(request);
		assertEquals(response, Boolean.TRUE);			
	}
	
	@Test
	public void test_addZipCode() throws Exception {
		String appUserName = "ABC";
		String vaFacilityCd = "10";
		String zip_code= "20855";
		String urban_indicator= "100";
		String fips_county_code= "100";
		String fips_state_code= "10";
		Boolean active= Boolean.TRUE;
		
		Visn toVisn = new Visn(Long.valueOf("16"), "CCSE", new Date());
		VaFacility facility = new VaFacility(vaFacilityCd, toVisn,
				new VaFacilityType("FAC", "CCSE", new Date()), true,"CCSE", new Date());
		when(iVaFacilityRepository.findVaFacilityByVaFacilityCd(vaFacilityCd)).thenReturn(facility);
		
		ZipCode zipcode = new ZipCode();			
		zipcode.setVaFacility(facility);
		zipcode.setCreatedBy(appUserName);
		zipcode.setDateCreated(new Date());
		zipcode.setActiveFlag(Boolean.TRUE.equals(active) ? 'Y' : 'N');
		zipcode.setFipsCountyCode(fips_county_code);
		zipcode.setFipsStateCode(fips_state_code);
		zipcode.setUrbanIndicator(urban_indicator);
		zipcode.setZipCode(zip_code);		
		
		ZipCodeRequest request = new ZipCodeRequest(appUserName, vaFacilityCd, zip_code, urban_indicator,
				fips_county_code, fips_state_code, active);
		Boolean response = Boolean.TRUE;
		
		when(zipCodeRepository.saveAndFlush(zipcode)).thenReturn(zipcode);
		
		setUpforZipCode();
		when(appUserRepository.findByAppUserName(appUserName)).thenReturn(list);
		
		response = vaFacilityServiceImpl.addZipCode(request);
		assertEquals(response, Boolean.TRUE);			
	}
	
	@Test
	public void test_editZipCode() throws Exception {
		String appUserName = "ABC";
		String vaFacilityCd = "10";
		String zip_code= "20855";
		String urban_indicator= "10";
		String fips_county_code= "10";
		String fips_state_code= "10";
		Boolean active= Boolean.TRUE;
		
		ZipCode zipcode = new ZipCode();			
		zipcode.setCreatedBy(appUserName);
		zipcode.setDateCreated(new Date());
		zipcode.setActiveFlag(Boolean.TRUE.equals(active) ? 'Y' : 'N');
		zipcode.setFipsCountyCode(fips_county_code);
		zipcode.setFipsStateCode(fips_state_code);
		zipcode.setUrbanIndicator(urban_indicator);
		zipcode.setZipCode(zip_code);	
		when(zipCodeRepository.findZipCodeByZipcode(zip_code)).thenReturn(zipcode);
				
		zipcode.setActiveFlag('N');
		zipcode.setFipsCountyCode("100");
		zipcode.setFipsStateCode("50");
		zipcode.setUrbanIndicator("100");
		when(zipCodeRepository.saveAndFlush(zipcode)).thenReturn(zipcode);
		
		setUpforZipCode();
		when(appUserRepository.findByAppUserName(appUserName)).thenReturn(list);
		
		ZipCodeRequest request = new ZipCodeRequest(appUserName, vaFacilityCd, zip_code, urban_indicator,
				fips_county_code, fips_state_code, active);
		
		Boolean response = vaFacilityServiceImpl.editZipCode(request);
		assertEquals(response, Boolean.TRUE);	
	}
	
	@Test
	public void test_removeZipCode() throws Exception {
		String appUserName = "ABC";
		String zip_code= "20855";
		String vaFacilityCd = "10";
		
		ZipCode zipcode = new ZipCode();			
		zipcode.setCreatedBy(appUserName);
		zipcode.setDateCreated(new Date());
		
		zipcode.setZipCode(zip_code);	
		when(zipCodeRepository.findZipCodeByZipcode(zip_code)).thenReturn(zipcode);

		setUpforZipCode();
		when(appUserRepository.findByAppUserName(appUserName)).thenReturn(list);
		
		Mockito.doNothing().when(zipCodeRepository).delete(zipcode);
		
		ZipCodeRequest request = new ZipCodeRequest();
		request.setZip_code(zip_code);
		request.setAppUserName(appUserName);
		request.setVaFacilityCd(vaFacilityCd);
		
		Boolean response = vaFacilityServiceImpl.removeZipCode(request);
		assertEquals(response, Boolean.TRUE);	
	}
		
	private void setUpforZipCode() {
		Set<UserRoleUsages> userRoles = new HashSet<>();
		AppUser appUser = new AppUser();
		appUser.setAppUserId(Long.valueOf(100));
		appUser.setUserName("ABC");
		appUser.setFirstName("Test");
		appUser.setLastName("Test");
		
		// Role Admin
		AppRole appRole1 = new AppRole();
		appRole1.setAppRoleId(Long.valueOf(1));
		appRole1.setRoleName("ADMIN");
		appRole1.setDescription("Admin can do everything");
		
		UserRoleUsagesId id1 = new UserRoleUsagesId(Long.valueOf(1), Long.valueOf(1), "VAFAC001", "Y", "", new Date());
		UserRoleUsages userRole1 = new UserRoleUsages(id1, new VaFacility(), appRole1, appUser);
		userRoles.add(userRole1);
		appUser.setUserRoleUsages(userRoles);
		
		list.add(appUser);
	}
	
	@Test
	public void test_getStationInfo() throws Exception {
		String stationName = "test va facility name";
		
		VaFacility request = new VaFacility();
		
		request.setAgedDefinition(new Long(1234));
		request.setCreatedBy("test user");
		request.setParentVaFacilityCd("parent cd");
		request.setReroutingAllowed(true);
		request.setShortName("test short name");
		request.setVaFacilityCd("test facility cd");
		request.setVaFacilityName(stationName);
		request.setVaFacilityType(new VaFacilityType("test type cd", "test user", new Date()));
		request.setVisn(new Visn(1234L, "test user", new Date()));
		request.setDateCreated(new Date());

		when(iVaFacilityRepository.findVaFacilityByStation(stationName)).thenReturn(request);
		VaFacilityRequest vaFacilityRequest = vaFacilityServiceImpl.getStationInfo(stationName);
		assertEquals(vaFacilityRequest.getVaFacilityName(), stationName);
		assertEquals(vaFacilityRequest.getVaFacilityCd(), request.getVaFacilityCd());
		assertEquals(vaFacilityRequest.getParentVaFacilityCd(), request.getParentVaFacilityCd());
		assertEquals(vaFacilityRequest.getShortName(), request.getShortName());
		assertEquals(vaFacilityRequest.getVisnIdCd(), request.getVisn().getVisnIdCd());
		assertEquals(vaFacilityRequest.getAgedDefinition(), request.getAgedDefinition());
	}	
}