package gov.va.med.fee.service.impl;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.junit.Assert.assertThat;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.runners.MockitoJUnitRunner;

import gov.va.med.domain.fee.AppRole;
import gov.va.med.domain.fee.AppUser;
import gov.va.med.domain.fee.SsoAuthentication;
import gov.va.med.domain.fee.UserRoleUsages;
import gov.va.med.domain.fee.UserRoleUsagesId;
import gov.va.med.domain.fee.VaFacility;
import gov.va.med.fee.dao.IAppUserRepository;
import gov.va.med.fee.dao.IAuthenticatorRepository;
import gov.va.med.fee.exceptions.GenericException;
import gov.va.med.fee.model.response.UserRoles;
import gov.va.med.fee.service.impl.AuthenticatorServiceImpl;

/**
 * @author Eyuel Taddese
 *
 */

@RunWith(MockitoJUnitRunner.class)
public class AuthenticatorServiceImplTest {

	@Mock
	private AuthenticatorServiceImpl authenticatorService;

	@Mock
	private IAuthenticatorRepository authenticatorRepository;

	@Mock
	private IAppUserRepository appUserRepository;

	/**
	 * 
	 */
	@Before
	public void setup() {
		authenticatorService = new AuthenticatorServiceImpl();

		authenticatorRepository = Mockito.mock(IAuthenticatorRepository.class);
		appUserRepository = Mockito.mock(IAppUserRepository.class);

		authenticatorService.setAuthenticatorDAO(authenticatorRepository);
		authenticatorService.setAppUserRoleRepository(appUserRepository);
	}

	@Test
	public void testMockIsWorking() {
		Assert.assertNotNull("Authenticator Service Injection Failed", authenticatorService);
		Assert.assertNotNull("AppUserRepository Injection failed", appUserRepository);
		Assert.assertNotNull("Authenticator Repository failed", authenticatorRepository);
	}

	@Test
	public void testgetUsernameFromAspAuth() throws Exception {
		// Setup
		List<SsoAuthentication> ssoauthentications = new ArrayList<>();
		ssoauthentications.add(new SsoAuthentication("DNS   AAA", "2222", "VHAISUSERNAME2"));

		// Stub
		Mockito.when(authenticatorRepository.findByGuid("DNS   AAA")).thenReturn(ssoauthentications);
		try {
			String userName = authenticatorService.getUsernameFromAspAuth("DNS   AAA");
			assertThat(userName, equalTo("VHAISUSERNAME2"));
			Mockito.verify(authenticatorRepository, Mockito.times(1)).findByGuid("DNS   AAA");
		} catch (Exception e) {
			Assert.fail("GetUserNameFromAspAuth() failed exception thrown");
		}
	}

	@Test
	public void testAuthenticate() {
		// Setup
		List<SsoAuthentication> ssoauthentications = new ArrayList<>();
		ssoauthentications.add(new SsoAuthentication("23456", "2222", "VHAISUSERNAME2"));

		List<AppUser> appUsers = new ArrayList<>();
		Set<UserRoleUsages> userRoles = new HashSet<>();

		// User named Alvin
		AppUser appUser = new AppUser();
		appUser.setAppUserId(Long.valueOf(1));
		appUser.setUserName("VHAISUSERNAME2");
		appUser.setFirstName("Alvin");
		appUser.setLastName("Stone");

		// Role Admin
		AppRole appRole1 = new AppRole();
		appRole1.setAppRoleId(Long.valueOf(1));
		appRole1.setRoleName("ADMIN");
		appRole1.setDescription("Admin can do everything");

		// Role Clerk
		AppRole appRole2 = new AppRole();
		appRole1.setAppRoleId(Long.valueOf(2));
		appRole2.setRoleName("Clerk");
		appRole2.setDescription("Clerk can view claims");

		UserRoleUsagesId id1 = new UserRoleUsagesId(Long.valueOf(1), Long.valueOf(1), "VAFAC001", "Y", "", new Date());
		UserRoleUsagesId id2 = new UserRoleUsagesId(Long.valueOf(2), Long.valueOf(1), "VAFAC001", "Y", "", new Date());

		// Each Indiviual Roles
		UserRoleUsages userRole1 = new UserRoleUsages(id1, new VaFacility(), appRole1, appUser);
		UserRoleUsages userRole2 = new UserRoleUsages(id2, new VaFacility(), appRole2, appUser);

		// Add to the UserRoleUsages list each roles
		userRoles.add(userRole1);
		userRoles.add(userRole2);

		// Add the Set of Roles to the user
		appUser.setUserRoleUsages(userRoles);

		// Stub
		Mockito.when(authenticatorRepository.findByGuid("23456")).thenReturn(ssoauthentications);
		Mockito.when(appUserRepository.findByAppUserName("VHAISUSERNAME2")).thenReturn(appUsers);
		try {
			boolean authenticateResult = authenticatorService.authenticate("45bRDDosGfqjZiScoXdCwQ==");
			Assert.assertTrue(authenticateResult);
		} catch (GenericException u) {
			Assert.fail("User Authentication failed");
		}
	}

	@Test
	public void when_no_validation_error_then_GetUserRole() {
		// Setup
		List<SsoAuthentication> ssoauthentications = new ArrayList<>();
		ssoauthentications.add(new SsoAuthentication("DNS   AAA", "2222", "VHAISUSERNAME2"));

		List<AppUser> appUsers = new ArrayList<>();
		Set<UserRoleUsages> userRoles = new HashSet<>();

		// User named Alvin
		AppUser appUser = new AppUser();
		appUser.setAppUserId(Long.valueOf(1));
		appUser.setUserName("VHAISUSERNAME2");
		appUser.setFirstName("Alvin");
		appUser.setLastName("Stone");

		// Role Admin
		AppRole appRole1 = new AppRole();
		appRole1.setAppRoleId(Long.valueOf(1));
		appRole1.setRoleName("ADMIN");
		appRole1.setDescription("Admin can do everything");

		// Role Clerk
		AppRole appRole2 = new AppRole();
		appRole1.setAppRoleId(Long.valueOf(2));
		appRole2.setRoleName("Clerk");
		appRole2.setDescription("Clerk can view claims");

		UserRoleUsagesId id1 = new UserRoleUsagesId(Long.valueOf(1), Long.valueOf(1), "VAFAC001", "Y", "", new Date());
		UserRoleUsagesId id2 = new UserRoleUsagesId(Long.valueOf(2), Long.valueOf(1), "VAFAC001", "Y", "", new Date());

		// Each Indiviual Roles
		UserRoleUsages userRole1 = new UserRoleUsages(id1, new VaFacility(), appRole1, appUser);
		UserRoleUsages userRole2 = new UserRoleUsages(id2, new VaFacility(), appRole2, appUser);

		// Add to the UserRoleUsages list each roles
		userRoles.add(userRole1);
		userRoles.add(userRole2);

		// Add the Set of Roles to the user
		appUser.setUserRoleUsages(userRoles);

		// Stub
		Mockito.when(authenticatorRepository.findByGuid("DNS   AAA")).thenReturn(ssoauthentications);
		Mockito.when(appUserRepository.findByAppUserName("VHAISUSERNAME2")).thenReturn(appUsers);

		// Expected UserRole
		UserRoles u1 = new UserRoles("ADMIN", "Admin can do everything");

		/*
		 * try { authenticatorService.authenticate("DNS   AAA"); List<UserRoles>
		 * userRoles1 = authenticatorService.getUserRoles(); if(userRoles1 !=
		 * null) { UserRoles userRole0 = userRoles1.get(0);
		 * Assert.assertThat(userRole0, equalTo(u1)); } } catch(Exception e) {
		 * Assert.fail(e.toString()+
		 * ": Exception thrown when_no_validation_error_then_GetUserRole"); }
		 */
	}
}
