package gov.va.med.fee.integration;

import static org.junit.Assert.assertFalse;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.content;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.test.context.web.WebAppConfiguration;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.ResultActions;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;
import org.springframework.web.context.WebApplicationContext;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.SerializationFeature;

import gov.va.med.fee.configuration.AppConfig;
import gov.va.med.fee.configuration.hibernate.PersistenceConfig;
import gov.va.med.fee.configuration.spring.SpringMvcConfig;
import gov.va.med.fee.model.request.CodeLookupRequest;

/**
 * 
 * @author Muneshwar Baiah
 *
 */

@WebAppConfiguration
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(classes = { AppConfig.class, PersistenceConfig.class, SpringMvcConfig.class })
public class CodeLookupControllerIntegrationTest {
	private static final Logger logger = LogManager.getLogger(CodeLookupControllerIntegrationTest.class);

	@Autowired
	private WebApplicationContext wac;
	private MockMvc mockMvc;

	@Before
	public void setup() {
		this.mockMvc = MockMvcBuilders.webAppContextSetup(this.wac).build();
	}

	@Test
	public void getMedicalCodesSuccess() throws Exception {
		
		CodeLookupRequest codeLookupRequest = new CodeLookupRequest();
		codeLookupRequest.setCode("");
		codeLookupRequest.setCodeType("");
		codeLookupRequest.setDescending(false);
		codeLookupRequest.setPageNumber(1);
		codeLookupRequest.setPageSize(2);
		codeLookupRequest.setSortColumn("code");
		
		ObjectMapper mapper = new ObjectMapper();
		mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
		ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();

		String requestJson = ow.writeValueAsString(codeLookupRequest);

		ResultActions s = mockMvc
				.perform(
						post("/api/v1/tools/medicalCodes").contentType(MediaType.APPLICATION_JSON).content(requestJson))
				.andDo(print()).andExpect(status().isOk())
				.andExpect(content().contentType("application/json;charset=UTF-8"));
		logger.info("getMedicalCodesSuccess:" + s.andReturn().getResponse().getStatus());
		assertFalse(HttpStatus.OK.value() != s.andReturn().getResponse().getStatus());
	}
	

	@Test
	public void getMedicalCodesFailure() throws Exception {	
		
		CodeLookupRequest codeLookupRequest = new CodeLookupRequest();
		codeLookupRequest.setCode("");
		codeLookupRequest.setCodeType("");
		codeLookupRequest.setDescending(false);
		codeLookupRequest.setPageNumber(1);
		//codeLookupRequest.setPageSize(2);
		codeLookupRequest.setSortColumn("code");
		
		ObjectMapper mapper = new ObjectMapper();
		mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
		ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();

		String requestJson = ow.writeValueAsString(codeLookupRequest);

		
		ResultActions s = mockMvc
				.perform(
						post("/api/v1/tools/medicalCodes").contentType(MediaType.APPLICATION_JSON).content(requestJson))
				.andDo(print()).andExpect(status().isInternalServerError());

		logger.info("getMedicalCodesFailure:" + s.andReturn().getResponse().getStatus());
		assertFalse(HttpStatus.INTERNAL_SERVER_ERROR.value() != s.andReturn().getResponse().getStatus());
	}
	
	@Test
	public void getDiagnosisCodesSuccess() throws Exception {

		CodeLookupRequest codeLookupRequest = new CodeLookupRequest();
		codeLookupRequest.setCode("");
		codeLookupRequest.setCodeType("");
		codeLookupRequest.setDescending(false);
		codeLookupRequest.setPageNumber(1);
		codeLookupRequest.setPageSize(2);
		codeLookupRequest.setSortColumn("code");
		
		ObjectMapper mapper = new ObjectMapper();
		mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
		ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();

		String requestJson = ow.writeValueAsString(codeLookupRequest);

		ResultActions s = mockMvc
				.perform(
						post("/api/v1/tools/diagnosisCodes").contentType(MediaType.APPLICATION_JSON).content(requestJson))
				.andDo(print()).andExpect(status().isOk())
				.andExpect(content().contentType("application/json;charset=UTF-8"));
		logger.info("getDiagnosisCodesSuccess:" + s.andReturn().getResponse().getStatus());
		assertFalse(HttpStatus.OK.value() != s.andReturn().getResponse().getStatus());
	}
	

	@Test
	public void getDiagnosisCodesFailure() throws Exception {	
		CodeLookupRequest codeLookupRequest = new CodeLookupRequest();
		codeLookupRequest.setCode("");
		codeLookupRequest.setCodeType("");
		codeLookupRequest.setDescending(false);
		codeLookupRequest.setPageNumber(1);
		//codeLookupRequest.setPageSize(2);
		codeLookupRequest.setSortColumn("code");
		
		ObjectMapper mapper = new ObjectMapper();
		mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
		ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();

		String requestJson = ow.writeValueAsString(codeLookupRequest);

		
		ResultActions s = mockMvc
				.perform(
						post("/api/v1/tools/diagnosisCodes").contentType(MediaType.APPLICATION_JSON).content(requestJson))
				.andDo(print()).andExpect(status().isInternalServerError());

		logger.info("getDiagnosisCodesFailure:" + s.andReturn().getResponse().getStatus());
		assertFalse(HttpStatus.INTERNAL_SERVER_ERROR.value() != s.andReturn().getResponse().getStatus());
	}
	
}