package gov.va.med.fee.controller;

import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.when;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.delete;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.put;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.content;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.mockito.runners.MockitoJUnitRunner;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.SerializationFeature;

import gov.va.med.fee.filter.CORSFilter;
import gov.va.med.fee.model.request.ContactInfoRequest;
import gov.va.med.fee.model.request.VaFacilityRequest;
import gov.va.med.fee.model.request.VaFacilitySearchRequest;
import gov.va.med.fee.model.request.ZipCodeRequest;
import gov.va.med.fee.model.response.GenericResponse;
import gov.va.med.fee.model.response.VaFacilityResponse;
import gov.va.med.fee.model.response.VaFacilityTypeResponse;
import gov.va.med.fee.model.response.VaFacilityVisnResponse;
import gov.va.med.fee.service.IVaFacilityService;

@RunWith(MockitoJUnitRunner.class)
public class VaFacilityControllerTest {
	private static final Logger logger = LogManager.getLogger(VaFacilityControllerTest.class);
	
	private MockMvc mockMvc;
	
	@InjectMocks
	private VaFacilityController vaFacilityController;
	
	@Mock
	IVaFacilityService ivaFacilityService;
	
	@Before
	public void init() throws Exception {
		logger.debug("init");
		MockitoAnnotations.initMocks(this);
		mockMvc = MockMvcBuilders.standaloneSetup(vaFacilityController).addFilter(new CORSFilter()).build();
	}
	
	@Test
	public void test_getAllVisn() throws Exception {
		VaFacilityVisnResponse response = new VaFacilityVisnResponse();
		List<VaFacilityVisnResponse> res = new ArrayList<VaFacilityVisnResponse>();
		
		response.setDescription("test description");
		response.setVisnIdCd(1234L);
		res.add(response);
		

		when(ivaFacilityService.getAllVisns()).thenReturn(res);
		
		mockMvc.perform(get("/api/v1/VaFacility/station/visn")).andDo(print()).andExpect(status().isOk())
		.andExpect(content().contentType(MediaType.APPLICATION_JSON_UTF8_VALUE))
		.andExpect(jsonPath("$.[0].description", is("test description")))
		.andExpect(jsonPath("$.[0].visnIdCd", is(1234)));
		
		verify(ivaFacilityService, times(1)).getAllVisns();
		verifyNoMoreInteractions(ivaFacilityService);		
	}
	
	@Test
	public void test_getAllFacilityType() throws Exception {
		VaFacilityTypeResponse response = new VaFacilityTypeResponse();
		List<VaFacilityTypeResponse> res = new ArrayList<VaFacilityTypeResponse>();
		
		response.setDescription("test description");
		response.setVaTypeCd("test123");
		res.add(response);
		

		when(ivaFacilityService.getAllVaFacilityType()).thenReturn(res);
		
		mockMvc.perform(get("/api/v1/VaFacility/station/type")).andDo(print()).andExpect(status().isOk())
		.andExpect(content().contentType(MediaType.APPLICATION_JSON_UTF8_VALUE))
		.andExpect(jsonPath("$.[0].description", is("test description")))
		.andExpect(jsonPath("$.[0].vaTypeCd", is("test123")));
		
		verify(ivaFacilityService, times(1)).getAllVaFacilityType();
		verifyNoMoreInteractions(ivaFacilityService);	
	}
	
	@Test
	public void test_searchStation() throws Exception {
		List<VaFacilityResponse> vaFacilityResponseList = new ArrayList<VaFacilityResponse>();
		Long resultsCount = 1L;

		BigDecimal value = new BigDecimal(30);
		VaFacilitySearchRequest vaFacilityRequest = new VaFacilitySearchRequest(100, null, false, 1, null, "TOGUS",
				null, "ADMIN", null);

		VaFacilityResponse vaFacilityResponse = new VaFacilityResponse("119", "ToGUs ME",
				"TOGUS VA MED/REGIONAL OFFICE CENTER", "FAC", "7", true, value, null, "CPE_OWNER",
				"06/01/2017 00:00:00 AM");
		vaFacilityResponseList.add(vaFacilityResponse);

		GenericResponse visnSearchResponse = new GenericResponse(1, 100, "station", resultsCount);
		visnSearchResponse.setResponse(vaFacilityResponseList);

		when(ivaFacilityService.getSearchVisn(vaFacilityRequest)).thenReturn(visnSearchResponse);

		mockMvc.perform(post("/api/v1/VaFacility/station/search").contentType(MediaType.APPLICATION_JSON)
				.content(getRequestJson(vaFacilityRequest))).andDo(print());

		verify(ivaFacilityService, times(1)).getSearchVisn(vaFacilityRequest);
		verifyNoMoreInteractions(ivaFacilityService);
	}
	
	@Test
	public void test_addFacility() throws Exception {
		VaFacilityRequest request = new VaFacilityRequest();
		request.setAgedDefinition(1234L);
		request.setAppUserName("Test");
		request.setParentVaFacilityCd("Test");
		request.setVaFacilityCd("test123");
		request.setReroutingAllowed(true);
		request.setShortName("test");
		request.setVaFacilityCd("test");
		request.setVaFacilityName("test");
		request.setVaTypeCd("test");
		request.setVisnIdCd(1234L);
		
		ObjectMapper mapper = new ObjectMapper();
	    mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
	    ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
		String requestJson = ow.writeValueAsString(request);
		
		when(ivaFacilityService.addVaFacility(Mockito.anyObject())).thenReturn(1);
		mockMvc.perform(post("/api/v1/VaFacility/station").contentType(MediaType.APPLICATION_JSON).content(requestJson)).andDo(print())
		.andExpect(status().is2xxSuccessful())
		.andExpect(content().contentType(MediaType.APPLICATION_JSON_UTF8_VALUE))
		.andExpect(jsonPath("$.result", is(1)));	
	}
	
	
	@Test
	public void test_modifyFacility() throws Exception {
		VaFacilityRequest request = new VaFacilityRequest();
		request.setAgedDefinition(1234L);
		request.setAppUserName("Test");
		request.setParentVaFacilityCd("Test");
		request.setVaFacilityCd("test123");
		request.setReroutingAllowed(true);
		request.setShortName("test");
		request.setVaFacilityCd("test");
		request.setVaFacilityName("test");
		request.setVaTypeCd("test");
		request.setVisnIdCd(1234L);
		boolean value = true;
		
		
		ObjectMapper mapper = new ObjectMapper();
	    mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
	    ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
		String requestJson = ow.writeValueAsString(request);
		
		when(ivaFacilityService.modifyStation(Mockito.anyObject())).thenReturn(value);
		mockMvc.perform(put("/api/v1/VaFacility/station").contentType(MediaType.APPLICATION_JSON).content(requestJson)).andDo(print())
		.andExpect(status().is2xxSuccessful())
		.andExpect(content().contentType(MediaType.APPLICATION_JSON_UTF8_VALUE))
		.andExpect(jsonPath("$.result", is(value)));	
	}
	
	@Test
	public void test_removeStationCode() throws Exception {
		String stationName= "20855";
		
		Boolean response = Boolean.TRUE;
		when(ivaFacilityService.deleteStationInfo(stationName)).thenReturn(response);
		
		mockMvc.perform(delete("/api/v1/VaFacility/station/{stationName}", "20855").contentType(MediaType.APPLICATION_JSON).content(getRequestJson(stationName))).andDo(print());
		
		verify(ivaFacilityService, times(1)).deleteStationInfo(stationName);
	}
	
	@Test
	public void test_addZipCode() throws Exception {
		String appUserName = "ABC";
		String vaFacilityCd = "10";
		String zip_code= "20855";
		String urban_indicator= "10";
		String fips_county_code= "10";
		String fips_state_code= "10";
		Boolean active= Boolean.TRUE;
		
		ZipCodeRequest request = new ZipCodeRequest(appUserName, vaFacilityCd, zip_code, urban_indicator,
				fips_county_code, fips_state_code, active);
		
		Boolean response = Boolean.TRUE;
		when(ivaFacilityService.addZipCode(request)).thenReturn(response);
		mockMvc.perform(post("/api/v1/VaFacility/station/zipcode").contentType(MediaType.APPLICATION_JSON)
				.content(getRequestJson(request))).andDo(print());
		verify(ivaFacilityService, times(1)).addZipCode(request);
	}
	
	@Test
	public void test_editZipCode() throws Exception {
		String appUserName = "ABC"; //Not null
		String vaFacilityCd = "10"; //Not null, size <= 12
		String zip_code= "20855"; //Not null, size <= 10
		String urban_indicator= "100"; //size <= 10
		String fips_county_code= "100"; //size <= 3
		String fips_state_code= "50"; //size <= 2
		Boolean active= Boolean.TRUE;
		
		ZipCodeRequest request = new ZipCodeRequest(appUserName, vaFacilityCd, zip_code, urban_indicator,
				fips_county_code, fips_state_code, active);
		
		Boolean response = Boolean.TRUE;
		when(ivaFacilityService.editZipCode(request)).thenReturn(response);
		
		mockMvc.perform(put("/api/v1/VaFacility/station/zipcode").contentType(MediaType.APPLICATION_JSON)
				.content(getRequestJson(request))).andDo(print());
		verify(ivaFacilityService, times(1)).editZipCode(request);
	}
	
	@Test
	public void test_removeZipCode() throws Exception {
		String zip_code= "20855";
		String appUserName = "ABC";
		String vaFacilityCd = "10";
		
		ZipCodeRequest request = new ZipCodeRequest();
		request.setZip_code(zip_code);
		request.setAppUserName(appUserName);
		request.setVaFacilityCd(vaFacilityCd);
		
		Boolean response = Boolean.TRUE;
		when(ivaFacilityService.removeZipCode(request)).thenReturn(response);
		
		mockMvc.perform(delete("/api/v1/VaFacility/station/zipcode").contentType(MediaType.APPLICATION_JSON)
				.content(getRequestJson(request))).andDo(print());
		
		verify(ivaFacilityService, times(1)).removeZipCode(request);
	}
	
	@Test
	public void test_addContactInfo() throws Exception {
		String appUserName = "ABC";
		String vaFacilityCd = "123";
					
		ContactInfoRequest request = new ContactInfoRequest();
		request.setAppUserName(appUserName);
		request.setVaFacilityCd(vaFacilityCd);
		request.setAddress1("1 Oracle Dr.");
		request.setCity("Palo Alto");
		request.setContactName("Mr. ABC");
		request.setContactType("Admin");
		request.setEmail("PII       ");
		request.setZip("99988");
		request.setPhone("1234567890");
	
		Boolean response = Boolean.TRUE;
		when(ivaFacilityService.postContactInfo(request)).thenReturn(response);
		
		mockMvc.perform(post("/api/v1/VaFacility/station/contactInfo").contentType(MediaType.APPLICATION_JSON)
				.content(getRequestJson(request))).andDo(print());
		
		verify(ivaFacilityService, times(1)).postContactInfo(request);
	}
	
	@Test
	public void test_modifyContactInfo() throws Exception {
		String appUserName = "ABC";
		String vaFacilityCd = "123";
					
		ContactInfoRequest request = new ContactInfoRequest();
		request.setAppUserName(appUserName);
		request.setVaFacilityCd(vaFacilityCd);
		request.setAddress1("1 Oracle Dr.");
		request.setCity("Palo Alto");
		request.setContactName("Mr. ABC");
		request.setContactType("Admin");
		request.setEmail("PII       ");
		request.setZip("99988");
		request.setPhone("1234567890");
	
		Boolean response = Boolean.TRUE;
		when(ivaFacilityService.postContactInfo(request)).thenReturn(response);
		
		mockMvc.perform(post("/api/v1/VaFacility/station/contactInfo").contentType(MediaType.APPLICATION_JSON)
				.content(getRequestJson(request))).andDo(print());
		
		verify(ivaFacilityService, times(1)).postContactInfo(request);
	}
	private String getRequestJson(Object request) {
		String requestJson = null;
		try {
			ObjectMapper mapper = new ObjectMapper();
		    mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
		    ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
			requestJson = ow.writeValueAsString(request);
			//System.out.println("requestJson - " + requestJson);
		} catch (JsonProcessingException e) {
			logger.error("getRequestJson JsonProcessingException -- ");
			e.printStackTrace();
		
		}		
		return requestJson;
	}
}