package gov.va.med.fee.controller;

import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.when;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.content;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

import java.util.ArrayList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.runners.MockitoJUnitRunner;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.SerializationFeature;

import gov.va.med.fee.filter.CORSFilter;
import gov.va.med.fee.model.request.UserSearchRequest;
import gov.va.med.fee.model.response.UserSearchPage;
import gov.va.med.fee.model.response.UserSearchResponse;
import gov.va.med.fee.service.IUserSearchService;

@RunWith(MockitoJUnitRunner.class)
public class UserSearchControllerTest {
	private static final Logger logger = LogManager.getLogger(UserSearchControllerTest.class);
	
	private MockMvc mockMvc;
	
	@InjectMocks
	private UserSearchController userSearchController;
	
	@Mock
	IUserSearchService iUserSearchService;
	
	@Before
	public void init() throws Exception {
		logger.debug("init");
		MockitoAnnotations.initMocks(this);
		mockMvc = MockMvcBuilders.standaloneSetup(userSearchController).addFilter(new CORSFilter()).build();
	}
	
	@Test
	public void test_searchUser() throws Exception {
		UserSearchRequest userSearchRequest = new UserSearchRequest();
		userSearchRequest.setUserName("DNS   ABC");
		userSearchRequest.setEmail("PII       ");
		userSearchRequest.setPhone("V6789872563");
		userSearchRequest.setFirstName("Alvin");
		userSearchRequest.setLastName("Stone");
		userSearchRequest.setStationId("120");
		
		UserSearchPage userSearchPage = new UserSearchPage();
		
		List<UserSearchResponse> usersfromSearch = new ArrayList<UserSearchResponse>();
		UserSearchResponse userSearchResponse = new UserSearchResponse();
		userSearchResponse.setActive("Y");
		userSearchResponse.setUserName("DNS   ABC");
		userSearchResponse.setFirstName("Alvin");
		userSearchResponse.setLastName("Stone");
		userSearchResponse.setEmail("PII       ");
		userSearchResponse.setPhone("6789985267");
		usersfromSearch.add(userSearchResponse);
		
		userSearchPage.setUsersfromSearch(usersfromSearch);
		
		when(iUserSearchService.searchUser(userSearchRequest)).thenReturn(userSearchPage);
		
		ObjectMapper mapper = new ObjectMapper();
	    mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
	    ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
		String requestJson = ow.writeValueAsString(userSearchRequest);
		
		mockMvc.perform(post("/api/v1/user/search").contentType(MediaType.APPLICATION_JSON).content(requestJson)).andDo(print())
			.andExpect(status().isOk())
			.andExpect(content().contentType(MediaType.APPLICATION_JSON_UTF8_VALUE))
			.andExpect(jsonPath("$.usersfromSearch[0].firstName", is("Alvin")))
			.andExpect(jsonPath("$.usersfromSearch[0].lastName", is("Stone")))
			.andExpect(jsonPath("$.usersfromSearch[0].email", is("PII       ")))
			.andExpect(jsonPath("$.usersfromSearch[0].phone", is("6789985267")))
			.andExpect(jsonPath("$.usersfromSearch[0].active", is("Y")))
			.andExpect(jsonPath("$.usersfromSearch[0].userName", is("DNS   ABC")));
		
		
		verify(iUserSearchService, times(1)).searchUser(userSearchRequest);
		verifyNoMoreInteractions(iUserSearchService);
	}
}