package gov.va.med.fee.controller;

import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.content;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.SerializationFeature;

import gov.va.med.fee.filter.CORSFilter;
import gov.va.med.fee.model.request.UnmatchedClaimPaymentRequest;
import gov.va.med.fee.model.request.UnmatchedReconciliationRequest;
import gov.va.med.fee.model.response.PaymentLineItemsResponse;
import gov.va.med.fee.model.response.UnmatchedClaimPaymentsResponse;
import gov.va.med.fee.model.response.UnmatchedClaimReconciliationResponse;
import gov.va.med.fee.service.IUnmatchedReconciliationService;


@RunWith(MockitoJUnitRunner.class)
public class UnmatchedClaimReconciliationControllerTest {
	private static final Logger logger = LogManager.getLogger(UnmatchedClaimReconciliationControllerTest.class);
	
	private MockMvc mockMvc;
	@InjectMocks
	UnmatchedClaimReconciliationController unmatchedClaimReconciliationController;
	@Mock
	IUnmatchedReconciliationService iUnmatchedReconciliationService;
	
	
	@Before
	public void init() throws Exception {
		mockMvc = MockMvcBuilders.standaloneSetup(unmatchedClaimReconciliationController).addFilters(new CORSFilter()).build();
	}

	@Test
	public void test_getClaimsToReconcile() throws Exception {
		//"claimType":"DENT","claimDate":"2017-06-13 23:15:23.0","claimAmount":1,"claimId":103
		UnmatchedClaimPaymentsResponse response = new UnmatchedClaimPaymentsResponse();
		response.setClaimType("DENT");
		response.setClaimDate("06/13/2017");
		response.setClaimAmount(new BigDecimal(1));
		response.setClaimId(103L);
				
		when(iUnmatchedReconciliationService.getUnmatchedClaimsInformation(103L)).thenReturn(response);
		
		mockMvc.perform(get("/api/v1/reconciliation/unmatchedClaim/{claimIndex}", 103L)).andDo(print()).andExpect(status().isOk())
			.andExpect(content().contentType(MediaType.APPLICATION_JSON_UTF8_VALUE))
			.andExpect(jsonPath("$.claimType", is("DENT")))
			.andExpect(jsonPath("$.claimDate", is("06/13/2017")))
			.andExpect(jsonPath("$.claimAmount", is(1)))
			.andExpect(jsonPath("$.claimId", is(103)));
		
		verify(iUnmatchedReconciliationService, times(1)).getUnmatchedClaimsInformation(103L);
	}

	@Test
	public void test_getclaimPaymentsInformation() throws Exception{
		
		UnmatchedReconciliationRequest request = new UnmatchedReconciliationRequest("C", "88", null,
				"ERROR", "424828", null, null, null);
		
		
		List<UnmatchedClaimPaymentsResponse> unmatchedClaimPaymentsList = new ArrayList<UnmatchedClaimPaymentsResponse>();
		UnmatchedClaimPaymentsResponse unmatchedClaimPaymentsResponse = new UnmatchedClaimPaymentsResponse();
		unmatchedClaimPaymentsResponse.setClaimId(900L);
		unmatchedClaimPaymentsResponse.setClaimType("INST");
		unmatchedClaimPaymentsResponse.setClaimDate("2014-10-01 10:21:25.0");
		unmatchedClaimPaymentsResponse.setClaimAmount(new BigDecimal(730));
		unmatchedClaimPaymentsList.add(unmatchedClaimPaymentsResponse);
		
		
		unmatchedClaimPaymentsResponse = new UnmatchedClaimPaymentsResponse();
		unmatchedClaimPaymentsResponse.setClaimId(1000L);
		unmatchedClaimPaymentsResponse.setClaimType("DENT");
		unmatchedClaimPaymentsResponse.setClaimDate("2015-10-01 10:21:25.0");
		unmatchedClaimPaymentsResponse.setClaimAmount(new BigDecimal(830));
		unmatchedClaimPaymentsList.add(unmatchedClaimPaymentsResponse);
		
		when(iUnmatchedReconciliationService.getUnmatchedClaimPayments(request)).thenReturn(unmatchedClaimPaymentsList);
		
		mockMvc.perform(post("/api/v1/reconciliation/unmatchedClaim/claimPayments").contentType(MediaType.APPLICATION_JSON)
				.content(getRequestJson(request))).andDo(print())
		.andExpect(jsonPath("$.[0].claimType", is("INST")))
		.andExpect(jsonPath("$.[0].claimDate", is("2014-10-01 10:21:25.0")))
		.andExpect(jsonPath("$.[0].claimAmount", is(730)))
		.andExpect(jsonPath("$.[0].claimId", is(900)))
		.andExpect(jsonPath("$.[1].claimType", is("DENT")))
		.andExpect(jsonPath("$.[1].claimDate", is("2015-10-01 10:21:25.0")))
		.andExpect(jsonPath("$.[1].claimAmount", is(830)))
		.andExpect(jsonPath("$.[1].claimId", is(1000)));
		
	}	
	
	@Test
	public void test_GetClaimPaymentLines() throws Exception {
		SimpleDateFormat dateFormat = new SimpleDateFormat("MM/dd/yyyy");
		UnmatchedClaimReconciliationResponse response = new UnmatchedClaimReconciliationResponse();
		Date date = new Date();
		BigDecimal amount = new BigDecimal("1234");
		
		Set<PaymentLineItemsResponse> paymentLineItemsResponseList = new HashSet<PaymentLineItemsResponse>();
		PaymentLineItemsResponse paymentResponse = new PaymentLineItemsResponse("test123", "test1234", "test1234", amount,
				amount, 1234, dateFormat.format(date));
		paymentLineItemsResponseList.add(paymentResponse);
		response.setPaymentLineItemsResponse(paymentLineItemsResponseList);

		when(iUnmatchedReconciliationService.getPaymentLineItems("25403F0D-B34A-2204-E040-BF0A070C5552")).thenReturn(response);
		mockMvc.perform(get("/api/v1/reconciliation/unmatchedClaim/paymentLines/{feeClaimPaymentId}", "25403F0D-B34A-2204-E040-BF0A070C5552")).andDo(print())
		.andExpect(status().isOk()).andExpect(content().contentType(MediaType.APPLICATION_JSON_UTF8_VALUE))
				.andExpect(jsonPath("$.paymentLineItemsResponse[0].feeClaimPaymentLineId", is("test123")))
				.andExpect(jsonPath("$.paymentLineItemsResponse[0].lineNumber", is("test1234")))
				.andExpect(jsonPath("$.paymentLineItemsResponse[0].lineSequenceNumber", is("test1234")))
				.andExpect(jsonPath("$.paymentLineItemsResponse[0].paidUnits", is(1234)))
				.andExpect(jsonPath("$.paymentLineItemsResponse[0].serviceDate", is(dateFormat.format(date))));
	}
	
	@Test
	public void test_SetEditPaymentInformation() throws Exception {
		UnmatchedClaimReconciliationResponse unmatchedClaimReconciliationResponse = new UnmatchedClaimReconciliationResponse();
		unmatchedClaimReconciliationResponse.setResponse("Claim Reconciled successfully ");
		
		BigDecimal number = new BigDecimal(10);
		UnmatchedClaimPaymentRequest unmatchedClaimPaymentRequest = new UnmatchedClaimPaymentRequest();
		unmatchedClaimPaymentRequest.setFeeClaimPaymentLineId("3EDAF282-95A2-5095-E053-4A10BF0AD9A5");
		unmatchedClaimPaymentRequest.setClaimId(101L);
		unmatchedClaimPaymentRequest.setLineNumber("1");
		unmatchedClaimPaymentRequest.setAdjustment1(number);
		unmatchedClaimPaymentRequest.setGroup1("CO");
		unmatchedClaimPaymentRequest.setReason1("100");
		
		
		when(iUnmatchedReconciliationService.EditPaymentInformation(unmatchedClaimPaymentRequest)).thenReturn(unmatchedClaimReconciliationResponse);
		
		mockMvc.perform(post("/api/v1/reconciliation/unmatchedClaim/editpaymentInformation")
				.contentType(MediaType.APPLICATION_JSON).content(getRequestJson(unmatchedClaimPaymentRequest))).andDo(print())
		.andExpect(jsonPath("$.response", is("Claim Reconciled successfully ")));
	}
	
	private String getRequestJson(Object request) {
		String requestJson = null;
		try {
			ObjectMapper mapper = new ObjectMapper();
		    mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
		    ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
			requestJson = ow.writeValueAsString(request);
			logger.debug("requestJson - " + requestJson);
		
		} catch (JsonProcessingException e) {
			logger.error("getRequestJson JsonProcessingException -- ");
			e.printStackTrace();
		
		}
		
		return requestJson;
	}
}
