package gov.va.med.fee.controller;

import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.put;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.SerializationFeature;

import gov.va.med.fee.filter.CORSFilter;
import gov.va.med.fee.model.request.RerouteClaimRequest;
import gov.va.med.fee.model.response.ReRoutingHistory;
import gov.va.med.fee.model.response.RerouteClaimInfo;
import gov.va.med.fee.model.response.RerouteClaimResponse;
import gov.va.med.fee.service.IRerouteClaimsService;


@RunWith(MockitoJUnitRunner.class)
public class RerouteClaimsControllerTest {
	private static final Logger logger = LogManager.getLogger(RerouteClaimsControllerTest.class);
	
	private final String DEFAULT_SELECT_VALUE = "-select-";
	
	private MockMvc mockMvc;
	
	@InjectMocks
	RerouteClaimsController rerouteClaimsController;
	@Mock
	IRerouteClaimsService rerouteClaimsService;
			
	@Before
	public void init() throws Exception {
		mockMvc = MockMvcBuilders.standaloneSetup(rerouteClaimsController).addFilters(new CORSFilter()).build();
	}
	
	@Test
	public void test_getReroutes() throws Exception {
		List<Long> claimsToReroute = new ArrayList<>();
		claimsToReroute.add(528L);
		claimsToReroute.add(529L);
		
		RerouteClaimResponse response = new RerouteClaimResponse();
		List<RerouteClaimInfo> claims_to_be_rerouted = new ArrayList<>();
		RerouteClaimInfo info = new RerouteClaimInfo(529L, "ABC EFG", "AAA");
		claims_to_be_rerouted.add(info);
		info = new RerouteClaimInfo(528L, "EFG ABC", "BBB");
		claims_to_be_rerouted.add(info);
		Long[] visns = new Long[] {7l,8l,9l,10l,11l,12l,13l,14l,15l};
		String[] facilities = new String[] {"563","564","580","586","598"};
		
		Map<String, String> reasonMap = new LinkedHashMap<>();
		reasonMap.put("1", "Services were authorized by another VA facility");
		reasonMap.put("2", "Veterans Fee ID Card from another VA facility");
		reasonMap.put("3", "Clinic of Jurisdiction resides elsewhere");
		
		Map<Long, List<ReRoutingHistory>> reroute_history = new LinkedHashMap<>();
		List<ReRoutingHistory> hList = new ArrayList<>();
		ReRoutingHistory h = new ReRoutingHistory();
		h.setReason("Transfer code 9");
		h.setTransferredBy("1");
		hList.add(h);
		reroute_history.put(529L, hList);
		
		hList = new ArrayList<>();
		h = new ReRoutingHistory();
		h.setReason("Transfer code 8");
		h.setTransferredBy("1");
		hList.add(h);
		reroute_history.put(528L, hList);
		
		response.setClaims_to_be_rerouted(claims_to_be_rerouted);
		response.setVisns(Arrays.asList(visns));
		response.setFacilities(Arrays.asList(facilities));
		response.setReasons(reasonMap);
		response.setReroute_history(reroute_history);
		
		///api/v1/rerouteClaims
		when(rerouteClaimsService.populateReroutes(claimsToReroute)).thenReturn(response);
		mockMvc.perform(post("/api/v1/rerouteClaims/").contentType(MediaType.APPLICATION_JSON)
				.content(getRequestJson(claimsToReroute))).andDo(print());
		
		verify(rerouteClaimsService, times(1)).populateReroutes(claimsToReroute);
	}
	
	@Test
	public void test_processReroutes() throws Exception {
		String current_user_name = "VHAISMBAIAHBBB";
		String selectedVisnId = "16";
		String selectedFacilityId = "563";
		String selectedReason = "1";
		List<RerouteClaimInfo> claims_to_be_rerouted = new ArrayList<>();
		RerouteClaimInfo info = new RerouteClaimInfo(528, "THOMASENA BIEDERMAN", "MYRTICE");
		claims_to_be_rerouted.add(info);

		RerouteClaimRequest request = new RerouteClaimRequest();
		request.setClaims_to_be_rerouted(claims_to_be_rerouted);
		request.setSelectedFacility(selectedFacilityId);
		request.setSelectedVisn(selectedVisnId);
		request.setSelectedReason(selectedReason);
		request.setCurrent_user_name(current_user_name);

		RerouteClaimResponse response = new RerouteClaimResponse();
		response.setReroutedSuccessful("Claim(s): 528 rerouted successfully.");
		
		when(rerouteClaimsService.reroute(request)).thenReturn(response);
		
		mockMvc.perform(put("/api/v1/rerouteClaims/reroutes/").contentType(MediaType.APPLICATION_JSON)
				.content(getRequestJson(request))).andDo(print());

		verify(rerouteClaimsService, times(1)).reroute(request);
	}
	
	
	
	@Test
	public void testGetFacilities_default() throws Exception {
		String selectedVisnId = DEFAULT_SELECT_VALUE;
		List<String> facilitiesList = new ArrayList<>();
		facilitiesList.add(DEFAULT_SELECT_VALUE);
		
		RerouteClaimRequest request= new RerouteClaimRequest();
		request.setSelectedVisn(selectedVisnId);
		
		when(rerouteClaimsService.getFacilitiesList(selectedVisnId)).thenReturn(facilitiesList);
		
		mockMvc.perform(post("/api/v1/rerouteClaims/facilities/").contentType(MediaType.APPLICATION_JSON)
				.content(getRequestJson(request))).andDo(print())
				.andExpect(jsonPath("$.facilities[0]", is("-select-")));
		
		verify(rerouteClaimsService, times(1)).getFacilitiesList(selectedVisnId);
		
	}
	
	@Test
	public void testGetFacilities() throws Exception {
		String selectedVisnId = "16";
		List<String> facilitiesList = new ArrayList<>();
		facilitiesList.add("563");
		facilitiesList.add("564");
		
		RerouteClaimRequest request= new RerouteClaimRequest();
		request.setSelectedVisn(selectedVisnId);
		
		when(rerouteClaimsService.getFacilitiesList(selectedVisnId)).thenReturn(facilitiesList);
				
		mockMvc.perform(post("/api/v1/rerouteClaims/facilities/").contentType(MediaType.APPLICATION_JSON)
				.content(getRequestJson(request))).andDo(print())
				.andExpect(jsonPath("$.facilities[0]", is("563")))
				.andExpect(jsonPath("$.facilities[1]", is("564")));
				
		verify(rerouteClaimsService, times(1)).getFacilitiesList(selectedVisnId);
	}
	
	/*@Test
	public void testGetUsers_default() throws Exception {
		String selectedFacilityId = DEFAULT_SELECT_VALUE;
		List<String> userList = new ArrayList<>();
		userList.add(DEFAULT_SELECT_VALUE);
		
		RerouteClaimRequest request= new RerouteClaimRequest();
		request.setSelectedFacility(selectedFacilityId);
		
		when(rerouteClaimsService.getUserList(selectedFacilityId)).thenReturn(userList);
		
		mockMvc.perform(post("/api/v1/rerouteClaims/users/").contentType(MediaType.APPLICATION_JSON)
				.content(getRequestJson(request))).andDo(print())
				.andExpect(jsonPath("$.names[0]", is("-select-")));
		
		verify(rerouteClaimsService, times(1)).getUserList(selectedFacilityId);
		verifyNoMoreInteractions(rerouteClaimsService);
	}
	
	@Test
	public void testGetUsers() throws Exception {
		String selectedFacilityId = "563";
		List<String> userList = new ArrayList<>();
		userList.add("ABC");
		userList.add("EFG");
		
		RerouteClaimRequest request= new RerouteClaimRequest();
		request.setSelectedFacility(selectedFacilityId);
		
		when(rerouteClaimsService.getUserList(selectedFacilityId)).thenReturn(userList);
				
		mockMvc.perform(post("/api/v1/rerouteClaims/users/").contentType(MediaType.APPLICATION_JSON)
				.content(getRequestJson(request))).andDo(print())
				.andExpect(jsonPath("$.names[0]", is("ABC")))
				.andExpect(jsonPath("$.names[1]", is("EFG")));
				
		verify(rerouteClaimsService, times(1)).getUserList(selectedFacilityId);
		verifyNoMoreInteractions(rerouteClaimsService);
	}*/
	
	
	
	private String getRequestJson(Object request) {
		String requestJson = null;
		try {
			ObjectMapper mapper = new ObjectMapper();
		    mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
		    ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
			requestJson = ow.writeValueAsString(request);
			System.out.println("requestJson - " + requestJson);
		
		} catch (JsonProcessingException e) {
			logger.error("testRequestAddUser JsonProcessingException -- ");
			e.printStackTrace();
		
		}
		
		return requestJson;
	}
	
}