package gov.va.med.fee.controller;

import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.when;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.content;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.LinkedList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.runners.MockitoJUnitRunner;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.SerializationFeature;

import gov.va.med.fee.filter.CORSFilter;
import gov.va.med.fee.model.request.RejectedClaimRequest;
import gov.va.med.fee.model.response.AdjustmentReasonResponse;
import gov.va.med.fee.model.response.ClaimsToDisapprove;
import gov.va.med.fee.model.response.DisapproveClaims;
import gov.va.med.fee.model.response.DisapproveClaimsPage;
import gov.va.med.fee.model.response.DisapproveClaimsReason;
import gov.va.med.fee.model.response.SuccessResponse;
import gov.va.med.fee.service.IDisapproveService;


@RunWith(MockitoJUnitRunner.class)
public class DisapproveClaimControllerTest {
	private static final Logger logger = LogManager.getLogger(DisapproveClaimControllerTest.class);
	
	private MockMvc mockMvc;
	
	@InjectMocks
	private DisapproveClaimController disapproveClaimController;
	
	@Mock
	IDisapproveService disapproveService;
		
	
	@Before
	public void init() throws Exception {
		logger.debug("init");
		MockitoAnnotations.initMocks(this);
		mockMvc = MockMvcBuilders.standaloneSetup(disapproveClaimController).addFilters(new CORSFilter()).build();
	}
	
	@Test
	public void test_disapproveClaims() throws Exception {
		List<Long> claimIdReq = new ArrayList<>();
		claimIdReq.add(123L);
		DisapproveClaims claims = new DisapproveClaims("VHAISMBAIAHBBB", "123","Rejection Rejection",claimIdReq);
				
		List<ClaimsToDisapprove> claimsToDisapprove = new ArrayList<>();
		ClaimsToDisapprove disapprove = new ClaimsToDisapprove(123L, "Susannah Messana", "Sharika", "","");
		claimsToDisapprove.add(disapprove);
		
		List<DisapproveClaimsReason> disapproveClaimsReasons = new ArrayList<>();
		DisapproveClaimsReason disapproveClaimsReason = new DisapproveClaimsReason("123", "Test123");
		disapproveClaimsReasons.add(disapproveClaimsReason);
		List<SuccessResponse> disapproveClaims = new ArrayList<>();
		SuccessResponse reason = new SuccessResponse("200","Claim number 123 have been Disapproved");
		disapproveClaims.add(reason);
		
		DisapproveClaimsPage page = new DisapproveClaimsPage();
		page.setDisapproveClaims(disapproveClaims);
		page.setClaimsToDisapprove(claimsToDisapprove);
		page.setDisapproveClaimsReasons(disapproveClaimsReasons);
		
		when(disapproveService.disapproveClaim(claims)).thenReturn(page);
		
		ObjectMapper mapper = new ObjectMapper();
	    mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
	    ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
		String requestJson = ow.writeValueAsString(claims);		
		
		mockMvc.perform(post("/api/v1/disapproveClaim").contentType(MediaType.APPLICATION_JSON).content(requestJson)).andDo(print())
				.andExpect(status().isOk())
				.andExpect(jsonPath("$.claimsToDisapprove[0].claimId", is(123)))
				.andExpect(jsonPath("$.claimsToDisapprove[0].veteranName", is("Susannah Messana")))
				.andExpect(jsonPath("$.claimsToDisapprove[0].serviceProvider", is("Sharika")))
				.andExpect(jsonPath("$.disapproveClaims[0].status", is("200")))
				.andExpect(jsonPath("$.disapproveClaims[0].message", is("Claim number 123 have been Disapproved")))
				.andExpect(jsonPath("$.disapproveClaimsReasons[0].code", is("123")))
				.andExpect(jsonPath("$.disapproveClaimsReasons[0].claimReasons", is("Test123")));
		
		verify(disapproveService, times(1)).disapproveClaim(claims);
		verifyNoMoreInteractions(disapproveService);
		
	}	

	@Test
	public void test_getClaimsToDisapprove() throws Exception {
		List<Long> claimIdReq = new ArrayList<>();
		claimIdReq.add(102L);
		claimIdReq.add(120L);
		DisapproveClaimsPage claimsToDisapprove = new DisapproveClaimsPage();
		
		List<ClaimsToDisapprove> listClaimsToDisapprove = new ArrayList<>();
		ClaimsToDisapprove claim = new ClaimsToDisapprove(120L, "Susannah Messana", "Sharika", "","");
		listClaimsToDisapprove.add(claim);
		
		claim = new ClaimsToDisapprove();
		claim.setClaimId(102L);
		claim.setMessage("The following claim cannot be disapproved because it is completed, partially reconciled, or rejected: 102");
		listClaimsToDisapprove.add(claim);
		claimsToDisapprove.setClaimsToDisapprove(listClaimsToDisapprove);
		
		when(disapproveService.getClaimsToDisapprove(claimIdReq)).thenReturn(claimsToDisapprove);
		
		ObjectMapper mapper = new ObjectMapper();
	    mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
	    ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
		String requestJson = ow.writeValueAsString(claimIdReq);
		
		mockMvc.perform(post("/api/v1//disapproveClaim/claims").contentType(MediaType.APPLICATION_JSON).content(requestJson)).andDo(print())
		.andExpect(status().isOk())
		.andExpect(jsonPath("$.claimsToDisapprove[0].claimId", is(120)))
		.andExpect(jsonPath("$.claimsToDisapprove[0].veteranName", is("Susannah Messana")))
		.andExpect(jsonPath("$.claimsToDisapprove[0].serviceProvider", is("Sharika")))
		.andExpect(jsonPath("$.claimsToDisapprove[1].claimId", is(102)))
		.andExpect(jsonPath("$.claimsToDisapprove[1].message", is("The following claim cannot be disapproved because it is completed, partially reconciled, or rejected: 102")));
	}
	
	@Test
	public void test_populateAdjustmentReasons_success() throws Exception {
				
		List<DisapproveClaimsReason> mockReasonCodes = new LinkedList<DisapproveClaimsReason>();
		
		final Date dateCreated = new GregorianCalendar(1990, Calendar.APRIL, 30).getTime();
		
		DisapproveClaimsPage reasonCodesfrmServiceCall = new DisapproveClaimsPage();
		
		List<AdjustmentReasonResponse> adjustmentReasonResponse = new LinkedList<AdjustmentReasonResponse>();
		AdjustmentReasonResponse adjustmentReasonResponse2 = new AdjustmentReasonResponse("123", "Test123", 'Y', "", dateCreated, 'Y');
		adjustmentReasonResponse.add(adjustmentReasonResponse2);
		
		DisapproveClaimsReason disapproveClaimsReasonMock = new DisapproveClaimsReason("123", "Test123");
		mockReasonCodes.add(disapproveClaimsReasonMock);
		
		reasonCodesfrmServiceCall.setDisapproveClaimsReasons(mockReasonCodes);
				
		when(disapproveService.populateAdjustmentReasons()).thenReturn(reasonCodesfrmServiceCall);
		
		mockMvc.perform(get("/api/v1/disapproveClaim/reasons")).andDo(print()).andExpect(status().isOk())
		.andExpect(content().contentType(MediaType.APPLICATION_JSON_UTF8_VALUE))
		.andExpect(jsonPath("$.disapproveClaimsReasons[0].code", is("123")))
		.andExpect(jsonPath("$.disapproveClaimsReasons[0].claimReasons", is("Test123")));
		
		verify(disapproveService, times(1)).populateAdjustmentReasons();
		verifyNoMoreInteractions(disapproveService);
		
	}
	
	@Test
	public void test_okDisapprovals() throws Exception {
		List<RejectedClaimRequest> rejectedClaimRequestList = new ArrayList<RejectedClaimRequest>();
		RejectedClaimRequest rejectedClaimRequest = new RejectedClaimRequest();
		rejectedClaimRequest.setClaimId(999997L);
		rejectedClaimRequest.setReviewersComments("Test1 Comments");
		rejectedClaimRequestList.add(rejectedClaimRequest);
		
		ObjectMapper mapper = new ObjectMapper();
	    mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
	    ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
		String requestJson = ow.writeValueAsString(rejectedClaimRequestList);		
		
		mockMvc.perform(post("/api/v1/okDisapproval/TestUser").contentType(MediaType.APPLICATION_JSON).content(requestJson)).andDo(print())
		.andExpect(status().isOk())
		.andExpect(content().contentType(MediaType.APPLICATION_JSON_UTF8_VALUE));
	}
	
	@Test
	public void test_reverseDisapprovals() throws Exception {
		List<RejectedClaimRequest> rejectedClaimRequestList = new ArrayList<RejectedClaimRequest>();
		RejectedClaimRequest rejectedClaimRequest = new RejectedClaimRequest();
		rejectedClaimRequest.setClaimId(999998L);
		rejectedClaimRequest.setReviewersComments("Test1 Comments");
		rejectedClaimRequestList.add(rejectedClaimRequest);
		
		ObjectMapper mapper = new ObjectMapper();
	    mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
	    ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
		String requestJson = ow.writeValueAsString(rejectedClaimRequestList);		
		
		mockMvc.perform(post("/api/v1/reverseDisapproval/TestUser").contentType(MediaType.APPLICATION_JSON).content(requestJson)).andDo(print())
		.andExpect(status().isOk())
		.andExpect(content().contentType(MediaType.APPLICATION_JSON_UTF8_VALUE));
	}
}
