package gov.va.med.fee.controller;

import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.when;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.content;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

import java.math.BigDecimal;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.runners.MockitoJUnitRunner;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

import gov.va.med.fee.constants.ClaimDetailsConstants;
import gov.va.med.fee.exceptions.GenericException;
import gov.va.med.fee.filter.CORSFilter;
import gov.va.med.fee.model.response.AdditionalInfoResponse;
import gov.va.med.fee.model.response.ClaimDetailsResponse;
import gov.va.med.fee.model.response.ClaimRejectionHistoryResponse;
import gov.va.med.fee.model.response.DiagnosisCodeResponse;
import gov.va.med.fee.model.response.DiagnosisCodes;
import gov.va.med.fee.model.response.LineItemResponse;
import gov.va.med.fee.model.response.OhiInfoResponse;
import gov.va.med.fee.model.response.ProviderInfoResponse;
import gov.va.med.fee.model.response.ReRoutingHistory;
import gov.va.med.fee.model.response.VeteranInfoResponse;
import gov.va.med.fee.service.IClaimDetailsService;

/**
 * 
 * @author DNS   SUNH
 * @version 1.0 This Class Test the Claim Controller.
 * 
 * @author Vamsi Krishna Gangarapu Added {@link DiagnosisCodeResponse} to the
 *         test case
 * @version 1.1
 *
 */

@RunWith(MockitoJUnitRunner.class)
public class ClaimDetailsControllerTest {
	private MockMvc mockMvc;

	@Mock
	private IClaimDetailsService claimDetailsServiceMock;

	@InjectMocks
	private ClaimDetailsController claimDetailsController;

	SimpleDateFormat formatter = new SimpleDateFormat("MM/dd/yyyy");

	/**
	 * @throws Exception
	 */
	@Before
	public void init() throws Exception {
		MockitoAnnotations.initMocks(this);
		mockMvc = MockMvcBuilders.standaloneSetup(claimDetailsController).addFilters(new CORSFilter()).build();
	}

	/**
	 * @throws Exception
	 */
	@Test
	public void testGetInstClaimDetailsByIndex_success() throws Exception {

		// BigDecimal number = new BigDecimal(1234);
		Date dischargeDate = new GregorianCalendar(2009, Calendar.JANUARY, 22).getTime();
		Date admissionDate = new GregorianCalendar(1990, Calendar.APRIL, 30).getTime();

		VeteranInfoResponse personInfoResponse = new VeteranInfoResponse(1L, "Doe", "John", "", "Sarno Road",
				"Sarno Plaza", "Melbourne", "Florida", "32935", "Male", "05-JUN-45", "");
		ProviderInfoResponse billingProviderResponse = new ProviderInfoResponse("Test", "FTC", "Sarno", "Melbourne",
				"FL", "32935", "9999999999", "1234567", "Team-9 Backend", "123456789", "john.doe@DOMAIN", "1237654");
		ProviderInfoResponse payToProviderResponse = new ProviderInfoResponse("Test", "FTC", "Sarno", "Melbourne", "FL",
				"32935", "9999999999", "12345678", "Team-9 Backend", "12345678", "john.doe@DOMAIN", "1237655");
		ProviderInfoResponse serviceProviderResponse = new ProviderInfoResponse("Test", "FTC", "Sarno", "Melbourne",
				"FL", "32935", "9999999990", "12345677", "Team-9 Backend", "12345677", "john.doe@DOMAIN", "1237656");

		List<DiagnosisCodes> diagnosisCodeList = new ArrayList<>();
		diagnosisCodeList.add(new DiagnosisCodes(1, "D1", "DQ1", "A", "diagnosisDescription 1"));
		diagnosisCodeList.add(new DiagnosisCodes(2, "D2", "DQ2", "J", "diagnosisDescription 2"));
		// DiagnosisCodes diagnosisCodes = new DiagnosisCod(diagnosisCodeList);
		List<ClaimRejectionHistoryResponse> claimRejectionHistoryResponseList = new ArrayList<>();
		claimRejectionHistoryResponseList.add(new ClaimRejectionHistoryResponse(admissionDate, "Janiya Howe",
				"DESCRIPTION", "ZXC", "Janiya Howe", "ZXC", "ZXC"));

		AdditionalInfoResponse additionalInfo = new AdditionalInfoResponse("1000", "Test", "1000", "Test", "Simply",
				"$10.00", "06/09/2017", "$10.00", "12348ABC", admissionDate.toString(), dischargeDate.toString(), "28",
				"11", "MICHAEL KUHU DAVID MR");
		List<LineItemResponse> lineItemsInfo = getInstitutionalLineItemResponseList();

		List<ReRoutingHistory> reRoutingHistoryList = new ArrayList<ReRoutingHistory>();
		List<OhiInfoResponse> ohiInfo = new ArrayList<OhiInfoResponse>();

		StringBuilder transferredFrom = new StringBuilder();
		transferredFrom.append("VISN:01/FAC:station2/USER:VHAIS_USER2");
		StringBuilder transferredTo = new StringBuilder();
		transferredTo.append("VISN:00/FAC:100/USER:VHAIS_USER1");

		reRoutingHistoryList.add(
				new ReRoutingHistory("07/07/2017", "Reason1", transferredTo, transferredFrom, "transferred", "CCSE"));

		ClaimDetailsResponse claimResponse = new ClaimDetailsResponse(1, "2.0.1.STD", "INPROCESS", "111111",
				personInfoResponse, serviceProviderResponse, billingProviderResponse, payToProviderResponse,
				additionalInfo, diagnosisCodeList, lineItemsInfo, reRoutingHistoryList,
				claimRejectionHistoryResponseList, ohiInfo);

		when(claimDetailsServiceMock.getClaimDetailsByClaimIndex((long) 1)).thenReturn(claimResponse);
		mockMvc.perform(get("/api/v1/claim/{claimIndex}", (long) 1)).andDo(print()).andExpect(status().isOk())
				.andExpect(content().contentType(MediaType.APPLICATION_JSON_UTF8_VALUE))
				.andExpect(jsonPath("$.claimId", is(1))).andExpect(jsonPath("$.patientControlNumber", is("2.0.1.STD")))
				.andExpect(jsonPath("$.claimStatus", is("INPROCESS")))
				.andExpect(jsonPath("$.preAuthorization", is("111111")))
				.andExpect(jsonPath("$.veteranInfo.firstName", is("John")))
				.andExpect(jsonPath("$.veteranInfo.middleName", is("")))
				.andExpect(jsonPath("$.veteranInfo.addressLine1", is("Sarno Road")))
				.andExpect(jsonPath("$.veteranInfo.addressLine2", is("Sarno Plaza")))
				.andExpect(jsonPath("$.veteranInfo.city", is("Melbourne")))
				.andExpect(jsonPath("$.veteranInfo.state", is("Florida")))
				.andExpect(jsonPath("$.veteranInfo.zipCode", is("32935")))
				.andExpect(jsonPath("$.veteranInfo.gender", is("Male")))
				.andExpect(jsonPath("$.billingProviderInfo.lastName", is("Test")))
				.andExpect(jsonPath("$.billingProviderInfo.addressLine1", is("FTC")))
				.andExpect(jsonPath("$.billingProviderInfo.addressLine2", is("Sarno")))
				.andExpect(jsonPath("$.billingProviderInfo.city", is("Melbourne")))
				.andExpect(jsonPath("$.billingProviderInfo.state", is("FL")))
				.andExpect(jsonPath("$.billingProviderInfo.zipCode", is("32935")))
				.andExpect(jsonPath("$.billingProviderInfo.phoneNumber", is("9999999999")))
				.andExpect(jsonPath("$.billingProviderInfo.taxId", is("1234567")))
				.andExpect(jsonPath("$.billingProviderInfo.contactName", is("Team-9 Backend")))
				.andExpect(jsonPath("$.billingProviderInfo.faxNumber", is("123456789")))
				.andExpect(jsonPath("$.billingProviderInfo.email", is("john.doe@DOMAIN")))
				.andExpect(jsonPath("$.billingProviderInfo.providerNpi", is("1237654")))
				.andExpect(jsonPath("$.payToProviderInfo.lastName", is("Test")))
				.andExpect(jsonPath("$.payToProviderInfo.addressLine1", is("FTC")))
				.andExpect(jsonPath("$.payToProviderInfo.addressLine2", is("Sarno")))
				.andExpect(jsonPath("$.payToProviderInfo.city", is("Melbourne")))
				.andExpect(jsonPath("$.payToProviderInfo.state", is("FL")))
				.andExpect(jsonPath("$.payToProviderInfo.zipCode", is("32935")))
				.andExpect(jsonPath("$.payToProviderInfo.phoneNumber", is("9999999999")))
				.andExpect(jsonPath("$.payToProviderInfo.taxId", is("12345678")))
				.andExpect(jsonPath("$.payToProviderInfo.contactName", is("Team-9 Backend")))
				.andExpect(jsonPath("$.payToProviderInfo.faxNumber", is("12345678")))
				.andExpect(jsonPath("$.payToProviderInfo.email", is("john.doe@DOMAIN")))
				.andExpect(jsonPath("$.payToProviderInfo.providerNpi", is("1237655")))
				.andExpect(jsonPath("$.serviceProviderInfo.lastName", is("Test")))
				.andExpect(jsonPath("$.serviceProviderInfo.addressLine1", is("FTC")))
				.andExpect(jsonPath("$.serviceProviderInfo.addressLine2", is("Sarno")))
				.andExpect(jsonPath("$.serviceProviderInfo.city", is("Melbourne")))
				.andExpect(jsonPath("$.serviceProviderInfo.state", is("FL")))
				.andExpect(jsonPath("$.serviceProviderInfo.zipCode", is("32935")))
				.andExpect(jsonPath("$.serviceProviderInfo.phoneNumber", is("9999999990")))
				.andExpect(jsonPath("$.serviceProviderInfo.taxId", is("12345677")))
				.andExpect(jsonPath("$.serviceProviderInfo.contactName", is("Team-9 Backend")))
				.andExpect(jsonPath("$.serviceProviderInfo.faxNumber", is("12345677")))
				.andExpect(jsonPath("$.serviceProviderInfo.email", is("john.doe@DOMAIN")))
				.andExpect(jsonPath("$.serviceProviderInfo.providerNpi", is("1237656")))
				.andExpect(jsonPath("$.diagnosisCodes[0].sequenceNumber", is(1)))
				.andExpect(jsonPath("$.diagnosisCodes[0].diagnosisCode", is("D1")))
				.andExpect(jsonPath("$.diagnosisCodes[0].diagnosisCodeQualifier", is("DQ1")))
				.andExpect(jsonPath("$.diagnosisCodes[0].poaIndicator", is("A")))
				.andExpect(jsonPath("$.diagnosisCodes[0].diagnosisDescription", is("diagnosisDescription 1")))
				.andExpect(jsonPath("$.diagnosisCodes[1].sequenceNumber", is(2)))
				.andExpect(jsonPath("$.diagnosisCodes[1].diagnosisCode", is("D2")))
				.andExpect(jsonPath("$.diagnosisCodes[1].diagnosisCodeQualifier", is("DQ2")))
				.andExpect(jsonPath("$.diagnosisCodes[1].poaIndicator", is("J")))
				.andExpect(jsonPath("$.diagnosisCodes[1].diagnosisDescription", is("diagnosisDescription 2")))
				.andExpect(jsonPath("$.lineItemsInfo[0].line_id", is(1234567)))
				.andExpect(jsonPath("$.lineItemsInfo[0].medical_code", is("Test103")))
				.andExpect(jsonPath("$.lineItemsInfo[0].revenue_code", is("Test103")))
				.andExpect(jsonPath("$.lineItemsInfo[0].service_from_date", is(formatter.format(new Timestamp(new Date().getTime())))))
				.andExpect(jsonPath("$.lineItemsInfo[0].service_to_date", is(formatter.format(new Timestamp(new Date().getTime())))))
				.andExpect(jsonPath("$.lineItemsInfo[0].billed_amount", is("$100.00")))
				.andExpect(jsonPath("$.lineItemsInfo[0].line_status", is("test103")))
				.andExpect(jsonPath("$.lineItemsInfo[0].treasury_payment_date", is(formatter.format(new Timestamp(new Date().getTime())))))
				.andExpect(jsonPath("$.lineItemsInfo[0].adjustment_amount", is("$0.00")))
				.andExpect(jsonPath("$.lineItemsInfo[0].paid_amount", is("$10.00")))
				.andExpect(jsonPath("$.lineItemsInfo[0].check_eft_trn", is("1234567")))
				.andExpect(jsonPath("$.lineItemsInfo[0].reason_code", is("97")))
				.andExpect(jsonPath("$.additionalInfo.otherInsuranceNumber_SecondaryPayer", is("1000")))
				.andExpect(jsonPath("$.additionalInfo.otherInsurancePlan_SecondaryPayer", is("Test")))
				.andExpect(jsonPath("$.additionalInfo.otherInsuranceNumber_TertiaryPayer", is("1000")))
				.andExpect(jsonPath("$.additionalInfo.otherInsurancePlan_TertiaryPayer", is("Test")))
				.andExpect(jsonPath("$.additionalInfo.ambulanceTransportReason", is("Simply")))
				.andExpect(jsonPath("$.additionalInfo.totalClaimAmountPaid", is("$10.00")))
				.andExpect(jsonPath("$.additionalInfo.treasuryPaymentDate_Pd", is("06/09/2017")))
				.andExpect(jsonPath("$.additionalInfo.totalPaidAmtPerPD", is("$10.00")))
				.andExpect(jsonPath("$.additionalInfo.check_EFT_TRN", is("12348ABC")))
				.andExpect(jsonPath("$.additionalInfo.admissionDate", is(admissionDate.toString())))
				.andExpect(jsonPath("$.additionalInfo.dischargeDate", is(dischargeDate.toString())))
				.andExpect(jsonPath("$.additionalInfo.coveredDays", is("28")))
				.andExpect(jsonPath("$.additionalInfo.providerDrg", is("11")))
				.andExpect(jsonPath("$.additionalInfo.attendingPhysician", is("MICHAEL KUHU DAVID MR")))
				.andExpect(jsonPath("$.rejectionHistory[0].rejectedBy", is("Janiya Howe")))
				.andExpect(jsonPath("$.rejectionHistory[0].reason", is("DESCRIPTION")))
				.andExpect(jsonPath("$.rejectionHistory[0].explanation", is("ZXC")))
				.andExpect(jsonPath("$.rejectionHistory[0].reviewedBy", is("Janiya Howe")))
				.andExpect(jsonPath("$.rejectionHistory[0].outcome", is("ZXC")))
				.andExpect(jsonPath("$.rejectionHistory[0].reviewerComments", is("ZXC")))
				.andExpect(jsonPath("$.reRoutingHistory[0].date", is("07/07/2017")))
				.andExpect(jsonPath("$.reRoutingHistory[0].reason", is("Reason1")))
				.andExpect(
						jsonPath("$.reRoutingHistory[0].transferredFrom", is("VISN:01/FAC:station2/USER:VHAIS_USER2")))
				.andExpect(jsonPath("$.reRoutingHistory[0].explanation", is("transferred")))
				.andExpect(jsonPath("$.reRoutingHistory[0].transferredTo", is("VISN:00/FAC:100/USER:VHAIS_USER1")));

		verify(claimDetailsServiceMock, times(1)).getClaimDetailsByClaimIndex((long) 1);
		verifyNoMoreInteractions(claimDetailsServiceMock);
	}

	/**
	 * @throws Exception
	 */
	@Test(expected = Exception.class)
	public void testGetClaimDetailsByIndex_Fail() throws Exception {
		long claimIndex = 1L;
		when(claimDetailsServiceMock.getClaimDetailsByClaimIndex(claimIndex)).thenThrow(new GenericException(ClaimDetailsConstants.CLAIM_NOT_FOUND, "No claim found with claim id " + claimIndex,
				HttpStatus.NOT_FOUND));

		mockMvc.perform(get("/api/v1/claim/{claimIndex}", claimIndex)).andExpect(status().isNotFound());
		verifyNoMoreInteractions(claimDetailsController);

	}

	/**
	 * @return lineItemResponses
	 */
	private List<LineItemResponse> getInstitutionalLineItemResponseList() {
		List<LineItemResponse> lineItemResponses = new ArrayList<>();
				
		BigDecimal claimProcedureSequence = new BigDecimal(1234567);
		Long claimProcedureSeq = claimProcedureSequence.longValue();
		String procCode = "Test103";
		String revenueCode = "Test103";
		String serviceFromDate = formatter.format(new Timestamp(new Date().getTime())) ;
		String serviceToDate = formatter.format(new Timestamp(new Date().getTime())) ;
		String billedAmount = "$100.00";
		String procCodeDesc = "test103";
		String diagPointer = "test103";
		Long billedUnits = null;
		String modifier1 = "test";
		String modifier2 = null;
		String modifier3 = null;
		String modifier4 = null;
		Timestamp paymentDate = new Timestamp(new Date().getTime());
		String paymentTrackingnumber = "1234567";
		String toothCode = null;
		String toothSurface = null;
		String toothSurfaceCd2 = null;
		String toothSurfaceCd3 = null;
		String toothSurfaceCd4 = null;
		String toothSurfaceCd5 = null;
		//String oralCavityCd = null;
		String lineStatus = "test103";
		
		String lineItemTreasuryPaymentDateFormatted = formatter.format(paymentDate);
		String lineItemPaidAmountFormatted = "$10.00"; 
		String lineItemCheckNumber = paymentTrackingnumber; 
		String lineItemAdjAmts = "$0.00";
		String lineItemCodes = "97";
		String lineItemCagcCodes = "ABC";
		String lineItemRarcCodes = "EFG";
		
		LineItemResponse newReqponse = new LineItemResponse(claimProcedureSeq, toothCode, toothSurface, toothSurfaceCd2,
				toothSurfaceCd3, toothSurfaceCd4, toothSurfaceCd5, procCode, revenueCode,
				procCodeDesc, diagPointer, serviceFromDate, serviceToDate,
				billedAmount, billedUnits , modifier1, modifier2, modifier3,
				modifier4, lineStatus, lineItemTreasuryPaymentDateFormatted,
				lineItemPaidAmountFormatted, lineItemCheckNumber, lineItemAdjAmts,
				lineItemCodes, lineItemCagcCodes, lineItemRarcCodes);
		
		
		lineItemResponses.add(newReqponse);
		
		return lineItemResponses;
	}
}
