package gov.va.med.fee.controller;

import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.content;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

import java.math.BigDecimal;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.runners.MockitoJUnitRunner;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.SerializationFeature;

import gov.va.med.fee.constants.ClaimDetailsConstants;
import gov.va.med.fee.model.request.ClaimRequest;
import gov.va.med.fee.model.response.ClaimLineItem;
import gov.va.med.fee.model.response.ClaimLineItems;
import gov.va.med.fee.model.response.ClaimResponse;
import gov.va.med.fee.service.IClaimService;

/**
 * 
 * @author Muneshwar Baiah
 *
 */

@RunWith(MockitoJUnitRunner.class)
public class ClaimControllerTest {
	private MockMvc mockMvc;
	public static final MediaType APPLICATION_JSON_UTF8 = new MediaType(MediaType.APPLICATION_JSON.getType(), MediaType.APPLICATION_JSON.getSubtype(), Charset.forName("utf8"));

	@Mock
	private IClaimService claimService;
	
	@InjectMocks
	private ClaimController claimController;

	@Before
	public void init() throws Exception {
		MockitoAnnotations.initMocks(this);
		//mockMvc = MockMvcBuilders.standaloneSetup(claimController).addFilters(new CORSFilter()).build();
		mockMvc = MockMvcBuilders.standaloneSetup(claimController).build();
	}
	
	/**
	 * @throws Exception
	 */
	@Test
	public void getAwaitingProcessingClaims_success() throws Exception {
		
		// Setup
		ClaimRequest claimRequest = new ClaimRequest();
		List<String> activeStations = new ArrayList<String>();
		List<Long> visns = new ArrayList<Long>();
		List<String> privacyRestrictions = new ArrayList<String>();
		
		activeStations.add("1");
		visns.add(1L);
		privacyRestrictions.add("ADMIN");
	
		claimRequest.setActiveStations(activeStations);
		claimRequest.setVisns(visns);
		claimRequest.setPrivacyRestrictions(privacyRestrictions);
	
		ObjectMapper mapper = new ObjectMapper();
	    mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
	    ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
	    String requestJson=ow.writeValueAsString(claimRequest);
	    	
		BigDecimal claimAmount = new BigDecimal("1000.00");
		ClaimResponse awaitingProcessing = new ClaimResponse(1L, "INPROGRESS","INST", 'Y',
				"MICHAEL", "044444444", "05/06/2017", claimAmount, 
				"MEMORIAL", 1000L, "GENERAL HOSP", 101L,"09/09/2017",100L);
		List<ClaimResponse> awaitingProcessingClaims = new ArrayList<ClaimResponse>();
		awaitingProcessingClaims.add(awaitingProcessing);
				
		when(claimService.getClaimsByStatus(claimRequest,ClaimDetailsConstants.AWAITING_STATUS_CODE)).thenReturn(awaitingProcessingClaims);
		mockMvc.perform(post("/api/v1/awaitingProcessing").contentType(MediaType.APPLICATION_JSON).content(requestJson)).andDo(print())
		.andExpect(status().isOk())
		.andExpect(content().contentType(MediaType.APPLICATION_JSON_UTF8_VALUE));
				
	}
	
	
	/**
	 * @throws Exception
	 */
	@Test
	public void updateAwaitingProcessingClaimsStatus_success() throws Exception {
		
		// Setup
		ClaimRequest claimRequest = new ClaimRequest();
		List<String> activeStations = new ArrayList<String>();
		List<Long> visns = new ArrayList<Long>();
		List<String> privacyRestrictions = new ArrayList<String>();
		
		activeStations.add("1");
		visns.add(1L);
		privacyRestrictions.add("ADMIN");
	
		claimRequest.setActiveStations(activeStations);
		claimRequest.setVisns(visns);
		claimRequest.setPrivacyRestrictions(privacyRestrictions);
	
		ObjectMapper mapper = new ObjectMapper();
	    mapper.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
	    ObjectWriter ow = mapper.writer().withDefaultPrettyPrinter();
	    String requestJson=ow.writeValueAsString(claimRequest);
	    	
		BigDecimal claimAmount = new BigDecimal("1000.00");
		ClaimResponse awaitingProcessing = new ClaimResponse(1L, "INPROGRESS","INST", 'Y',
				"MICHAEL", "044444444", "05/06/2017", claimAmount, 
				"MEMORIAL", 1000L, "GENERAL HOSP", 101L,"09/09/2017",100L);
		List<ClaimResponse> awaitingProcessingClaims = new ArrayList<ClaimResponse>();
		awaitingProcessingClaims.add(awaitingProcessing);
				
		when(claimService.updateClaimsStatus(claimRequest,"INPROCESS")).thenReturn(awaitingProcessingClaims);
		mockMvc.perform(post("/api/v1/awaitingProcessing").contentType(MediaType.APPLICATION_JSON).content(requestJson)).andDo(print())
		.andExpect(status().isOk())
		.andExpect(content().contentType(MediaType.APPLICATION_JSON_UTF8_VALUE));
				
	}
	
	@Test
	public void test_getClaimLineItems() throws Exception {
		List<ClaimLineItem> claimLineItemList = new ArrayList<>();		
		
		long claimProcId = 104; 
	    BigDecimal billedAmount = new BigDecimal(101);
	    long billedUnits = 100;
	    String description = "26";
	    String serviceDateFrom ="06/16/2017";
	    String serviceDateTo ="06/16/2017";
	    long claimIndex = 103;
	    long claimProcedureSeq = 1234567 ;
	    String procCode = "26";
	    String procCodeDesc = "26";
	    String revenueCodeDesc = "26";
	    String placeOfServiceCd = "test2";
	    String modifier1 = "test";
	    String modifier2 = "test";
	    String modifier3 = "test";
	    String modifier4 = "test";
	    BigDecimal denyChargeAmt = new BigDecimal(231);
	    String emergencyInd = "Y";
	    String diagPointer = "test103";
	    String epsdtInd = "y";
	    String familyPlan = "y";
	    String oralCavityCd = "y";	    
	    String prescriptionNumber = "test103";
	    String lineItemControlCode = "test103";	    
	    String heroFlag = "Y";
	    String lineStatus = "REJECTED";
	    String createdBy = "VA Test";
	    
	    ClaimLineItem claimLineItem = new ClaimLineItem(claimProcId, billedAmount, billedUnits, description,
				serviceDateFrom, serviceDateTo, claimIndex, claimProcedureSeq, procCode,
				procCodeDesc, revenueCodeDesc, placeOfServiceCd, modifier1, modifier2,
				modifier3, modifier4, denyChargeAmt, emergencyInd, diagPointer,
				epsdtInd, familyPlan, oralCavityCd, null, prescriptionNumber,
				lineItemControlCode, null, heroFlag, lineStatus, createdBy, null); 
	    
	    claimLineItemList.add(claimLineItem);
	    
	    ClaimLineItems claimLineItems = new ClaimLineItems();
	    claimLineItems.setClaimLineItems(claimLineItemList);
	    
		when(claimService.getClaimLineItems(103L)).thenReturn(claimLineItems);
		
		mockMvc.perform(get("/api/v1/claim/lineItems/{claimId}", 103L)).andDo(print()).andExpect(status().isOk())
			.andExpect(content().contentType(MediaType.APPLICATION_JSON_UTF8_VALUE))
			.andExpect(jsonPath("$.claimLineItems[0].claimProcId", is(104)))
			.andExpect(jsonPath("$.claimLineItems[0].claimIndex", is(103)))
			.andExpect(jsonPath("$.claimLineItems[0].billedAmount", is(101)))
			.andExpect(jsonPath("$.claimLineItems[0].serviceDateFrom", is("06/16/2017")))
			.andExpect(jsonPath("$.claimLineItems[0].serviceDateTo", is("06/16/2017")))
			.andExpect(jsonPath("$.claimLineItems[0].prescriptionNumber", is("test103")))
			.andExpect(jsonPath("$.claimLineItems[0].lineItemControlCode", is("test103")))
			.andExpect(jsonPath("$.claimLineItems[0].heroFlag", is("Y")))
			.andExpect(jsonPath("$.claimLineItems[0].lineStatus", is("REJECTED")))
			.andExpect(jsonPath("$.claimLineItems[0].emergencyInd", is("Y")))
			.andExpect(jsonPath("$.claimLineItems[0].revenueCodeDesc", is("26")))
			.andExpect(jsonPath("$.claimLineItems[0].procCode", is("26")))
			.andExpect(jsonPath("$.claimLineItems[0].procCodeDesc", is("26")))
			.andExpect(jsonPath("$.claimLineItems[0].modifier1", is("test")))
			.andExpect(jsonPath("$.claimLineItems[0].modifier2", is("test")))
			.andExpect(jsonPath("$.claimLineItems[0].modifier3", is("test")));
		
		verify(claimService, times(1)).getClaimLineItems(103L);
	}
	
	
	/**
	 * @param obj
	 * @return jsonContent
	 */
	public static String asJsonString(final Object obj) {
	    try {
	        final ObjectMapper mapper = new ObjectMapper();
	        final String jsonContent = mapper.writeValueAsString(obj);
	        return jsonContent;
	    } catch (Exception e) {
	        throw new RuntimeException(e);
	    }
	} 
	
}
