package gov.va.med.fee.service.impl;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Set;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import gov.va.med.domain.fee.Claim;
import gov.va.med.domain.fee.ContactInfo;
import gov.va.med.domain.fee.VaFacility;
import gov.va.med.domain.fee.VaFacilityType;
import gov.va.med.domain.fee.Visn;
import gov.va.med.domain.fee.ZipCode;
import gov.va.med.fee.constants.ClaimDetailsConstants;
import gov.va.med.fee.dao.IClaimRepository;
import gov.va.med.fee.dao.IContactInfoRepository;
import gov.va.med.fee.dao.IVaFacilityRepository;
import gov.va.med.fee.dao.IVaFacilityRepositoryTemplate;
import gov.va.med.fee.dao.IVaFacilityTypeRepository;
import gov.va.med.fee.dao.IVisnRepository;
import gov.va.med.fee.dao.IZipCodeRepository;
import gov.va.med.fee.exceptions.GenericException;
import gov.va.med.fee.model.request.ContactInfoRequest;
import gov.va.med.fee.model.request.VaFacilityRequest;
import gov.va.med.fee.model.request.VaFacilitySearchRequest;
import gov.va.med.fee.model.request.ZipCodeRequest;
import gov.va.med.fee.model.response.GenericResponse;
import gov.va.med.fee.model.response.VaFacilities;
import gov.va.med.fee.model.response.VaFacilityResponse;
import gov.va.med.fee.model.response.VaFacilityTypeResponse;
import gov.va.med.fee.model.response.VaFacilityVisnResponse;
import gov.va.med.fee.service.IVaFacilityService;

/**
 * @author David dragon
 * @author Vamsi Krishna Gangarapu
 * @author Ya-Huei
 * 
 *         This page gives REST calls for Station pages with Add, Modify and
 *         delete information for Station, Contact info and Zip code
 *
 */

@Service
public class VaFacilityServiceImpl extends BaseService implements IVaFacilityService {

	private static final Logger logger = LogManager.getLogger(VaFacilityServiceImpl.class);

	@Autowired
	IVaFacilityRepositoryTemplate vaFacilityRepoTemplate;

	@Autowired
	IVaFacilityRepository iVaFacilityRepository;

	@Autowired
	IContactInfoRepository iContactInfoRepository;

	@Autowired
	IZipCodeRepository iZipCodeRepository;

	@Autowired
	IClaimRepository iClaimRepository;

	@Autowired
	IVisnRepository iVisnRepository;

	@Autowired
	IVaFacilityTypeRepository iVaFacilityTypeRepository;

	SimpleDateFormat sdf = new SimpleDateFormat("MM/dd/yyyy hh:mm:ss a");
	Date date = null;
	BigDecimal agedDef = null;

	private static final String[] appUserRoles = { "FEE_SUPERVISOR", "FEE_VISN_PGM_MANAGER", "ADMIN" };

	@Override
	public int addVaFacility(VaFacilityRequest request) throws GenericException {
		int rowCount = 0;

		// check if app user can add Va Facility
		if (!hasRole(request.getAppUserName(), appUserRoles)) {
			logger.error("VaFacilityServiceImpl.addVaFacility() : invalid_input_error for role ");
			throw new GenericException("Add Va Facility error", "Invalid role", HttpStatus.BAD_REQUEST);
		}

		rowCount = vaFacilityRepoTemplate.insertNewVaFacility(request);

		if (rowCount == 0) {
			throw new GenericException("Add a new facility error", "Unable to add a new facility",
					HttpStatus.BAD_REQUEST);
		}

		return rowCount;
	}

	@Override
	public List<VaFacilityVisnResponse> getAllVisns() throws GenericException {

		List<VaFacilityVisnResponse> vaFacilityVisn = new ArrayList<VaFacilityVisnResponse>();

		vaFacilityVisn = vaFacilityRepoTemplate.getAllVisnsForFormLoad();

		return vaFacilityVisn;
	}

	@Override
	public List<VaFacilityTypeResponse> getAllVaFacilityType() throws GenericException {

		List<VaFacilityTypeResponse> vaFacilityType = new ArrayList<VaFacilityTypeResponse>();
		vaFacilityType = vaFacilityRepoTemplate.getAllVaFacilityTypeForFormLoad();

		return vaFacilityType;
	}

	/**
	 * @author Vamsi Krishna Gangarapu
	 * 
	 */
	@Override
	public boolean modifyStation(VaFacilityRequest request) throws GenericException {
		boolean successResponse = false;
		VaFacility vaFacility = null;
		VaFacility vaFacilityReturnType = null;
		Visn visn = null;
		VaFacilityType vaFacilityType = null;
		date = new Date();

		// check if app user can add Va Facility
		if (!hasRole(request.getAppUserName(), appUserRoles)) {
			logger.error("VaFacilityServiceImpl.modifyStation() : invalid_input_error for role ");
			throw new GenericException("Modify Va Facility error", "Invalid role", HttpStatus.BAD_REQUEST);
		}

		try {
			logger.info("VaFacilityServiceImpl.modifyStation() : modifiedDate is :" + date.toString());
			vaFacility = iVaFacilityRepository.findVaFacilityByVaFacilityCd(request.getVaFacilityCd());
			visn = iVisnRepository.findOne(request.getVisnIdCd());
			vaFacilityType = iVaFacilityTypeRepository.findOne(request.getVaTypeCd());
			if (vaFacility != null && visn != null && vaFacilityType != null) {
				vaFacility.setVisn(visn);
				vaFacility.setVaFacilityType(vaFacilityType);
				// vaFacility.setVaFacilityCd(request.getVaFacilityCd());
				vaFacility.setVaFacilityName(request.getVaFacilityName());
				vaFacility.setShortName(request.getShortName());
				vaFacility.setAgedDefinition(request.getAgedDefinition());
				vaFacility.setReroutingAllowed(request.isReroutingAllowed());
				vaFacility.setParentVaFacilityCd(request.getParentVaFacilityCd());
				vaFacility.setModifiedBy(request.getAppUserName());
				vaFacility.setDateModified(date);
				vaFacilityReturnType = iVaFacilityRepository.save(vaFacility);
			} else {
				throw new GenericException("Modiy a facility error", "Unable to modify a facility",
						HttpStatus.BAD_REQUEST);
			}

			if (vaFacilityReturnType != null) {
				successResponse = true;
			}
		} catch (Exception e) {
			logger.error("UnmatchedReconciliationServiceImpl.getUnmatchedClaimsInformation() : data_access_error " + e);
			throw new GenericException(ClaimDetailsConstants.INTERNAL_SERVER_ERROR, e.getMessage(),
					HttpStatus.INTERNAL_SERVER_ERROR);
		}
		return successResponse;
	}

	/**
	 * @author Vamsi Krishna Gangarapu
	 * 
	 */
	@Override
	public GenericResponse getSearchVisn(VaFacilitySearchRequest searchRequest) throws GenericException {
		GenericResponse visnSearchResponse = null;
		List<VaFacilityResponse> vaFacilityResponse = new ArrayList<VaFacilityResponse>();
		int pageNumber = 0;
		try {
			if (searchRequest != null) {
				if ((searchRequest.getUserRole().equalsIgnoreCase("ADMIN")
						|| searchRequest.getUserRole().equalsIgnoreCase("FEE_VISN_PGM_MANAGER")
						|| searchRequest.getUserRole().equalsIgnoreCase("FEE_SUPERVISOR"))) {
					logger.info(" getSearchVisn() : getting the Visn Search list");
					Long resultsCount = vaFacilityRepoTemplate.getSearchResultsCount(searchRequest);

					if (resultsCount <= (searchRequest.getPageSize() * searchRequest.getPageNumber())) {
						Integer resultCountIntVal = resultsCount.intValue();
						float pageNumberFloat = (float) resultCountIntVal / (float) searchRequest.getPageSize();
						pageNumber = (int) Math.ceil(pageNumberFloat);
						// pageSize = resultCountIntVal % pageSize;
						logger.debug("getSearchVisn() : pageNumber is :" + pageNumber + " and pageSize is :"
								+ searchRequest.getPageSize());
					}

					if (resultsCount != 0 && resultsCount > 0) {
						logger.debug(
								"vaFacilityRepoTemplate.getSearchResults() : calling the respository to query the database");

						if (pageNumber == 0) {
							vaFacilityResponse = vaFacilityRepoTemplate.getSearchResults(searchRequest);
							visnSearchResponse = new GenericResponse(searchRequest.getPageNumber(),
									searchRequest.getPageSize(), searchRequest.getSortColumn(), resultsCount);
						} else {
							searchRequest.setPageNumber(pageNumber);
							vaFacilityResponse = vaFacilityRepoTemplate.getSearchResults(searchRequest);
							visnSearchResponse = new GenericResponse(pageNumber, searchRequest.getPageSize(),
									searchRequest.getSortColumn(), resultsCount);
						}

						visnSearchResponse.setResponse(vaFacilityResponse);
						if (vaFacilityResponse.isEmpty()) {
							logger.info("VaFacilityServiceImpl.getSearchVisn() : visn_search_information_not_found "
									+ vaFacilityResponse.toString());

							logger.error("VaFacilityServiceImpl.getSearchVisn() : visn_search_information_not_found for"
									+ searchRequest.toString());
						}
					} else {
						logger.error("VaFacilityServiceImpl.getSearchVisn() : invalid_input_error resulted in no rows "
								+ resultsCount);
						throw new GenericException(ClaimDetailsConstants.NO_CONTENT, "Invalid input resulted in 0 rows",
								HttpStatus.NO_CONTENT);
					}
				} else {
					logger.error("VaFacilityServiceImpl.getSearchVisn() : invalid_input_error for role "
							+ searchRequest.getUserRole());
					throw new GenericException(ClaimDetailsConstants.INVALID_REQUEST, "Invalid role",
							HttpStatus.BAD_REQUEST);
				}
			} else {
				logger.error("VaFacilityServiceImpl.getSearchVisn() : BAD_REQUEST for " + searchRequest);
				throw new GenericException(ClaimDetailsConstants.NO_CONTENT, "The post request is empty ",
						HttpStatus.NO_CONTENT);
			}
		} catch (Exception e) {
			logger.error("VaFacilityServiceImpl.getSearchVisn() : data_access_error for " + searchRequest.toString()
					+ " " + e);
			throw new GenericException(ClaimDetailsConstants.INTERNAL_SERVER_ERROR, e.getMessage(),
					HttpStatus.INTERNAL_SERVER_ERROR);
		}

		return visnSearchResponse;
	}

	@Override
	public boolean checkVaFacility(String stationName) throws GenericException {

		boolean value = false;
		try {
			int count = vaFacilityRepoTemplate.checkVaFacility(stationName);
			if (count > 0) {
				value = true;
			}
		} catch (Exception e) {
			logger.error("VaFacilityServiceImpl.checkVaFacility() : data_access_error for " + value + " " + e);
			throw new GenericException(ClaimDetailsConstants.INTERNAL_SERVER_ERROR, e.getMessage(),
					HttpStatus.INTERNAL_SERVER_ERROR);
		}

		return value;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.fee.service.IContactInfoService#postContactInfo(gov.va.med.fee
	 * .model.ContactInfoRequest)
	 */
	@Override
	public boolean postContactInfo(ContactInfoRequest contactInfo) throws GenericException {
		Boolean result = false;
		ContactInfo resultContactInfoModel = null;

		if (!hasRole(contactInfo.getAppUserName(), appUserRoles)) {
			logger.error(
					"VaFacilityServiceImpl.postContactInfo() : invalid_input_error for role " + contactInfo.toString());
			throw new GenericException(ClaimDetailsConstants.BAD_REQUEST, "Invalid role", HttpStatus.BAD_REQUEST);
		}

		VaFacility vaFacility = iVaFacilityRepository.findVaFacilityByVaFacilityCd(contactInfo.getVaFacilityCd());
		if (vaFacility == null) {
			throw new GenericException("Post Contact Info error",
					"Va Facility '" + contactInfo.getVaFacilityCd() + "' is not found", HttpStatus.BAD_REQUEST);
		}

		try {
			logger.info(" VaFacilityServiceImpl.postContactInfo() : posting the contactInfo");
			ContactInfo contactInfoModel = new ContactInfo();
			Date date = new Date();

			contactInfoModel.setVaFacility(vaFacility);

			if (contactInfo.getContactInfoId() != null) {
				resultContactInfoModel = iContactInfoRepository.findOneByVaFacilityName(contactInfo.getContactInfoId(),
						contactInfoModel.getVaFacility().getVaFacilityCd());
			}

			// Update old
			if (resultContactInfoModel != null) {
				resultContactInfoModel.setContactName(contactInfo.getContactName());
				resultContactInfoModel.setContactType(contactInfo.getContactType());
				resultContactInfoModel.setDateCreated(date);
				resultContactInfoModel.setAddress1(contactInfo.getAddress1());
				resultContactInfoModel.setAddress2(contactInfo.getAddress2());
				resultContactInfoModel.setCity(contactInfo.getCity());
				resultContactInfoModel.setZip(contactInfo.getZip());
				resultContactInfoModel.setEmail(contactInfo.getEmail());
				resultContactInfoModel.setPhone(contactInfo.getPhone());
				resultContactInfoModel.setFax(contactInfo.getFax());
				resultContactInfoModel.setCreatedBy(contactInfo.getAppUserName());
				logger.info("vaFacilityRepoTemplate.postContactInfo() : to modify the information");
				logger.debug(
						"vaFacilityRepoTemplate.postContactInfo() : calling the respository to query the database");
				iContactInfoRepository.save(resultContactInfoModel);

			} else {
				// Create new
				contactInfoModel.setContactName(contactInfo.getContactName());
				contactInfoModel.setContactType(contactInfo.getContactType());
				contactInfoModel.setDateCreated(date);
				contactInfoModel.setAddress1(contactInfo.getAddress1());
				contactInfoModel.setAddress2(contactInfo.getAddress2());
				contactInfoModel.setCity(contactInfo.getCity());
				contactInfoModel.setZip(contactInfo.getZip());
				contactInfoModel.setEmail(contactInfo.getEmail());
				contactInfoModel.setPhone(contactInfo.getPhone());
				contactInfoModel.setFax(contactInfo.getFax());
				contactInfoModel.setCreatedBy(contactInfo.getAppUserName());
				logger.info("vaFacilityRepoTemplate.postContactInfo() : to create new information");
				logger.debug(
						"vaFacilityRepoTemplate.postContactInfo() : calling the respository to query the database");
				iContactInfoRepository.saveAndFlush(contactInfoModel);
			}
			result = Boolean.TRUE;
			/*
			 * if (resultContactInfoModel != null) { result = true; } else {
			 * logger.error(
			 * "ContactInfoServiceImpl.postContactInfo() : bad_request for " +
			 * contactInfo.toString()); throw new
			 * GenericException(ClaimDetailsConstants.INTERNAL_SERVER_ERROR,
			 * "Unable to add/modify the contact Info",
			 * HttpStatus.INTERNAL_SERVER_ERROR); }
			 */
		} catch (Exception e) {
			logger.error("ContactInfoServiceImpl.postContactInfo() : bad_request for " + contactInfo.toString());
			throw new GenericException(ClaimDetailsConstants.INTERNAL_SERVER_ERROR,
					"Unable to add/modify the contact Info", HttpStatus.INTERNAL_SERVER_ERROR);
		}

		return result;
	}

	@Override
	public List<ContactInfoRequest> getContactInfo(String station) throws GenericException {
		List<ContactInfo> contactInfoList = new ArrayList<ContactInfo>();
		ContactInfoRequest contactInfoRequest = null;
		List<ContactInfoRequest> contactInfo = new ArrayList<ContactInfoRequest>();
		try {
			if (station != null) {
				logger.info("VaFacilityServiceImpl.getContactInfo() : to create new information");
				logger.debug("VaFacilityServiceImpl.getContactInfo() : calling the respository to query the database");
				contactInfoList = iContactInfoRepository.findContactInfo(station);
				if (!contactInfoList.isEmpty()) {
					for (ContactInfo contactInfoLoop : contactInfoList) {
						contactInfoRequest = new ContactInfoRequest();
						contactInfoRequest.setAddress1(contactInfoLoop.getAddress1());
						contactInfoRequest.setAddress2(contactInfoLoop.getAddress2());
						contactInfoRequest.setCity(contactInfoLoop.getCity());
						contactInfoRequest.setContactName(contactInfoLoop.getContactName());
						contactInfoRequest.setContactType(contactInfoLoop.getContactType());
						contactInfoRequest.setDateCreated(sdf.format(contactInfoLoop.getDateCreated()));
						contactInfoRequest.setZip(contactInfoLoop.getZip());
						contactInfoRequest.setEmail(contactInfoLoop.getEmail());
						contactInfoRequest.setPhone(contactInfoLoop.getPhone());
						contactInfoRequest.setFax(contactInfoLoop.getFax());
						contactInfoRequest.setCreatedBy(contactInfoLoop.getCreatedBy());
						contactInfoRequest.setVaFacilityCd(station);
						contactInfo.add(contactInfoRequest);
					}
					// jsonObject.put("contactInfo", contactInfo);
				} /*
					 * else { logger.
					 * error("VaFacilityServiceImpl.getContactInfo() : invalid_input_error for "
					 * + contactInfoList); throw new
					 * GenericException(ClaimDetailsConstants.INVALID_REQUEST,
					 * "Station doesn't exist",
					 * HttpStatus.INTERNAL_SERVER_ERROR); }
					 */
			} else {
				logger.error("ContactInfoServiceImpl.getContactInfo() : invalid_request for " + station);
				throw new GenericException(ClaimDetailsConstants.BAD_REQUEST, "station is Null",
						HttpStatus.BAD_REQUEST);
			}

		} catch (Exception e) {
			logger.error("ContactInfoServiceImpl.getContactInfo() : internal_server_error for " + station + " " + e);
			throw new GenericException(ClaimDetailsConstants.INTERNAL_SERVER_ERROR, e.getMessage(),
					HttpStatus.INTERNAL_SERVER_ERROR);
		}
		return contactInfo;
	}

	@Override
	public VaFacilityResponse getVaFacilityInfo(String station) throws GenericException {
		VaFacilityResponse vaFacilityResponse = null;
		try {
			if (station != null) {
				logger.info("VaFacilityServiceImpl.getContactInfo() : to create new information");
				logger.debug("VaFacilityServiceImpl.getContactInfo() : calling the respository to query the database");
				VaFacility vaFacilitydata = iVaFacilityRepository.findVaFacilityByVaFacilityCd(station);

				if (vaFacilitydata != null) {
					logger.info("iVaFacilityRepository.findVaFacilityByVaFacilityCd() : information from db "
							+ vaFacilitydata.toString());
					if(vaFacilitydata.getAgedDefinition() != null) {
						agedDef = new BigDecimal(vaFacilitydata.getAgedDefinition());
					}
					
					vaFacilityResponse = new VaFacilityResponse();
					vaFacilityResponse.setStation(vaFacilitydata.getVaFacilityCd());
					vaFacilityResponse.setShortName(vaFacilitydata.getShortName());
					vaFacilityResponse.setStationName(vaFacilitydata.getVaFacilityName());
					vaFacilityResponse.setStationType(vaFacilitydata.getVaFacilityType().getVaTypeCd());
					vaFacilityResponse.setVisnNumber(String.valueOf(vaFacilitydata.getVisn().getVisnIdCd()));
					vaFacilityResponse.setAllowReroute(vaFacilitydata.isReroutingAllowed());
					vaFacilityResponse.setAgedDef(agedDef);
					vaFacilityResponse.setParentStation(vaFacilitydata.getParentVaFacilityCd());
					vaFacilityResponse.setCreatedBy(vaFacilitydata.getCreatedBy());
					vaFacilityResponse.setDateCreated(sdf.format(vaFacilitydata.getDateCreated()));

					vaFacilityResponse.setContactInfoRequest(getContactInfo(station));
					vaFacilityResponse.setZipcodes(getZipCodeList(station));

				} else {
					logger.error("VaFacilityServiceImpl.getVaFacilityInfo() : data_access_error for " + station);
					throw new GenericException(ClaimDetailsConstants.DATA_ACCESS_ERROR, "Station doesn't exist",
							HttpStatus.INTERNAL_SERVER_ERROR);
				}
			} else {
				logger.error("VaFacilityServiceImpl.getVaFacilityInfo() : bad_request for " + station);
				throw new GenericException(ClaimDetailsConstants.BAD_REQUEST, "station is Null",
						HttpStatus.BAD_REQUEST);
			}
		} catch (Exception e) {
			logger.error("VaFacilityServiceImpl.getVaFacilityInfo() : internal_server_error for " + station + " " + e);
			throw new GenericException(ClaimDetailsConstants.INTERNAL_SERVER_ERROR, e.getMessage(),
					HttpStatus.INTERNAL_SERVER_ERROR);
		}
		return vaFacilityResponse;
	}

	// Station Maintenance - Add ZIP Code
	@Override
	public Boolean addZipCode(ZipCodeRequest request) throws GenericException {
		Boolean result = Boolean.FALSE;

		if (!hasRole(request.getAppUserName(), appUserRoles)) {
			logger.error("VaFacilityServiceImpl.addZipCode() : invalid_input_error for role ");
			throw new GenericException("Add ZIP Code error", "Invalid role", HttpStatus.BAD_REQUEST);
		}

		// check if there is a duplicate zip code
		ZipCode zipcode = iZipCodeRepository.findZipCodeByZipcode(request.getZip_code());
		VaFacility vaFacility = iVaFacilityRepository.findVaFacilityByVaFacilityCd(request.getVaFacilityCd());

		if (vaFacility == null) {
			throw new GenericException("Add ZIP Code error",
					"Va Facility '" + request.getVaFacilityCd() + "' is not found", HttpStatus.BAD_REQUEST);
		}

		if (zipcode == null) {
			zipcode = new ZipCode();
			zipcode.setVaFacility(vaFacility);
			zipcode.setCreatedBy(request.getAppUserName());
			zipcode.setDateCreated(new Date());
			zipcode.setActiveFlag(Boolean.TRUE.equals(request.getActive()) ? 'Y' : 'N');
			zipcode.setFipsCountyCode(request.getFips_county_code());
			zipcode.setFipsStateCode(request.getFips_state_code());
			zipcode.setUrbanIndicator(request.getUrban_indicator());
			zipcode.setZipCode(request.getZip_code());
			iZipCodeRepository.saveAndFlush(zipcode);
			result = Boolean.TRUE;
		} else {
			throw new GenericException("Add ZIP Code error",
					"The zip code '" + request.getZip_code() + "' already exists", HttpStatus.BAD_REQUEST);
		}

		return result;
	}

	// Station Maintenance - Edit ZIP Code
	@Override
	public Boolean editZipCode(ZipCodeRequest request) throws GenericException {
		Boolean result = Boolean.FALSE;

		// check if app user can edit ZIP Code
		if (!hasRole(request.getAppUserName(), appUserRoles)) {
			logger.error("VaFacilityServiceImpl.editZipCode() : invalid_input_error for role ");
			throw new GenericException("Edit ZIP Code error", "Invalid role", HttpStatus.BAD_REQUEST);
		}

		// check if there is the zipcode entity exist
		ZipCode entity = iZipCodeRepository.findZipCodeByZipcode(request.getZip_code());
		if (entity == null) {
			throw new GenericException("Edit ZIP Code error",
					"The zip code '" + request.getZip_code() + "' is not found", HttpStatus.BAD_REQUEST);
		} else {
			try {
				entity.setFipsCountyCode(request.getFips_county_code());
				entity.setFipsStateCode(request.getFips_state_code());
				entity.setUrbanIndicator(request.getUrban_indicator());
				entity.setActiveFlag(Boolean.TRUE.equals(request.getActive()) ? 'Y' : 'N');

				iZipCodeRepository.saveAndFlush(entity);
				result = Boolean.TRUE;
			} catch (Exception e) {
				throw new GenericException("Edit ZIP Code error", e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);
			}
		}

		return result;
	}

	@Override
	public Boolean removeZipCode(ZipCodeRequest request) throws GenericException {
		Boolean result = Boolean.FALSE;

		// check if app user can delete ZIP Code
		if (!hasRole(request.getAppUserName(), appUserRoles)) {
			logger.error("VaFacilityServiceImpl.deleteZipCode() : invalid_input_error for role ");
			throw new GenericException("Remove ZIP Code error", "Invalid role", HttpStatus.BAD_REQUEST);
		}

		ZipCode entity = iZipCodeRepository.findZipCodeByZipcode(request.getZip_code());
		if (entity == null) {
			throw new GenericException("Remove ZIP Code error ",
					"The zip code '" + request.getZip_code() + "' is not found", HttpStatus.BAD_REQUEST);

		} else {
			try {
				iZipCodeRepository.delete(entity);
				result = Boolean.TRUE;
			} catch (Exception e) {
				throw new GenericException("Remove ZIP Code error", e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);
			}
		}
		return result;
	}

	@Override
	public List<ZipCodeRequest> getZipCodeList(String vaFacilityCd) throws GenericException {
		List<ZipCodeRequest> zipCodeList = new ArrayList<>();

		try {
			List<ZipCode> list = iZipCodeRepository.findZipCodeByFacilityCd(vaFacilityCd);
			for (ZipCode z : list) {
				ZipCodeRequest request = new ZipCodeRequest(z.getVaFacility().getVaFacilityCd(), z.getZipCode(),
						z.getUrbanIndicator(), z.getFipsCountyCode(), z.getFipsStateCode(),
						('Y' == z.getActiveFlag()) ? true : false, z.getCreatedBy(), sdf.format(z.getDateCreated()));

				zipCodeList.add(request);
			}
		} catch (Exception e) {
			throw new GenericException("Get ZIP Code error", e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);
		}
		return zipCodeList;
	}

	@Override
	public boolean deleteContactInfo(Long contactInfoId) throws GenericException {
		boolean success = false;
		ContactInfo contactInfo = null;
		try {
			if (contactInfoId != null) {
				logger.info("VaFacilityServiceImpl.getContactInfo() : to create new information");
				logger.debug("VaFacilityServiceImpl.getContactInfo() : calling the respository to query the database");
				contactInfo = iContactInfoRepository.findOneBycontactInfoId(contactInfoId);
				if (contactInfo != null) {
					iContactInfoRepository.delete(contactInfo);
					success = true;
				} else {
					logger.error("VaFacilityServiceImpl.getContactInfo() : data_access_error for " + contactInfoId);
					throw new GenericException(ClaimDetailsConstants.DATA_ACCESS_ERROR, "contactInfoId doesn't exist",
							HttpStatus.INTERNAL_SERVER_ERROR);
				}
			} else {
				logger.error("VaFacilityServiceImpl.getContactInfo() : bad_request for " + contactInfoId);
				throw new GenericException(ClaimDetailsConstants.BAD_REQUEST, "contactInfoId doesn't exist",
						HttpStatus.BAD_REQUEST);
			}
		} catch (Exception e) {
			logger.error(
					"VaFacilityServiceImpl.deleteContactInfo() : internal_server_error for " + contactInfoId + " " + e);
			throw new GenericException(ClaimDetailsConstants.INTERNAL_SERVER_ERROR, e.getMessage(),
					HttpStatus.INTERNAL_SERVER_ERROR);
		}
		return success;
	}

	@Override
	public boolean checkZipExists(String zipCode) throws GenericException {
		boolean value = false;
		try {
			logger.info("VaFacilityServiceImpl.checkZipExists() : to get the information");
			logger.debug("VaFacilityServiceImpl.checkZipExists() : calling the respository to query the database");
			int count = iZipCodeRepository.checkZipCodeExists(zipCode);
			if (count > 0) {
				value = true;
			}
		} catch (Exception e) {
			logger.error("VaFacilityServiceImpl.checkZipExists() : bad_request for " + value + " " + e);
			throw new GenericException(ClaimDetailsConstants.INTERNAL_SERVER_ERROR, e.getMessage(),
					HttpStatus.INTERNAL_SERVER_ERROR);
		}
		return value;
	}

	@Override
	public VaFacilityRequest getStationInfo(String stationName) throws GenericException {
		VaFacilityRequest vaFacilityRequest = null;
		VaFacility vaFacility = null;
		try {
			if (stationName != null) {
				logger.info("VaFacilityServiceImpl.getStationInfo() : to get the information");
				logger.debug("VaFacilityServiceImpl.getStationInfo() : calling the respository to query the database");
				vaFacility = iVaFacilityRepository.findVaFacilityByStation(stationName);
				if (vaFacility != null) {
					vaFacilityRequest = new VaFacilityRequest("", vaFacility.getVaFacilityCd(),
							vaFacility.getVisn().getVisnIdCd(), vaFacility.getVaFacilityType().getVaTypeCd(),
							vaFacility.getParentVaFacilityCd(), vaFacility.getVaFacilityName(),
							vaFacility.getShortName(), vaFacility.getAgedDefinition(), vaFacility.isReroutingAllowed(),
							sdf.format(vaFacility.getDateCreated()), vaFacility.getCreatedBy());
				} else {
					logger.error("VaFacilityServiceImpl.getStationInfo() : data_access_error for " + vaFacility);
					throw new GenericException(ClaimDetailsConstants.DATA_ACCESS_ERROR,
							"Couldn't find information with the given station number",
							HttpStatus.INTERNAL_SERVER_ERROR);
				}
			} else {
				logger.error("VaFacilityServiceImpl.getStationInfo() : bad_request_error for " + stationName);
				throw new GenericException(ClaimDetailsConstants.BAD_REQUEST, "stationName doesn't exist",
						HttpStatus.BAD_REQUEST);
			}
		} catch (Exception e) {
			logger.error("VaFacilityServiceImpl.getStationInfo() : data_access_error for " + stationName + " " + e);
			throw new GenericException(ClaimDetailsConstants.INTERNAL_SERVER_ERROR, e.getMessage(),
					HttpStatus.INTERNAL_SERVER_ERROR);
		}

		return vaFacilityRequest;
	}

	@Override
	public boolean deleteStationInfo(String stationName) throws GenericException {
		VaFacility vaFacility = null;
		boolean successResponse = false;
		try {
			if (stationName != null) {
				logger.info("VaFacilityServiceImpl.getStationInfo() : to get the information");
				logger.debug("VaFacilityServiceImpl.getStationInfo() : calling the respository to query the database");
				vaFacility = iVaFacilityRepository.findVisnBystationNumber(stationName);
				if (vaFacility != null) {
					if (vaFacility.getClaims().size() > 0) {
						changeClaims(vaFacility.getClaims());
					}
				}
				vaFacility = iVaFacilityRepository.findVaFacilityByContactInfo(stationName);
				if (vaFacility != null) {
					iContactInfoRepository.delete(vaFacility.getContactInfos());
				}
				vaFacility = iVaFacilityRepository.findVaFacilityByZipCode(stationName);
				if (vaFacility != null) {
					iZipCodeRepository.delete(vaFacility.getZipCodes());
				}
				vaFacility = iVaFacilityRepository.findVaFacilityByVaFacilityCd(stationName);
				if (vaFacility != null) {
					iVaFacilityRepository.delete(vaFacility);
					successResponse = true;
				}
			} else {
				logger.error("iVaFacilityRepository.deleteStationInfo() : bad_request_error for " + stationName);
				throw new GenericException(ClaimDetailsConstants.BAD_REQUEST,
						"Couldn't find information with the given station number", HttpStatus.BAD_REQUEST);
			}
		} catch (Exception e) {
			logger.error("VaFacilityServiceImpl.deleteStationInfo() : data_access_error for " + stationName + " " + e);
			throw new GenericException(ClaimDetailsConstants.INTERNAL_SERVER_ERROR, e.getMessage(),
					HttpStatus.INTERNAL_SERVER_ERROR);
		}
		return successResponse;
	}

	private void changeClaims(Set<Claim> claims) {
		for (Claim c : claims) {
			if (c.getVaFacility() != null) {
				c.setVaFacility(null);
				iClaimRepository.save(c);
			}
		}
	}

	@SuppressWarnings("unused")
	@Override
	public VaFacilityVisnResponse getVisnInfo(Long visnNumber) throws GenericException {
		List<VaFacilityResponse> vaFacilityResponseList = new ArrayList<VaFacilityResponse>();
		VaFacilityResponse vaFacilityResponse = null;
		VaFacilityVisnResponse vaFacilityVisnResponse = null;
		try {
			logger.info("VaFacilityServiceImpl.getVisnInfo() : to get the information");
			logger.debug("VaFacilityServiceImpl.getVisnInfo() : calling the respository to query the database");
			Visn visn = iVisnRepository.findAllVisn(visnNumber);
			logger.info("VaFacilityServiceImpl.getVisnInfo() : information we got" + visn.toString());
			if (visn != null) {
				vaFacilityVisnResponse = new VaFacilityVisnResponse();
				vaFacilityVisnResponse.setVisnIdCd(visn.getVisnIdCd());
				vaFacilityVisnResponse.setDescription(visn.getDescription());
				vaFacilityVisnResponse.setCreatedBy(visn.getCreatedBy());
				vaFacilityVisnResponse.setDateCreated(sdf.format(visn.getDateCreated()));
				if (visn.getVaFacilities().size() > 0) {
					for (VaFacility vaFacility : visn.getVaFacilities()) {
						if(vaFacility.getAgedDefinition() != null) {
							agedDef = new BigDecimal(vaFacility.getAgedDefinition());
						}
						vaFacilityResponse = new VaFacilityResponse();
						vaFacilityResponse.setStation(vaFacility.getVaFacilityCd());
						vaFacilityResponse.setShortName(vaFacility.getShortName());
						vaFacilityResponse.setStationName(vaFacility.getVaFacilityName());
						vaFacilityResponse.setAgedDef(agedDef);
						vaFacilityResponse.setParentStation(vaFacility.getParentVaFacilityCd());
						vaFacilityResponse.setAllowReroute(vaFacility.isReroutingAllowed());
						vaFacilityResponse.setCreatedBy(vaFacility.getCreatedBy());
						vaFacilityResponse.setDateCreated(sdf.format(vaFacility.getDateCreated()));
						vaFacilityResponse.setStationType(vaFacility.getVaFacilityType().getVaTypeCd());
						vaFacilityResponse.setVisnNumber(visnNumber.toString());
						vaFacilityResponseList.add(vaFacilityResponse);
					}
					vaFacilityVisnResponse.setVaFacilityResponse(vaFacilityResponseList);
				}
			} else {
				logger.error("iVaFacilityRepository.getVisnInfo() : bad_request_error for " + visnNumber);
				throw new GenericException(ClaimDetailsConstants.BAD_REQUEST,
						"Couldn't find information with the given visn number", HttpStatus.BAD_REQUEST);
			}
		} catch (Exception e) {
			logger.error("VaFacilityServiceImpl.getVisnInfo() : data_access_error for " + visnNumber + " " + e);
			throw new GenericException(ClaimDetailsConstants.INTERNAL_SERVER_ERROR, e.getMessage(),
					HttpStatus.INTERNAL_SERVER_ERROR);
		}
		return vaFacilityVisnResponse;
	}

	@SuppressWarnings("unused")
	@Override
	public VaFacilityTypeResponse getStationTypeInfo(String typeNumber) throws GenericException {
		List<VaFacilityResponse> vaFacilityResponseList = new ArrayList<VaFacilityResponse>();
		VaFacilityResponse vaFacilityResponse = null;
		VaFacilityTypeResponse vaFacilityTypeResponse = null;
		try {
			logger.info("VaFacilityServiceImpl.getStationTypeInfo() : to get the information");
			logger.debug("VaFacilityServiceImpl.getStationTypeInfo() : calling the respository to query the database");
			VaFacilityType vaFacilityType = iVaFacilityTypeRepository.findVaFacilityTypeInfo(typeNumber);
			logger.info("VaFacilityServiceImpl.getVisnInfo() : information we got" + vaFacilityType.toString());
			if (vaFacilityType != null) {
				vaFacilityTypeResponse = new VaFacilityTypeResponse();
				vaFacilityTypeResponse.setVaTypeCd(vaFacilityType.getVaTypeCd());
				vaFacilityTypeResponse.setDescription(vaFacilityType.getDescription());
				vaFacilityTypeResponse.setCreatedBy(vaFacilityType.getCreatedBy());
				vaFacilityTypeResponse.setDateCreated(sdf.format(vaFacilityType.getDateCreated()));
				if (vaFacilityType.getVaFacilities().size() > 0) {
					for (VaFacility vaFacility : vaFacilityType.getVaFacilities()) {
						if(vaFacility.getAgedDefinition() != null) {
							agedDef = new BigDecimal(vaFacility.getAgedDefinition());
						}
						vaFacilityResponse = new VaFacilityResponse();
						vaFacilityResponse = new VaFacilityResponse();
						vaFacilityResponse.setStation(vaFacility.getVaFacilityCd());
						vaFacilityResponse.setShortName(vaFacility.getShortName());
						vaFacilityResponse.setStationName(vaFacility.getVaFacilityName());
						vaFacilityResponse.setAgedDef(agedDef);
						vaFacilityResponse.setParentStation(vaFacility.getParentVaFacilityCd());
						vaFacilityResponse.setAllowReroute(vaFacility.isReroutingAllowed());
						vaFacilityResponse.setCreatedBy(vaFacility.getCreatedBy());
						vaFacilityResponse.setDateCreated(sdf.format(vaFacility.getDateCreated()));
						vaFacilityResponse.setStationType(vaFacility.getVaFacilityType().getVaTypeCd());
						vaFacilityResponse.setVisnNumber(String.valueOf(vaFacility.getVisn().getVisnIdCd()));
						vaFacilityResponseList.add(vaFacilityResponse);
					}
					vaFacilityTypeResponse.setVaFacilityResponse(vaFacilityResponseList);
				}
			} else {
				logger.error("iVaFacilityRepository.getStationTypeInfo() : bad_request_error for " + typeNumber);
				throw new GenericException(ClaimDetailsConstants.BAD_REQUEST,
						"Couldn't find information with the given station type", HttpStatus.BAD_REQUEST);
			}
		} catch (Exception e) {
			logger.error("VaFacilityServiceImpl.getStationTypeInfo() : data_access_error for " + typeNumber + " " + e);
			throw new GenericException(ClaimDetailsConstants.INTERNAL_SERVER_ERROR, e.getMessage(),
					HttpStatus.INTERNAL_SERVER_ERROR);
		}
		return vaFacilityTypeResponse;
	}

	@Override
	public List<VaFacilities> getAllVaFacilites() throws GenericException {
		VaFacilities vaFacilities = null;
		List<VaFacilities> vaFacilitiesList = new ArrayList<VaFacilities>();
		try {
			logger.info("VaFacilityServiceImpl.getAllVaFacilites() : to get the information");
			logger.debug("VaFacilityServiceImpl.getAllVaFacilites() : calling the respository to query the database");
			List<VaFacility> vafacilityList = iVaFacilityRepository.findAll();
			logger.info("VaFacilityServiceImpl.getAllVaFacilites() : iVaFacilityRepository.findAll() "
					+ Arrays.toString(vafacilityList.toArray()));
			if (vafacilityList.size() > 0) {
				for (VaFacility vaFacility : vafacilityList) {
					vaFacilities = new VaFacilities(vaFacility.getVaFacilityCd(), vaFacility.getVaFacilityName());
					vaFacilitiesList.add(vaFacilities);
				}
			} else {
				logger.error("iVaFacilityRepository.findAll() : bad_request_error ");
				throw new GenericException(ClaimDetailsConstants.BAD_REQUEST, "Couldn't find stations",
						HttpStatus.BAD_REQUEST);
			}
		} catch (Exception e) {
			logger.error("VaFacilityServiceImpl.getAllVaFacilites() : data_access_error " + e);
			throw new GenericException(ClaimDetailsConstants.INTERNAL_SERVER_ERROR, e.getMessage(),
					HttpStatus.INTERNAL_SERVER_ERROR);
		}
		return vaFacilitiesList;
	}

}
