package gov.va.med.fee.service.impl;

import java.util.ArrayList;
import java.util.List;

import javax.mail.MessagingException;
import javax.mail.internet.AddressException;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import gov.va.med.domain.fee.AppUser;
import gov.va.med.domain.fee.AppUserRequest;
import gov.va.med.fee.dao.IAppUserRepository;
import gov.va.med.fee.dao.IUserRequestRepositoryTemplate;
import gov.va.med.fee.exceptions.GenericException;
import gov.va.med.fee.model.request.StationRequest;
import gov.va.med.fee.model.request.UserReqRequest;
import gov.va.med.fee.model.response.UserReqResponse;
import gov.va.med.fee.model.response.UserRoles;
import gov.va.med.fee.service.IUserRequestService;
import gov.va.med.fee.util.Email;

@Service
@Transactional
public class UserRequestServiceImpl extends BaseService implements IUserRequestService {
	private static final Logger logger = LogManager.getLogger(UserRequestServiceImpl.class);

	@Autowired
	IUserRequestRepositoryTemplate userRequestRepositoryTemplate;
	
	@Autowired
	IAppUserRepository appUserRepository;
	
	private static final String USER_REQUEST_NOTIFICATION_FROM = "PII                   ";
	private static final String USER_REQUEST_NOTIFICATION_RECIPIENTS[] = { "PII               " };// "PII                   "
	private static final String[] appUserRoles = {"FEE_VISN_PGM_MANAGER", "FEE_SUPERVISOR", "FPPS_HELP_DESK"};
	
	@Override
	public AppUserRequest requestExist(String name) {
		AppUserRequest existingRequests = userRequestRepositoryTemplate.findRequestByAppUserName(name);
		
		return existingRequests;
	}
	
	@Override
	public int requestAddUser(UserReqRequest userRequest) throws GenericException {
		
		if (!hasRole(userRequest.getCurrent_user_name(), appUserRoles)) {
			logger.error("UserRequestServiceImpl.requestAddUser() : invalid_input_error for role ");
			throw new GenericException("Add User Request error", "Invalid role", HttpStatus.BAD_REQUEST);
		}	
		
		int rowInserted = 0;
		AppUserRequest existingRequests = requestExist(userRequest.getWindow_nt_name());
		
		if(existingRequests == null) {
			userRequest.setRequest_type(0); //type 0 = create user request, type 1 = modify user request
			rowInserted = userRequestRepositoryTemplate.requestCreateUserRequest(userRequest, userRequest.getCurrent_user_name());
		
			if (rowInserted > 0) { 
				// insert success then get current app user info by current user name
				List<AppUser> userList = appUserRepository.findByAppUserName(userRequest.getCurrent_user_name());
				
				if (userList != null && !userList.isEmpty()) {
					sendEmailMessage(userRequest, userList.get(0), "FPPS Create User Request", "Create");
				}				
			} else {
				throw new GenericException("Create new user request error","Unable to create new user request", HttpStatus.BAD_REQUEST);
			}
		}
		return rowInserted;
	}
	
	/* Find current existing App User Request by Window_nt_name
	 * Create new if not found, update otherwise */
	@Override
	public int requestModifyUser(UserReqRequest userRequest) throws GenericException {
		
		if (!hasRole(userRequest.getCurrent_user_name(), appUserRoles)) {
			logger.error("UserRequestServiceImpl.requestModifyUser() : invalid_input_error for role ");
			throw new GenericException("tModify User Request error", "Invalid role", HttpStatus.BAD_REQUEST);
		}	
		
		int rowInserted = 0;
		AppUserRequest existingRequests = requestExist(userRequest.getWindow_nt_name());
		userRequest.setRequest_type(1); //type 0 = create user request, type 1 = modify user request
		
		if(existingRequests == null) {  //insert
			rowInserted = userRequestRepositoryTemplate.requestCreateUserRequest(userRequest, userRequest.getCurrent_user_name());
		
		} else {  //update
			rowInserted = userRequestRepositoryTemplate.requestModifyUserRequest(userRequest, userRequest.getCurrent_user_name());
		}
		
		if (rowInserted > 0) { 
			// insert/update success then get current app user info by current user name
			List<AppUser> userList = appUserRepository.findByAppUserName(userRequest.getCurrent_user_name());
			
			if (userList != null && !userList.isEmpty()) {
				logger.debug("Sending FPPS Modify User Request email here ..... ");
				sendEmailMessage(userRequest, userList.get(0), "FPPS Modify User Request", "Modify");
			}				
		} else {
			throw new GenericException("Modify user request error","Unable to modify user request", HttpStatus.BAD_REQUEST);
		}		
		
		return rowInserted;
	}
	
	private void sendEmailMessage(UserReqRequest userToCreate, AppUser appUser, String subject, String type) {
		String emailstatus = "";
		Email email = new Email();		
		try {
			String emailMessage = "There is a new request to create an FPPS user:<br />" + "Type: "+type+" <br />"
					+ "User logon ID: "	+ (userToCreate.getWindow_nt_name() != null ? userToCreate.getWindow_nt_name() : "") + "<br />"
					+ "First Name: " + (userToCreate.getFirst_name() != null ? userToCreate.getFirst_name() : "") + "<br />" 
					+ "Middle Name: "+ (userToCreate.getMiddle_name() != null ? userToCreate.getMiddle_name() : "") + "<br />"
					+ "Last Name: " + (userToCreate.getLast_name() != null ? userToCreate.getLast_name() : "")+ "<br />" 
					+ "Phone: " + (userToCreate.getPhone() != null ? userToCreate.getPhone() : "") + "<br />"
					+ "Email: " + (userToCreate.getEmail() != null ? userToCreate.getEmail() : "")+ "<br />"
					+ "Requestor: "
					+ (appUser.getFirstName() != null ? appUser.getFirstName().trim().toUpperCase() : "") + " "
					+ (appUser.getLastName() != null ? appUser.getLastName().trim().toUpperCase() : "") + " ("
					+ (appUser.getUserName() != null ? appUser.getUserName().trim().toUpperCase() : "") + ")<br />"
					+ "Requestor Phone: " + (appUser.getPhoneNumber() != null ? appUser.getPhoneNumber().trim().toUpperCase() : "") + "<br />"
					+ "Requestor Email: " + (appUser.getEmailAddress() != null ? appUser.getEmailAddress().trim().toUpperCase() : "")+ "<br />" 
					+ "Requested Role(s): " + (userToCreate.getRequestedRoles() != null ? userToCreate.getRequestedRoles() : "") + "<br />"
					+ "Requested Station(s): " + (userToCreate.getRequestedFacilities() != null ? userToCreate.getRequestedFacilities():"") + "<br />";

			emailstatus = email.postMail(USER_REQUEST_NOTIFICATION_RECIPIENTS, subject,
					emailMessage, USER_REQUEST_NOTIFICATION_FROM, false, "");
		
		} catch (AddressException e1) {
			logger.error(e1.getMessage());
		} catch (MessagingException e1) {
			logger.error(e1.getMessage());
		}

		logger.debug("--------------------------------- email status: " + emailstatus);
	}
	
	@Override
	public UserReqResponse loadUserRequestForm() throws GenericException {
		
		UserReqResponse reqResponse = null;
		try{
			List<UserRoles> user_roles = userRequestRepositoryTemplate.getRolesForRequest();
			List<StationRequest> facilities = userRequestRepositoryTemplate.getFacilitiesForRequest();
	
			List<String> available_roles = new ArrayList<>(); 
			for (UserRoles role : user_roles) {
				available_roles.add(role.getRoleName().replaceAll("_", " "));
			}
			
			List<String> available_facilities = getAvailable_facilities(facilities);
			reqResponse = new UserReqResponse(available_roles, available_facilities);
		
		} catch(Exception e) {
			logger.info("Loading Create User Request fail" + e.getMessage());
			throw new GenericException("Loading Create User Request fail","Fail to load available roles and facilities", HttpStatus.INTERNAL_SERVER_ERROR);
		}
		
		logger.debug("loadUserRequestForm: " + reqResponse.toString());
		return reqResponse;
	}
		
	
	@Override
	public UserReqResponse loadModifyUserRequestFormByName(String appUserName) throws GenericException {
		logger.debug("loadModifyUserRequestForm by appUserName: " + appUserName);
		List<AppUser> userList = appUserRepository.findByAppUserName(appUserName);
		return loadModifyUserRequestForm(userList.get(0));
	}
	
	private UserReqResponse loadModifyUserRequestForm(AppUser appUser) throws GenericException {
		UserReqResponse reqResponse = null;
		try{
			List<UserRoles> userRoles = userRequestRepositoryTemplate.findUserRolesForModify(appUser.getAppUserId());
			List<StationRequest> userFacilities = userRequestRepositoryTemplate.findUserFacilities(appUser.getAppUserId());
			List<String> rolesAvailableToUserForUserModifyRequest = userRequestRepositoryTemplate.getRolesAvailableToUserForUserModifyRequest();
			List<StationRequest> facilitiesAvailableToUser = userRequestRepositoryTemplate.getFacilitiesAvailableToUser();
			
			List<String> request_roles = new ArrayList<>(); //userRolesForModify
			for(UserRoles role : userRoles) {
				request_roles.add(role.getRoleName()); //role_name only 
			}
			
			//format VA facility to strings like - "Visn: 16 / Facility: 520 (BILOXI MS)"
			List<String> request_facilities = getAvailable_facilities(userFacilities);
			List<String> available_facilities = getAvailable_facilities(facilitiesAvailableToUser);
			
			reqResponse = new UserReqResponse(appUser.getUserName(),appUser.getLastName(), appUser.getFirstName(), appUser.getMiddleName(),
					appUser.getPhoneNumber(), appUser.getEmailAddress(), appUser.getEnabled().equalsIgnoreCase("Y") ? true : false, 
					appUser.getDeactivationComments(), request_roles, request_facilities);
					
			reqResponse.setAvailable_roles(rolesAvailableToUserForUserModifyRequest);
			reqResponse.setAvailable_facilities(available_facilities);
		} catch(Exception e) {
			logger.info("Loading Modify User Request fail");
			throw new GenericException("Loading Modify User Request fail","Fail to load Modify User Request", HttpStatus.NOT_FOUND);
		}
		logger.debug("loadUserRequestForm: " + reqResponse.toString());
		return reqResponse;
	}

	private List<String> getAvailable_facilities(List<StationRequest> facilities){
		List<String> available_facilities = new ArrayList<>();
		
		for (StationRequest req : facilities) {
			String shortName = req.getShortName() != null ? " (" + req.getShortName() + ")" : "";
			String facLabel = "Visn: " + req.getVisnIdCd() + " / Facility: " + req.getVaFacilityCd() + shortName;
			available_facilities.add(facLabel);
		}
		
		return available_facilities;
	}
}
