package gov.va.med.fee.service.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import gov.va.med.domain.fee.AppUser;
import gov.va.med.domain.fee.Claim;
import gov.va.med.domain.fee.ClaimProcedure;
import gov.va.med.domain.fee.PersonInfo;
import gov.va.med.domain.fee.ProviderInfo;
import gov.va.med.domain.fee.PrintedClaim;
import gov.va.med.domain.fee.FeeClaimRouting;
import gov.va.med.domain.fee.FeeClaimRoutingsReason;
import gov.va.med.domain.fee.VaFacility;
import gov.va.med.domain.fee.Visn;
import gov.va.med.domain.fee.WsFacStatus;
import gov.va.med.domain.fee.WsProviderLu;
import gov.va.med.domain.fee.AppRole;
import gov.va.med.domain.fee.UserRoleUsages;
import gov.va.med.fee.dao.IAppUserRepository;
import gov.va.med.fee.dao.IClaimPaymentRepository;
import gov.va.med.fee.dao.IClaimRepository;
import gov.va.med.fee.dao.IFeeClaimRoutingRepository;
import gov.va.med.fee.dao.IFeeClaimRoutingsReasonRepository;
import gov.va.med.fee.dao.IPrintedClaimRepository;
import gov.va.med.fee.dao.IReRoutingHistoryRepositoryTemplate;
import gov.va.med.fee.dao.IRerouteClaimsRepositoryTemplate;
import gov.va.med.fee.dao.IVaFacilityRepository;
import gov.va.med.fee.dao.IVeteranInfoRepository;
import gov.va.med.fee.dao.IVisnRepository;
import gov.va.med.fee.dao.IWsFacStatusRepository;
import gov.va.med.fee.dao.IWsProviderLuRepository;
import gov.va.med.fee.exceptions.GenericException;
import gov.va.med.fee.model.request.RerouteClaimRequest;
import gov.va.med.fee.model.response.ReRoutingHistory;
import gov.va.med.fee.model.response.RerouteClaimInfo;
import gov.va.med.fee.model.response.RerouteClaimResponse;
import gov.va.med.fee.service.IRerouteClaimsService;

@Service
@Transactional
public class RerouteClaimsServiceImpl extends BaseService implements IRerouteClaimsService {
	private static final Logger logger = LogManager.getLogger(RerouteClaimsServiceImpl.class);

	//private List<RerouteClaimInfo> reroutes = new ArrayList<RerouteClaimInfo>();
	
	private final String DEFAULT_SELECT_VALUE = "-select-";
	private static final Pattern DIGITS = Pattern.compile("[0-9]*");
		
	@Autowired
	IRerouteClaimsRepositoryTemplate rerouteClaimsRepository;
	@Autowired
	IClaimPaymentRepository claimPaymentRepository;
	@Autowired
	IClaimRepository claimRepository;
	@Autowired
	IReRoutingHistoryRepositoryTemplate reRoutingHistoryRepository;
	@Autowired
	IPrintedClaimRepository printedClaimRepository;
	@Autowired
	IVaFacilityRepository vaFacilityRepository;
	@Autowired
	IVisnRepository visnRepository;
	@Autowired
	IFeeClaimRoutingsReasonRepository feeClaimRoutingsReasonRepository;
	@Autowired
	IAppUserRepository appUserRepository;
	@Autowired
	IVeteranInfoRepository veteranInfoRepository;
	@Autowired
	IFeeClaimRoutingRepository feeClaimRoutingRepository;
	@Autowired
	IWsFacStatusRepository wsFacStatusRepository;
	@Autowired
	IWsProviderLuRepository wsProviderLuRepository;
	
	private static final String[] appUserRoles = {"FEE_SUPERVISOR", "FEE_VISN_PGM_MANAGER", "ADMIN", "FEE_CLERK", "REJECTED_CLAIMS_CLERK"};
	
	public RerouteClaimResponse populateReroutes(List<Long> claimsToReroute) throws GenericException {
		List<RerouteClaimInfo> reroutes = new ArrayList<RerouteClaimInfo>();
		RerouteClaimResponse response = new RerouteClaimResponse();
		Map<Long, List<ReRoutingHistory>> allClaimsRerouteHistory = new LinkedHashMap<>();
		List<Long> nonRoutableClaims = new LinkedList<>();
		List<Long> printedClaims = new LinkedList<>();
		
		String nonRoutableStr = null;
		String printedClaimsStr = null;
		
		List<Claim> claims = claimRepository.findAll(claimsToReroute);
		
		for (Claim c : claims) {
			//check if a claim can be reroute
			if (routable(c)) {
				RerouteClaimInfo cle = new RerouteClaimInfo();
				cle.setClaim_id(c.getClaimIndex());
				cle.setVeteran_name(getVeteranName(c.getPersonInfos()));
				cle.setProvider_name(getServiceProvName(c.getProviderInfos()));
				
				//check if a claim was printed before
				if (isPrintedClaim(c)) {
					printedClaims.add(c.getClaimIndex());
				}
				reroutes.add(cle);
			} else {
				nonRoutableClaims.add(c.getClaimIndex());
			}
		}
		
		//keep response format clean
		if(reroutes.isEmpty()) {
			reroutes = null;
		}
		response.setClaims_to_be_rerouted(reroutes);
		
		if (nonRoutableClaims.size() > 0) {
			String msg = null;
			for (Long i : nonRoutableClaims) {
				msg = msg == null ? i.toString() : (msg + ", " + i.toString());
			}
			nonRoutableStr = "The following " + (nonRoutableClaims.size() > 1 ? "claims" : "claim")
					+ " cannot be rerouted because " + (nonRoutableClaims.size() > 1 ? "they are" : "it is")
					+ " completed, partially reconciled, rejected, or not FPPS: " + msg;
			response.setNonRouted(nonRoutableStr);
		}
				
		if (printedClaims.size() > 0) {
			String pcMsg = null;
			for (Long i : printedClaims){
				pcMsg = pcMsg == null ? i.toString() : (pcMsg + ", " + i.toString());
			}
			printedClaimsStr = "The following " + (printedClaims.size() > 1 ? "claims have" : "claim has")
					+ " been printed: " + pcMsg + ",  and will be rerouted if you continue.";
			response.setPrintedClaimsStr(printedClaimsStr);
		}

		//Only retrieve history and destination lists when there are reroute claims
		if(reroutes != null) {
			for (Long index : claimsToReroute) {
				List<ReRoutingHistory> hist = reRoutingHistoryRepository.findReRoutingHistory(index);//loadRerouteHistory(index);
				if(hist != null && !hist.isEmpty()) {
					allClaimsRerouteHistory.put(index, hist);
				}
			}
			
			if(allClaimsRerouteHistory.isEmpty()) {
				allClaimsRerouteHistory = null;
			}
			response.setReroute_history(allClaimsRerouteHistory);
			
			List<Long> visnList = getVisnList();
			response.setVisns(visnList);
			
			List<String> facilitiesList = getFacilitiesList(DEFAULT_SELECT_VALUE);
			response.setFacilities(facilitiesList);
			
			/*List<String> userList = getUserList(DEFAULT_SELECT_VALUE); 
			response.setNames(userList);*/
			
			Map<String, String> reasonMap = getRerouteReasons();
			response.setReasons(reasonMap);
		}
		return response;
	}
	
	boolean routable(Claim claim) throws GenericException {
		boolean routable = false;
		// only FPPS claims can be rerouted
		//not able to route if claim.VA_IMP_TYPE is 'FBCS', or is not empty/null value
		if (claim.getVaImpTypeCd() == null || claim.getVaImpTypeCd().trim().length() == 0) {
			if ("ESTABLISH".equalsIgnoreCase(claim.getClaimStatusCd())) {
				routable = true;
			} else if ("INPROCESS".equalsIgnoreCase(claim.getClaimStatusCd()) && !hasReconciledLine(claim)) {
				routable = true;
			}
		}

		// Vitria payment received, block reroute if payment has been made.
		if (routable) {
			try {
				List<Long> claimPayments = claimPaymentRepository.findClaimPaymentsByClaimIndex(claim.getClaimIndex());
				logger.debug("routable - claim payment: " + claimPayments);
				if (claimPayments != null && claimPayments.size() > 0) {
					routable = false;
				}
			} catch (Exception ex) {
				logger.error(ex);
				ex.printStackTrace();
				throw new GenericException("Check routable error","Getting claim payment for claim index " + claim.getClaimIndex() +" error", HttpStatus.INTERNAL_SERVER_ERROR);
			}
		}

		return routable;
	}	
	
	boolean isPrintedClaim(Claim claim) throws GenericException {
		PrintedClaim result = null;
		try {
			result = printedClaimRepository.findPrintedClaim(claim.getClaimIndex());
		} catch (Exception ex) {
			logger.error(ex);
			ex.printStackTrace();
			throw new GenericException("Check isPrintedClaim error","Find Printed Claim for claim index " + claim.getClaimIndex() +" error", HttpStatus.INTERNAL_SERVER_ERROR);
		}
		return result != null ? Boolean.TRUE: Boolean.FALSE;
	}

	private String getServiceProvName(Set<ProviderInfo> providers) {
		for (ProviderInfo p : providers) {
			if ("SERVICE FACILITY".equals(p.getProviderType().getProviderTypeCd())
					|| "SERVICE LOCATION".equals(p.getProviderType().getProviderTypeCd())) {
				return p.getLastName();
			}
		}
		return null;
	}

	private String getVeteranName(Set<PersonInfo> persons) {
		for (PersonInfo pi : persons) {
			if ("PATIENT".equals(pi.getPersonType().getPersonTypeCd())) {
				return pi.getFirstName() + " " + pi.getLastName();
			}
		}
		return null;
	}

	//claim status = INPROCESS, check if CLAIM_PROCEDURE associate with the claim that also associate with any CLAIM_PROC_RECON
	//routable = true if there's a CLAIM_PROC_RECON associate with CLAIM_PROCEDURE and a claim
	private boolean hasReconciledLine(Claim claim) {
		boolean routable = false;
		for (ClaimProcedure cp : claim.getClaimProcedures()) {
			if (cp.getClaimProcRecons().size() > 0) {
				routable = true;
				break;
			}
		}
		return routable;
	}

	/**
	 * retrieve list of VISNS from VISN table
	 */
	public List<Long> getVisnList() throws GenericException {
		List<Long> visns = null;
		try {
			visns = visnRepository.findAllVisnId();
		} catch (Exception e) {
			throw new GenericException("Reroute page error", "Getting Visn list error", HttpStatus.BAD_REQUEST);
		}
		return visns;
	}

	/**
	 * retrieve list of facilities from VA_FACILITY table
	 */
	public List<String> getFacilitiesList(String selectedVisnId) throws GenericException {
		String visnId;
		List<String> facilitiesList = new ArrayList<>();

		// if no visn selected use default for facility
		if (DEFAULT_SELECT_VALUE.equals(selectedVisnId) || selectedVisnId == null) { 
			facilitiesList.add(DEFAULT_SELECT_VALUE);
			return facilitiesList;
		} else {
			visnId = selectedVisnId;
		}

		String visnNum = visnId;
		if (visnNum.equals(DEFAULT_SELECT_VALUE)) {
			return facilitiesList; // cannot search on -select-
		}
		// Invalid visn
		Matcher m = DIGITS.matcher(visnNum);
		if (!m.matches()) {
			return facilitiesList;
		}

		try {
			facilitiesList = vaFacilityRepository.findFacilitiesList(Long.valueOf(visnNum));
		} catch (Exception e) {
			throw new GenericException("Reroute page error", "Getting Facility list error", HttpStatus.BAD_REQUEST);
		}

		return facilitiesList;
	}
	
	/**
	 * retrieve list of Fee claim reroute reasons
	 */
	public Map<String, String> getRerouteReasons() throws GenericException {
		List<FeeClaimRoutingsReason> rerouteReasons = feeClaimRoutingsReasonRepository.getRerouteReasons();
		Map<String, String> rerouteReasonMap = new LinkedHashMap<>();
		try {
			for(FeeClaimRoutingsReason reason : rerouteReasons) {
				rerouteReasonMap.put(reason.getTransferReasonCd(), reason.getDescription());
			}
			
		} catch (Exception e) {
			throw new GenericException("Reroute page error", "Getting Reroute Reasons error", HttpStatus.BAD_REQUEST);
		}
		
		for(String s : rerouteReasonMap.keySet()) {
			logger.debug("RerouteReason -" + s);
		}
		return rerouteReasonMap;
	}
	
	//Verify selected visn is valid
	private Visn getVisn(long longVisnId) throws GenericException {
		Visn toVisn = visnRepository.findOne(longVisnId); // getVISN(longVisnId);
		if (toVisn == null) {
			throw new GenericException("Claim(s) not rerouted. ","Invalid VISN: " + longVisnId, HttpStatus.BAD_REQUEST);
		}
		
		return toVisn;
	}
	
	//Verify selected facility code is valid
	private VaFacility checkValidVaFacility(long longVisnId, String facilityId) throws GenericException {
		VaFacility facility = null;
		if (!"".equals(facilityId)) {
			facility = vaFacilityRepository.findByVisnIdVaFacilityCd(longVisnId, facilityId);
			if(facility == null) {
				throw new GenericException("Claim(s) not rerouted. ","Invalid facility: " + facilityId, HttpStatus.BAD_REQUEST);
			}
		}
		return facility;
	}
	
	public RerouteClaimResponse reroute(RerouteClaimRequest request) throws GenericException {
		
		if (!hasRole(request.getCurrent_user_name(), appUserRoles)) {
			logger.error("RerouteClaimServiceImpl.reroute() : invalid_input_error for role ");
			throw new GenericException("Reroute Claims error", "Invalid role", HttpStatus.BAD_REQUEST);
		}
		
		RerouteClaimResponse response = new RerouteClaimResponse();
		List<String> errorMessages = new ArrayList<>();
		//String selectedVisnId = DEFAULT_SELECT_VALUE;
		String facilityId = DEFAULT_SELECT_VALUE;
		
		String visnRestrictClaimsMsg = "";// string for visn xfer restricted claims
		boolean hasVisnRestrict = false;// flag for indicating existence of visn xfer restriction

		Visn toVisn = getVisn(Long.valueOf(request.getSelectedVisn())); // getVISN(longVisnId);
				
		// verify facility
		if ("".equals(request.getSelectedFacility()) || request.getSelectedFacility() == null) {
			facilityId = "";
		} else {
			facilityId = request.getSelectedFacility();
		}		
		checkValidVaFacility(Long.valueOf(request.getSelectedVisn()), facilityId);		
		FeeClaimRoutingsReason fcrReason = feeClaimRoutingsReasonRepository.getFeeClaimRoutingsReason(request.getSelectedReason()); //null;
		
		// Check if reroute is to a facility with FBCS implementation
		boolean nonFPPS = false;
		if (!"".equals(facilityId)) { 
			List<String> type = this.rerouteClaimsRepository.findVaImplementationList(facilityId);
			if (type != null && type.size() > 0) {
				// there shouldn't be more than one result
				String program = (String) type.get(0);
				if (program != null && program.trim().length() > 0) {
					nonFPPS = true;
				}
			}
		}

		// do FPPS reroute		
		List<RerouteClaimInfo> reroutes = request.getClaims_to_be_rerouted();
		List<RerouteClaimInfo> deniedReroutes = new ArrayList<>();

		StringBuilder reroutedClaimsStr = new StringBuilder();
		for(RerouteClaimInfo claim2reroute : reroutes) {
			Claim c = claimRepository.findClaimByClaimIndex(claim2reroute.getClaim_id());//getClaim(claim2reroute.getClaim_id());
			if (c == null) {
				errorMessages.add("Claim " + claim2reroute.getClaim_id() + " not found.");
				deniedReroutes.add(claim2reroute);
				continue;
			}

			if (nonFPPS && !"DENT".equals(c.getServiceType().getServiceTypeCd())) {
				boolean transfered = rerouteClaimsRepository.transferToNonFPPS(c, request.getCurrent_user_name(), 
							request.getSelectedVisn(), facilityId, fcrReason.getTransferReasonCd(), request.getComment());
				if (transfered) {
					if (reroutedClaimsStr.length() != 0) {
						reroutedClaimsStr.append(", ");
					}
					reroutedClaimsStr.append(c.getClaimIndex());
					continue;
				} else {
					// get out of the loop to display error
					errorMessages.add("Claim " + claim2reroute.getClaim_id() + " not able to be transfered.");
					break;
				}
			}
			
			// MAKE SURE TO CHECK REROUTING_ALLOWED FLAG FOR THIS FACILITY!!
			// perform a lookup in the VISN table..
			// convert String selectedVisnId to a long value get facility object associated with claim
			String targetVisn = String.valueOf(c.getVisn().getVisnIdCd());
			
			// check REROUTING_ALLOWED flag if the user is NOT a FEE_VISN_PGM_MANAGER then this check must be performed
			boolean canReroute = false;
			if (hasRole(request.getCurrent_user_name(), "FEE_VISN_PGM_MANAGER") || 
					hasRole(request.getCurrent_user_name(), "ADMIN")) {
				canReroute = true;
			} else {
				if (c.getVaFacility() != null) {
					logger.debug("isReroutingAllowed -- " +c.getVaFacility().isReroutingAllowed());
					canReroute = c.getVaFacility().isReroutingAllowed();
				}
			}
			// get VISN id code and compare it to the VISN selected value
			// if they are not the same AND the REROUTING_ALLOWED flag == 0 -DISALLOW routing
			if (!request.getSelectedVisn().equals(targetVisn) && !canReroute) {
				// identify claim and put in msg queue
				hasVisnRestrict = true;
				visnRestrictClaimsMsg += " " + c.getClaimIndex();
				deniedReroutes.add(claim2reroute);
				continue; // skip this claim and move on to next....
			}
			
			//Patient info
			PersonInfo pi = veteranInfoRepository.findOneByClaimId(c.getClaimIndex());
			if (pi == null) {
				errorMessages.add("Person with claim index " + c.getClaimIndex() + " not found.");
				deniedReroutes.add(claim2reroute);
				continue;
			}
			
			FeeClaimRouting fcr = new FeeClaimRouting();
			fcr.setClaim(c);
			fcr.setTransferFromUser(c.getUserName());
			VaFacility vf = c.getVaFacility() == null ? new VaFacility() : c.getVaFacility();
			String vfc = vf.getVaFacilityCd() == null ? "" : vf.getVaFacilityCd();
			fcr.setTransferFromFacilityCd(vfc);
			fcr.setTransferFromVisnCd("" + c.getVisn().getVisnIdCd());
			
			//if (this.userName.equals("") || this.userName.equals(DEFAULT_SELECT_VALUE))
            	c.setUserName("");
            //else
            //   c.setUserName(this.userName);
		
			vf = vaFacilityRepository.findVaFacilityByVaFacilityCd(facilityId);
			c.setVaFacility(vf);//getVaFacility(facilityId)); // if no selection facility, set to null
			c.setVisn(toVisn);
			c.setClaimStatusCd("ESTABLISH");
			c.setModifiedBy(request.getCurrent_user_name());

			Set<ClaimProcedure> claimProcs = c.getClaimProcedures();
			for (ClaimProcedure cp : claimProcs) {
				cp.setLineStatus("ESTABLISH");
			}

			// all claim (c) data now set to new owner
			fcr.setTransferToFacilityCd(facilityId);
			fcr.setTransferToVisnCd(request.getSelectedVisn());
			java.sql.Date sqlDate = new java.sql.Date(new Date().getTime());
			c.setDateModified(sqlDate);
			fcr.setTransferDate(sqlDate);
			fcr.setDateCreated(sqlDate); // this is the same as the xfer date??
			fcr.setComments(request.getComment());
			fcr.setTransferInitator(request.getCurrent_user_name());
			fcr.setCreatedBy(request.getCurrent_user_name());
			fcr.setFeeClaimRoutingsReason(fcrReason);
			
			WsFacStatus wfs = wsFacStatusRepository.findOneByClaimId(c.getClaimIndex());
			if(wfs != null) {
				wfs.setUserName(c.getUserName()); // SCOTT 11/05/2009
				wfs.setVaFacilityCd(facilityId);
				wfs.setClaimStatusCd("ESTABLISH");
				wfs.setDateInProcess(null);
				wfs.setVisnIdCd(c.getVisn().getVisnIdCd());
				wsFacStatusRepository.saveAndFlush(wfs);
			}
						
			// Update the provider lookup table with facilityData.
			WsProviderLu wProvider = wsProviderLuRepository.findOneByClaimId(c.getClaimIndex());
			if (wProvider != null) {
				wProvider.setVaFacilityCd(facilityId);
				wsProviderLuRepository.saveAndFlush(wProvider);
			}
						
			// build String of rerouted claims to display in confirmation msg
			if (reroutedClaimsStr.length() != 0) {
				reroutedClaimsStr.append(", ");
			}
			reroutedClaimsStr.append(c.getClaimIndex());
						
			try {
				feeClaimRoutingRepository.saveAndFlush(fcr);
		
	        } catch (Exception ex) {
	            logger.error("Exception persisting reroute to database");
	            ex.printStackTrace();
	            throw new GenericException("Claim(s) not rerouted. ","Bad data" , HttpStatus.INTERNAL_SERVER_ERROR);
	         }			
		} 
		
		// check denied reroutes and display appropriate error message(s)
		if (hasVisnRestrict) {
			errorMessages.add("You do not have VISN transfer permission for claim(s): " + visnRestrictClaimsMsg);
		}
		
		if (reroutedClaimsStr.length() > 0) {
			response.setReroutedSuccessful("Claim(s): " + reroutedClaimsStr + " rerouted successfully.");
		}		
			
		// populate claims that did NOT get rerouted
		if(deniedReroutes != null && !deniedReroutes.isEmpty()) {
			response.setDenied_reroutes(deniedReroutes);
			
			//reset selection boxes
			/* 10/5/2017 Not sure if there are denied reroute claims, do we need to reload the Reroute page?
			try {
				response.setVisns(getVisnList());
				response.setFacilities(getFacilitiesList(selectedVisnId));
				response.setReasons(getRerouteReasons());
			} catch (GenericException e) {
				e.printStackTrace();
				throw e;
			}*/
		} 
		
		if(!errorMessages.isEmpty()) {
			response.setErrorMessages(errorMessages);
		} 		
		
		return response;
	}
		
	public boolean hasRole(String name, String role) {
		List<AppUser> userList = appUserRepository.findByAppUserName(name);
		AppUser appUser = null;
		if (userList != null && !userList.isEmpty()) {
			appUser = userList.get(0);
		}

		if (appUser != null) {
			Set<UserRoleUsages> roles = appUser.getUserRoleUsages();

			for (UserRoleUsages uru : roles) {
				AppRole appRole = uru.getAppRole();
				if (appRole.getRoleName().equals(role)) {
					return true;
				}
			}
		}

		return false;
	}
}
