package gov.va.med.fee.service.impl;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import gov.va.med.domain.fee.AdjustmentGroup;
import gov.va.med.domain.fee.AdjustmentReason;
import gov.va.med.domain.fee.CalcMethod;
import gov.va.med.domain.fee.Claim;
import gov.va.med.domain.fee.ClaimPayment;
import gov.va.med.domain.fee.ClaimPaymentType;
import gov.va.med.domain.fee.ClaimProcRecAdjust;
import gov.va.med.domain.fee.ClaimProcRecon;
import gov.va.med.domain.fee.ClaimProcRemitRemark;
import gov.va.med.domain.fee.ClaimProcStatus;
import gov.va.med.domain.fee.ClaimProcedure;
import gov.va.med.domain.fee.ClaimReconciliation;
import gov.va.med.domain.fee.PersonInfo;
import gov.va.med.domain.fee.ProviderInfo;
import gov.va.med.domain.fee.RemittanceRemark;
import gov.va.med.domain.fee.WsFacStatus;
import gov.va.med.fee.dao.IAdjustmentGroupRepository;
import gov.va.med.fee.dao.IAdjustmentReasonRepository;
import gov.va.med.fee.dao.ICalcMethodRepository;
import gov.va.med.fee.dao.IClaimPaymentRepository;
import gov.va.med.fee.dao.IClaimPaymentTypeRepository;
import gov.va.med.fee.dao.IClaimProcReconRepository;
import gov.va.med.fee.dao.IClaimProcRemitRemarkRepository;
import gov.va.med.fee.dao.IClaimProcStatusRepository;
import gov.va.med.fee.dao.IClaimProcedureRepository;
import gov.va.med.fee.dao.IClaimReconciliationRepository;
import gov.va.med.fee.dao.IClaimRepository;
import gov.va.med.fee.dao.IWsFacStatusRepository;
import gov.va.med.fee.enums.CalcMethodCode;
import gov.va.med.fee.enums.ClaimPaymentTypeCode;
import gov.va.med.fee.errorhandling.ValidationMessage;
import gov.va.med.fee.exceptions.GenericException;
import gov.va.med.fee.exceptions.ValidationException;
import gov.va.med.fee.model.request.AdjustmentRequest;
import gov.va.med.fee.model.request.ManualReconRequest;
import gov.va.med.fee.model.response.ClaimLineItems;
import gov.va.med.fee.model.response.ManualReconResponse;
import gov.va.med.fee.service.IClaimService;
import gov.va.med.fee.service.IManualReconClaimService;

@Service
@Transactional
public class ManualReconClaimServiceImpl implements IManualReconClaimService {

	private static final Logger logger = LogManager.getLogger(ManualReconClaimServiceImpl.class);

	@Autowired
	IClaimRepository claimRepository;

	@Autowired
	IClaimService claimService;

	@Autowired
	IClaimProcedureRepository claimProcedureRepository;

	@Autowired
	ICalcMethodRepository calcMethodRepository;

	@Autowired
	IClaimReconciliationRepository claimReconciliationRepository;

	@Autowired
	IClaimPaymentTypeRepository claimPaymentTypeRepository;

	@Autowired
	IClaimPaymentRepository claimPaymentRepository;

	@Autowired
	IClaimProcStatusRepository claimProcStatusRepository;

	@Autowired
	IClaimProcReconRepository claimProcReconRepository;

	@Autowired
	IAdjustmentGroupRepository adjustmentGroupRepository;

	@Autowired
	IAdjustmentReasonRepository adjustmentReasonRepository;

	@Autowired
	IClaimProcRemitRemarkRepository claimProcRemitRemarkRepository;

	@Autowired
	IWsFacStatusRepository wsFacStatusRepository;

	@Value("${validation.maximum.selected.remarks}")
	private String maximumSelectedRemarks;

	@Value("${validation.selection.required}")
	private String selectionRequired;

	@Value("${validation.paymentDate.min.date}")
	private String paymentDateMinDate;

	@Value("${validation.invalid.paymentDate}")
	private String invalidPaymentDate;

	@Value("${validation.currency.format.error}")
	private String currencyFormatError;

	@Value("${validation.adjustmentAmount.required}")
	private String adjustmentAmountRequired;

	@Value("${validation.amount.less.than.billed}")
	private String amountLessThanBilled;

	@Value("${validation.amount.more.than.billed}")
	private String amountMoreThanBilled;

	/** The Constant CURRENCY_FORMAT. */
	private static final DecimalFormat CURRENCY_FORMAT = new DecimalFormat("$###,###.##");

	/** The Constant CURRENCY_PATTERN. */
	private static final Pattern CURRENCY_PATTERN = Pattern.compile("[0-9]*\\.?[0-9]{0,2}+");

	@Override
	public ManualReconResponse getClaimDetails(long claimId) throws GenericException {
		logger.info("getClaimDetails:");
		Claim claim = claimRepository.findClaimStatusByClaimIndex(claimId);

		ManualReconResponse manualReconResponse = new ManualReconResponse();
		manualReconResponse.setClaimId(claimId);
		manualReconResponse.setClaimType(claim.getServiceType().getServiceTypeCd());
		manualReconResponse.setClaimStatus(claim.getClaimStatusCd());
		manualReconResponse.setVeteranName(getVeteranName(claim.getPersonInfos()));
		manualReconResponse.setBillingProvName(getBillingProvName(claim.getProviderInfos()));
		manualReconResponse.setServiceProvName(getServiceProvName(claim.getProviderInfos()));

		ClaimLineItems claimLineItems = claimService.getClaimLineItems(claimId);

		manualReconResponse.setClaimLineItems(claimLineItems.getClaimLineItems());

		return manualReconResponse;
	}

	/**
	 * Gets the service prov name.
	 *
	 * @param providers
	 *            the providers
	 *
	 * @return the service prov name
	 */
	private String getServiceProvName(Set<ProviderInfo> providers) {
		for (ProviderInfo p : providers) {
			if ("SERVICE FACILITY".equals(p.getProviderType().getProviderTypeCd())
					|| "SERVICE LOCATION".equals(p.getProviderType().getProviderTypeCd())) {
				return p.getLastName();
			}
		}
		return null;
	}

	/**
	 * Gets the billing prov name.
	 *
	 * @param providers
	 *            the providers
	 *
	 * @return the billing prov name
	 */
	private String getBillingProvName(Set<ProviderInfo> providers) {
		for (ProviderInfo p : providers) {
			if ("BILLING PROVIDER".equals(p.getProviderType().getProviderTypeCd())) {
				return p.getLastName();
			}
		}
		return null;
	}

	/**
	 * Gets the veteran name.
	 *
	 * @param persons
	 *            the persons
	 *
	 * @return the veteran name
	 */
	private String getVeteranName(Set<PersonInfo> persons) {
		for (PersonInfo pi : persons) {
			if ("PATIENT".equals(pi.getPersonType().getPersonTypeCd())) {
				return pi.getFirstName() + " " + pi.getLastName();
			}
		}
		return null;
	}

	@Override
	public ManualReconResponse submitManualRecon(ManualReconRequest manualReconRequest)
			throws GenericException, ValidationException {
		logger.info("submitManualRecon:");

		try {
			process(manualReconRequest);
		} catch (Exception e) {
			e.printStackTrace();
			logger.error("There was an error in processing submit request for Manual Recon - Out of System Payment");
			throw e;
		}

		ManualReconResponse manualReconResponse = getClaimDetails(manualReconRequest.getClaimIndex());

		return manualReconResponse;
	}

	private boolean process(ManualReconRequest manualReconRequest) throws GenericException, ValidationException {
		logger.info("process:");
		Date currentDate = new Date();

		// get claim
		Claim claim = claimRepository.findByClaimIndex(manualReconRequest.getClaimIndex());

		// get claim procedure
		ClaimProcedure claimProcedure = claimProcedureRepository.findOne(manualReconRequest.getClaimProcId());

		checkForm(manualReconRequest, claimProcedure);

		// Get out of system payment calc method
		CalcMethod calcMethod = calcMethodRepository
				.findCalcMethod((short) CalcMethodCode.OUT_OF_SYSTEM_PAYMENT.getCalcMethodCode());

		// save claim reconciliation
		ClaimReconciliation claimReconciliation = new ClaimReconciliation();
		claimReconciliation.setCalcMethod(calcMethod);
		claimReconciliation.setSent835Flag('N');
		claimReconciliation.setCreatedBy(manualReconRequest.getUserName());
		claimReconciliation.setDateCreated(currentDate);
		claimReconciliation.setClaim(claim);
		claimReconciliationRepository.save(claimReconciliation);

		// Get claim payment type
		ClaimPaymentType claimPaymentType = claimPaymentTypeRepository.findClaimPaymentTypeByClaimPaymentTypeCode(
				ClaimPaymentTypeCode.valueOf(manualReconRequest.getPaymentType()).name());

		// save claim payment
		ClaimPayment claimPayment = new ClaimPayment();
		claimPayment.setClaimPaymentType(claimPaymentType);
		claimPayment.setClaim(claim);
		claimPayment.setCreatedBy(manualReconRequest.getUserName());
		claimPayment.setDateCreated(currentDate);
		claimPayment.setPaidAmount(manualReconRequest.getVaAmount());
		claimPayment.setPaymentBatchDate(manualReconRequest.getPaymentDate());
		claimPayment.setPaymentTrackingNumber(manualReconRequest.getCheckNo());
		claimPayment.setClaim(claim);
		claimPaymentRepository.save(claimPayment);

		// Get claim proc status
		// TODO Need to fix getting the right claim proc status
		ClaimProcStatus claimProcStatus = claimProcStatusRepository
				.findClaimProcStatus(gov.va.med.fee.enums.ClaimProcStatus.ACCEPTED.name());

		// save claim proc recon
		ClaimProcRecon claimProcRecon = new ClaimProcRecon();
		claimProcRecon.setClaimProcStatus(claimProcStatus);
		claimProcRecon.setCreatedBy(manualReconRequest.getUserName());
		claimProcRecon.setDateCreated(currentDate);
		claimProcRecon.setPaidAmount(manualReconRequest.getVaAmount());
		claimProcRecon.setClaimPayment(claimPayment);
		claimProcRecon.setClaimProcedure(claimProcedure);
		claimProcRecon.setClaim(claim);
		claimProcRecon.setClaimReconciliation(claimReconciliation);
		claimProcReconRepository.save(claimProcRecon);

		for (AdjustmentRequest adjustmentRequest : manualReconRequest.getAdjustments()) {
			// Add Remittance Remark1
			addClaimProcRemittanceRemark(adjustmentRequest.getRARCs(), currentDate, claimProcRecon,
					adjustmentRequest.getAmount(), adjustmentRequest.getGroupCode(), adjustmentRequest.getCARC(),
					manualReconRequest.getUserName());
		}

		// Set line to complete
		claimProcedure.setLineStatus("COMPLETE");
		claimProcedureRepository.save(claimProcedure);

		// Set claim to completed if all lines has been closed.
		WsFacStatus fs = wsFacStatusRepository.findOneByClaimId(manualReconRequest.getClaimIndex());
		Set<ClaimProcedure> cps = claim.getClaimProcedures();
		if (cps != null && cps.size() > 0) {
			int countRecon = 0;
			for (ClaimProcedure cp : cps) {
				if ("COMPLETE".equals(cp.getLineStatus())) {
					countRecon++;
				}
			}

			// all lines resolved, set claim to complete
			claim.setModifiedBy(manualReconRequest.getUserName());
			claim.setDateModified(currentDate);
			String status;
			if (countRecon == cps.size()) {
				status = "COMPLETE";
			} else {
				status = "INPROCESS";
			}
			claim.setClaimStatusCd(status);

			if (fs != null) {
				fs.setClaimStatusCd(status);
				fs.setDateModified(currentDate);
				fs.setUserName(manualReconRequest.getUserName());
				fs.setLinesClosed((long) countRecon);
				if (countRecon == 1) {
					fs.setDateInProcess(currentDate);
				}
			}
		}

		claim = claimRepository.save(claim);

		return true;
	}

	private void addClaimProcRemittanceRemark(String[] remitCode, Date currentDate, ClaimProcRecon claimProcRecon,
			BigDecimal adjAmount, String adjGroup, String adjReason, String username) {

		// String[] remitCode = manualReconRequest.getRemitCode();

		// If the Remittance Code exist for Claim Adjustment Reason Code #1,
		// process Remittance Code #1
		if (remitCode != null && remitCode.length > 0) {
			// pass array and get back ArrayList of RemittanceRemarks
			List<RemittanceRemark> remitRemarkList1 = setRemitRemarks(remitCode);

			if (remitRemarkList1 != null) {
				addClaimProcRecAdjust(currentDate, claimProcRecon, adjAmount, adjGroup, adjReason, username);

				for (RemittanceRemark rr : remitRemarkList1) {
					ClaimProcRemitRemark cprr = new ClaimProcRemitRemark();
					cprr.setCreatedBy(username);
					cprr.setDateCreated(currentDate);
					cprr.setClaimProcRecon(claimProcRecon);
					cprr.setRemittanceRemark(rr);

					claimProcRemitRemarkRepository.save(cprr);
				} // end of remitRemarkList loop
			}
		} // end if remitcode check
	}

	/**
	 * Adds the claim proc rec adjust.
	 *
	 * @param claimProcRecon
	 *            the claim proc recon
	 * @param adjAmount
	 *            the adjustment amount
	 * @param group
	 *            the group
	 * @param reason
	 *            the reason
	 */
	private void addClaimProcRecAdjust(Date currentDate, ClaimProcRecon claimProcRecon, BigDecimal adjAmount,
			String group, String reason, String username) {

		if (adjAmount == null) {
			return;
		}

		ClaimProcRecAdjust cpra = new ClaimProcRecAdjust();
		cpra.setAdjustmentAmount(adjAmount);

		AdjustmentGroup ag = adjustmentGroupRepository.findAdjustmentGroup(group);
		cpra.setAdjustmentGroup(ag);

		AdjustmentReason ar = adjustmentReasonRepository.findAdjustmentReason(reason);

		cpra.setAdjustmentReason(ar);
		cpra.setCreatedBy(username);
		cpra.setDateCreated(currentDate);

		// ClaimProcRecon - ClaimProcRecAdjust
		claimProcRecon.getClaimProcRecAdjusts().add(cpra);
		cpra.setClaimProcRecon(claimProcRecon);
	}

	private List<RemittanceRemark> setRemitRemarks(String[] remarks) {
		List<RemittanceRemark> rr = new ArrayList<RemittanceRemark>();

		if ((remarks.length == 0) || (remarks == null) || (remarks[0].toString().equalsIgnoreCase(""))) {
			rr = null;
			return rr;
		}

		for (String s : remarks) {
			RemittanceRemark r = new RemittanceRemark();
			r.setRemittanceRemarkCd(s);
			rr.add(r);
		}
		return rr;
	}

	/**
	 * Check form.
	 *
	 * @return true, if successful
	 */
	private boolean checkForm(ManualReconRequest manualReconRequest, ClaimProcedure claimProcedure)
			throws ValidationException {
		boolean status = true;

		List<ValidationMessage> validationMessages = new ArrayList<>();

		ValidationMessage validationMessage = null;

		BigDecimal total = manualReconRequest.getTotalAmount();

		BigDecimal billedAmount = claimProcedure.getBilledAmount() != null ? claimProcedure.getBilledAmount()
				: BigDecimal.ZERO;
		
		//Validate amounts if there are not adjustments
		if(manualReconRequest.getAdjustments() == null || manualReconRequest.getAdjustments().isEmpty()) {
			if (total.compareTo(billedAmount) == -1) {

					StringBuilder sb = new StringBuilder();
					sb.append(amountLessThanBilled);
					sb.append(" ");
					sb.append(CURRENCY_FORMAT.format(billedAmount.subtract(total)));
					validationMessage = new ValidationMessage("ERR", ("vaAmount"), sb.toString());
					validationMessages.add(validationMessage);
					status = false;
			}			
		}
		
		int i = 1;
		for (AdjustmentRequest adjustmentRequest : manualReconRequest.getAdjustments()) {
			String[] remitCode = adjustmentRequest.getRARCs();
			int remitCount = 0;
			if (remitCode == null || remitCode.length == 100) {
				remitCount = 0;
			} else {
				remitCount = remitCode.length;
			}
			// validate count of remarks
			if (remitCount > 2) {
				validationMessage = new ValidationMessage("ERR", ("remCode" + i), maximumSelectedRemarks);
				validationMessages.add(validationMessage);
				status = false;
			}

			// validate currency format
			BigDecimal adjAmount = adjustmentRequest.getAmount();
			if (adjAmount != null) {
				Matcher m = CURRENCY_PATTERN.matcher(adjAmount.toString());
				if (!m.matches()) {
					validationMessage = new ValidationMessage("ERR", ("adjAmount" + i), currencyFormatError);
					validationMessages.add(validationMessage);
					status = false;
				}
			}

			// Validate adjustment1
			if (i == 1) {
				if (total.compareTo(billedAmount) == -1) {
					if (adjAmount == null && adjAmount == null) {
						validationMessage = new ValidationMessage("ERR", ("adjAmount" + i), adjustmentAmountRequired);
						validationMessages.add(validationMessage);
						status = false;
					} else {
						StringBuilder sb = new StringBuilder();
						sb.append(amountLessThanBilled);
						sb.append(" ");
						sb.append(CURRENCY_FORMAT.format(billedAmount.subtract(total)));
						validationMessage = new ValidationMessage("ERR", ("adjAmount" + i), sb.toString());
						validationMessages.add(validationMessage);
						status = false;
					}
				}
			}

			if (adjAmount != null && !BigDecimal.ZERO.equals(adjAmount)) {
				if (adjustmentRequest.getGroupCode() == null) {
					validationMessage = new ValidationMessage("ERR", ("adjGroup" + i), selectionRequired);
					validationMessages.add(validationMessage);
					status = false;
				}
				if (adjustmentRequest.getCARC() == null) {
					validationMessage = new ValidationMessage("ERR", ("adjReason" + i), selectionRequired);
					validationMessages.add(validationMessage);
					status = false;
				}
			}
			i++;
		}

		if ("0".equals(manualReconRequest.getPaymentType())) {
			validationMessage = new ValidationMessage("ERR", "paymentType", selectionRequired);
			validationMessages.add(validationMessage);
			status = false;
		}

		try {
			Calendar paymentDate = Calendar.getInstance();
			paymentDate.setTime(manualReconRequest.getPaymentDate());

			if (paymentDate.get(Calendar.YEAR) < 1900) {
				validationMessage = new ValidationMessage("ERR", "paymentBatchDate", paymentDateMinDate);
				validationMessages.add(validationMessage);
				status = false;
			}
		} catch (Exception ex) {
			validationMessage = new ValidationMessage("ERR", "paymentBatchDate", invalidPaymentDate);
			validationMessages.add(validationMessage);
			status = false;
		}

		try {
			BigDecimal vaAmount = manualReconRequest.getVaAmount();
			if (vaAmount != null) {
				Matcher m = CURRENCY_PATTERN.matcher(vaAmount.toString());
				if (!m.matches()) {
					validationMessage = new ValidationMessage("ERR", "paidAmount", currencyFormatError);
					validationMessages.add(validationMessage);
					status = false;
				}
			}

			if (total.compareTo(billedAmount) == 1) {
				/*
				 * facesMessages.addFromResourceBundle(Severity.ERROR,
				 * "gov.va.med.fee.ManualRecon.AMOUNT_MORE_THAN_BILLED",
				 * CURRENCY_FORMAT.format(total.subtract(billedAmount)));
				 */
				StringBuilder sb = new StringBuilder();
				sb.append(amountMoreThanBilled);
				sb.append(" ");
				sb.append(CURRENCY_FORMAT.format(total.subtract(billedAmount)));
				validationMessage = new ValidationMessage("ERR", "billedAmount", sb.toString());
				validationMessages.add(validationMessage);
				status = false;
			}
		} catch (Exception ex) {
			status = false;
		}

		if (!status) {
			throw new ValidationException("", validationMessages);
		}

		return status;

	} // end checkForm()
}
