package gov.va.med.fee.service.impl;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import javax.persistence.NoResultException;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import gov.va.med.domain.fee.AdjustmentGroup;
import gov.va.med.domain.fee.AdjustmentReason;
import gov.va.med.domain.fee.CalcMethod;
import gov.va.med.domain.fee.Claim;
import gov.va.med.domain.fee.ClaimPayment;
import gov.va.med.domain.fee.ClaimPaymentType;
import gov.va.med.domain.fee.ClaimProcRecAdjust;
import gov.va.med.domain.fee.ClaimProcRecon;
import gov.va.med.domain.fee.ClaimProcRemitRemark;
import gov.va.med.domain.fee.ClaimProcStatus;
import gov.va.med.domain.fee.ClaimProcedure;
import gov.va.med.domain.fee.ClaimQueue;
import gov.va.med.domain.fee.ClaimReconciliation;
import gov.va.med.domain.fee.PersonInfo;
import gov.va.med.domain.fee.ProviderInfo;
import gov.va.med.domain.fee.RemittanceRemark;
import gov.va.med.domain.fee.WsFacStatus;
import gov.va.med.fee.constants.ClaimDetailsConstants;
import gov.va.med.fee.dao.IAdjustmentGroupRepository;
import gov.va.med.fee.dao.IAdjustmentReasonRepository;
import gov.va.med.fee.dao.ICalcMethodRepository;
import gov.va.med.fee.dao.IClaimByStatusCodeRepository;
import gov.va.med.fee.dao.IClaimPaymentTypeRepository;
import gov.va.med.fee.dao.IClaimProcStatusRepository;
import gov.va.med.fee.dao.IClaimProcedureRepository;
import gov.va.med.fee.dao.IClaimQueueRepository;
import gov.va.med.fee.dao.IClaimRepository;
import gov.va.med.fee.dao.IDisapproveClaimRepository;
import gov.va.med.fee.dao.IRejectedClaimRepository;
import gov.va.med.fee.dao.IRemittanceRemarkRepository;
import gov.va.med.fee.enums.AdjustmentGroupCode;
import gov.va.med.fee.enums.AdjustmentReasonCode;
import gov.va.med.fee.enums.CalcMethodCode;
import gov.va.med.fee.enums.ClaimPaymentTypeCode;
import gov.va.med.fee.enums.RemittanceRemarkCode;
import gov.va.med.fee.exceptions.GenericException;
import gov.va.med.fee.model.request.RejectedClaimRequest;
import gov.va.med.fee.model.response.AdjustmentReasonResponse;
import gov.va.med.fee.model.response.ClaimsToDisapprove;
import gov.va.med.fee.model.response.DisapproveClaims;
import gov.va.med.fee.model.response.DisapproveClaimsPage;
import gov.va.med.fee.model.response.DisapproveClaimsReason;
import gov.va.med.fee.model.response.SuccessResponse;
import gov.va.med.fee.service.IDisapproveService;

/**
 * 
 * @author vhaisbjahank
 * 
 * 
 * @Modifiedauthor Vamsi Krishna Gangarapu corrected populateAdjustmentReasons()
 *                 Added getClaimsToDisapprove() and disapproveClaim()
 *
 */

@Service
@Transactional
public class DisapproveServiceImpl implements IDisapproveService {

	private static final Logger logger = LogManager.getLogger(DisapproveServiceImpl.class);

	@Autowired
	IDisapproveClaimRepository iDisClaimRepository;

	@Autowired
	IClaimRepository claimRepository;
	
	@Autowired
	IClaimByStatusCodeRepository claimByStatusCodeRepository;

	@Autowired
	IRejectedClaimRepository rejectedClaimRepository;
	
	@Autowired
	IClaimQueueRepository claimQueueRepository;
	
	@Autowired
	IClaimPaymentTypeRepository claimPaymentTypeRepository;
	
	@Autowired
	ICalcMethodRepository calcMethodRepository;
	
	@Autowired
	IClaimProcedureRepository claimProcedureRepository;
	
	@Autowired
	IClaimProcStatusRepository claimProcStatusRepository;
	
	@Autowired
	IAdjustmentGroupRepository adjustmentGroupRepository;
	
	@Autowired
	IAdjustmentReasonRepository adjustmentReasonRepository;
	
	@Autowired
	IRemittanceRemarkRepository remittanceRemarkRepository;

	@Override
	public DisapproveClaimsPage getClaimsToDisapprove(List<Long> claimIds) throws GenericException {
		List<Long> reconciled = new LinkedList<Long>();
		List<ClaimsToDisapprove> claimsToDisapproveList = new ArrayList<ClaimsToDisapprove>();
		DisapproveClaimsPage disapproveClaimsPages = new DisapproveClaimsPage();
		for (Long claimId : claimIds) {
			try {
				ClaimsToDisapprove claimsToDisapprove;
				Claim claim = claimRepository.findClaimStatusByClaimIndex(claimId);
				if (claim != null) {
					if (!(isReconciled(claimId, claim.getClaimStatusCd()))) {
						claimsToDisapprove = new ClaimsToDisapprove();
						claimsToDisapprove.setClaimId(claimId);
						claimsToDisapprove.setBillingProvider(getBillingProviderName(claim.getProviderInfos()));
						claimsToDisapprove.setServiceProvider(getServiceProviderName(claim.getProviderInfos()));
						claimsToDisapprove.setVeteranName(getVeteranName(claim.getPersonInfos()));
						claimsToDisapproveList.add(claimsToDisapprove);
					} else {
						reconciled.add(claimId);
					}
				} else {
					claimsToDisapprove = new ClaimsToDisapprove();
					claimsToDisapprove.setMessage("No claim found for :" + claimId);
				}
			} catch (NoResultException e) {
				logger.error("error:", e);
				logger.error("unable to find claim with index {0} for disaproval", claimId);
			}

		}
		if (reconciled.size() > 0) {
			String msg = null;
			for (Long i : reconciled) {
				msg = msg == null ? i.toString() : (msg + ", " + i.toString());
			}
			String reconStr = "The following " + (reconciled.size() > 1 ? "claims" : "claim")
					+ " cannot be disapproved because " + (reconciled.size() > 1 ? "they are " : "it is")
					+ " completed, partially reconciled, or rejected: " + msg;
			ClaimsToDisapprove claimsToDisapprove = new ClaimsToDisapprove();
			claimsToDisapprove.setMessage(reconStr);
			claimsToDisapproveList.add(claimsToDisapprove);
		}

		disapproveClaimsPages.setClaimsToDisapprove(claimsToDisapproveList);

		return disapproveClaimsPages;
	}

	@Override
	public DisapproveClaimsPage populateAdjustmentReasons() throws GenericException {

		List<DisapproveClaimsReason> claimReasonsList = new LinkedList<DisapproveClaimsReason>();
		List<DisapproveClaimsReason> tmp = new LinkedList<DisapproveClaimsReason>();
		DisapproveClaimsPage disapproveClaimsPage = new DisapproveClaimsPage();

		List<AdjustmentReasonResponse> adjReasonList = new ArrayList<AdjustmentReasonResponse>();

		logger.info("DisapproveServiceImpl.populateAdjustmentReasons() - getting all the distict reason codes.");

		adjReasonList = iDisClaimRepository.populateAdjustmentReasons();

		for (AdjustmentReasonResponse reason : adjReasonList) {
			DisapproveClaimsReason claimsReason = new DisapproveClaimsReason();
			if (reason.getAdjustmentReasonCd().matches("^[A-Z].*")) {
				claimsReason.setCode(reason.getAdjustmentReasonCd());
				claimsReason.setClaimReasons(reason.getDescription());
				tmp.add(claimsReason);
			} else {
				claimsReason.setCode(reason.getAdjustmentReasonCd());
				claimsReason.setClaimReasons(reason.getDescription());
				claimReasonsList.add(claimsReason);
			}
		}

		Collections.sort(claimReasonsList, new Comparator<DisapproveClaimsReason>() {

			@Override
			public int compare(DisapproveClaimsReason o1, DisapproveClaimsReason o2) {
				//return o1.getClaimReasons().compareTo(o2.getClaimReasons());
				return o1.getCode().compareTo(o2.getCode());
			}
		});

		Collections.sort(tmp, new Comparator<DisapproveClaimsReason>() {

			@Override
			public int compare(DisapproveClaimsReason o1, DisapproveClaimsReason o2) {
				//return o1.getClaimReasons().compareTo(o2.getClaimReasons());
				return o1.getCode().compareTo(o2.getCode());
			}
		});
		claimReasonsList.addAll(tmp);

		disapproveClaimsPage.setDisapproveClaimsReasons(claimReasonsList);
		return disapproveClaimsPage;
	}

	@Override
	public DisapproveClaimsPage disapproveClaim(DisapproveClaims disapproveClaims) throws GenericException {

		List<SuccessResponse> successResponseList = new ArrayList<SuccessResponse>();
		DisapproveClaimsPage disapproveClaimsPage = null;
		try {
			for (int i = 0; i < disapproveClaims.getClaimId().size(); i++) {

				//boolean truncatedComment = false;
				// ClaimQueue claimQueue =
				// iClaimQueueRepository.findClaimByClaimIndex(disapproveClaims.getClaimId().get(i));
				Set<ClaimQueue> claimQueueSet = new HashSet<ClaimQueue>();
				ClaimQueue claimQueue = new ClaimQueue();
				claimQueue.setReason(disapproveClaims.getReason());
				claimQueue.setClaimQueueTypeCd("REJECTION_APPROVAL");
				claimQueue.setCreatedBy(disapproveClaims.getUserName());
				java.util.Date today = new java.util.Date();
				java.sql.Date sqldate = new java.sql.Date(today.getTime());
				claimQueue.setDateCreated(sqldate);
				claimQueue.setEnqueueDate(sqldate);

				/*
				 * if (disapproveClaims.getComments() != null &&
				 * disapproveClaims.getComments().length() > 250) {
				 * disapproveClaims.setComments(disapproveClaims.getComments().
				 * substring(0, 250)); truncatedComment = true;
				 */

				if (disapproveClaims.getComments() == null || disapproveClaims.getComments().length() <= 250 || disapproveClaims.getComments().isEmpty()) {
					claimQueue.setEnqueueComments(disapproveClaims.getComments());
					//Claim claim = iClaimRepository.findClaimQueueByClaimIndex(disapproveClaims.getClaimId().get(i));
					Claim claim = claimRepository.findClaimByClaimIndex(disapproveClaims.getClaimId().get(i));
					claimQueue.setClaim(claim);
					// iClaimQueueRepository.saveAndFlush(claimQueue);
					claim.setClaimStatusCd("REJECTED");
					//claim.getClaimQueues().add(claimQueue);
					claimQueueSet.add(claimQueue);
					claim.setClaimQueues(claimQueueSet);
					claim.setModifiedBy(disapproveClaims.getUserName());
					claim.setDateModified(sqldate);
					claimRepository.saveAndFlush(claim);

					try {
						Boolean value = iDisClaimRepository
								.updateWsFacStatusTable(disapproveClaims.getClaimId().get(i));
						if (value) {
							SuccessResponse successResponse = new SuccessResponse();
							successResponse.setStatus(HttpStatus.OK.toString());
							//to send the Claim number
							successResponse.setMessage((disapproveClaims.getClaimId().get(i)).toString());
							successResponseList.add(successResponse);
						} else {
							SuccessResponse successResponse = new SuccessResponse();
							successResponse.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.toString());
							//to send the claim Number
							successResponse.setMessage((disapproveClaims.getClaimId().get(i)).toString());
						}
					} catch (Exception e) {
						System.out.println(e);
					}
				} else {
					SuccessResponse successResponse = new SuccessResponse();
					successResponse.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.toString());
					//to send the claim Number
					successResponse.setMessage((disapproveClaims.getClaimId().get(i)).toString() + "has more than 250 characters in the comments");
				}
			}
		} catch (Exception e) {

		}

		disapproveClaimsPage = new DisapproveClaimsPage();
		disapproveClaimsPage.setDisapproveClaims(successResponseList);
		return disapproveClaimsPage;
	}

	private String getVeteranName(Set<PersonInfo> personInfos) {
		for (PersonInfo pi : personInfos) {
			if ("PATIENT".equals(pi.getPersonType().getPersonTypeCd())) {
				return pi.getFirstName() + " " + pi.getLastName();
			}
		}
		return null;
	}

	private String getServiceProviderName(Set<ProviderInfo> providerInfos) {
		for (ProviderInfo p : providerInfos) {
			if ("SERVICE FACILITY".equals(p.getProviderType().getProviderTypeCd())
					|| "SERVICE LOCATION".equals(p.getProviderType().getProviderTypeCd())) {
				return p.getLastName();
			}
		}
		return null;
	}

	private String getBillingProviderName(Set<ProviderInfo> providerInfos) {
		for (ProviderInfo p : providerInfos) {
			if ("BILLING PROVIDER".equals(p.getProviderType().getProviderTypeCd())
					||"REMITADDR".equals(p.getProviderType().getProviderTypeCd())) {
				return p.getLastName();
			}
		}
		return null;
	}

	final boolean isReconciled(Long claimId, String claimStatus) {

		if ("ESTABLISH".equalsIgnoreCase(claimStatus)) {
			return false;
		} else if ("INPROCESS".equalsIgnoreCase(claimStatus)) {
			BigDecimal count = iDisClaimRepository.getCount(claimId);
			// BigDecimal count = jdbcTemplate.queryForObject(sql, new Object[]
			// { claimId }, BigDecimal.class);
			if (count.intValue() > 0) {
				return true;
			} else {
				return false;
			}
		} else {
			return true;
		}
	}

	/**
	 * Perform ok disapproval action requested by the user
	 * @param rejectedClaimRequest
	 * @throws GenericException
	 */
	@Override
	public boolean okDisapprovals(
			List<RejectedClaimRequest> rejectedClaimRequestList, String userName) throws GenericException {
		// Get all the entities
		HashMap<Long, String> rejectedClaimMap = getReviewersComments(rejectedClaimRequestList);
		
		List<WsFacStatus> wsFacStatusList = claimByStatusCodeRepository.findAll(rejectedClaimMap.keySet());		
		List<Claim> claimsList = claimRepository.findAll(rejectedClaimMap.keySet());
		logger.debug("claims are updated by userName"+userName);
		String claimIds = rejectedClaimMap.keySet().toArray().toString();//Arrays.toString(rejectedClaimMap.keySet().toArray()).replaceAll("[\\[\\]]", "");
		Date date = new Date();
				
				
				// Update the WsFacStatus
				for (WsFacStatus wsFacStatus : wsFacStatusList) {
					wsFacStatus.setClaimStatusCd(ClaimDetailsConstants.COMPLETE_STATUS_CODE);
					wsFacStatus.setDateModified(date);
					Claim claim = getClaimFromList(claimsList, wsFacStatus.getClaimIndex());
					Long linesClosed = ( claim.getClaimProcedures() != null ? claim.getClaimProcedures().size() : new Long(0));
					wsFacStatus.setLinesClosed(linesClosed);
				}
				claimByStatusCodeRepository.save(wsFacStatusList);
				logger.info("Updated the WsFacStatus for claimIds "+claimIds);
				
				List<ClaimQueue> claimQueueList = claimQueueRepository.findRejectedClaimQueue(rejectedClaimMap.keySet());

				for(ClaimQueue claimQueue: claimQueueList) {
					claimQueue.setDequeueComments(rejectedClaimMap.get(claimQueue.getClaim().getClaimIndex()));
					claimQueue.setDequeueDate(date);
					claimQueue.setModifiedBy(userName);
					claimQueue.setDateModified(date);
					claimQueue.setOutcome("OK");
				}
				//save claim queue
				claimQueueRepository.save(claimQueueList);
				
				//Get Non payment claim payment type
				ClaimPaymentType claimPaymentTypeNonPayment = claimPaymentTypeRepository.findClaimPaymentTypeByClaimPaymentTypeCode(ClaimPaymentTypeCode.NON.name());
				
				//Get disapproval calc method
				CalcMethod calcMethodDisapproval = calcMethodRepository.findCalcMethod((short)CalcMethodCode.DISAPPROVAL.getCalcMethodCode());
				
				//Get rejected claim proc status
				ClaimProcStatus claimProcStatusRejected = claimProcStatusRepository.findClaimProcStatus(gov.va.med.fee.enums.ClaimProcStatus.REJECTED.name());
				
				//
				AdjustmentGroup adjustmentGroup = adjustmentGroupRepository.findAdjustmentGroup(AdjustmentGroupCode.OA.name());
				
				//Get adjustment reason Payment made to patient/insured/responsible party/employer.
				AdjustmentReason adjustmentReason = adjustmentReasonRepository.findAdjustmentReason(AdjustmentReasonCode.NINETY_SIX.getAdjustmentReasonCode());
				
				//
				RemittanceRemark remittanceRemark1 = remittanceRemarkRepository.findRemittanceRemark(RemittanceRemarkCode.N1.name());
				
				RemittanceRemark remittanceRemark2 = remittanceRemarkRepository.findRemittanceRemark(RemittanceRemarkCode.M118.name());
				
				// Update Claim
				for ( Claim claim:claimsList) {
					claim.setClaimStatusCd(ClaimDetailsConstants.COMPLETE_STATUS_CODE);
					claim.setDateModified(date);
					claim.setModifiedBy(userName);
					
	                // add claim payment data to claim
	                ClaimPayment cp = new ClaimPayment();
	                cp.setPaymentBatchDate(date);
	                cp.setClaimPaymentType(claimPaymentTypeNonPayment);
	                cp.setCreatedBy(userName);
	                cp.setDateCreated(date);
	                cp.setClaim(claim);
	                claim.getClaimPayments().add(cp);
	                
	                // add claim reconciliation data to claim
	                ClaimReconciliation cr = new ClaimReconciliation();
	                cr.setSent835Flag('N');
	                cr.setCalcMethod(calcMethodDisapproval);
	                cr.setCreatedBy(userName);
	                cr.setDateCreated(date);
	                cr.setClaim(claim);
	                claim.getClaimReconciliations().add(cr);
	                
	                //claim procedure
	                List<ClaimProcedure> claimProcedureList = claimProcedureRepository.findClaimProcedure(claim.getClaimIndex());
	                for (ClaimProcedure claimProc : claimProcedureList) {
	                    ClaimProcRecon cpr = new ClaimProcRecon();

	                    cpr.setClaim(claim);
	                    claim.getClaimProcRecons().add(cpr);

	                    cpr.setClaimProcedure(claimProc);
	                    claimProc.getClaimProcRecons().add(cpr);

	                    cpr.setClaimReconciliation(cr);
	                    cr.getClaimProcRecons().add(cpr);

	                    cpr.setClaimPayment(cp);
	                    cp.getClaimProcRecons().add(cpr);

	                    cpr.setSent835Flag('N');

	                    cpr.setBundleFlag('N');

	                    cpr.setClaimProcStatus(claimProcStatusRejected);

	                    cpr.setCreatedBy(userName);

	                    cpr.setDateCreated(date);

	                    //create procedures reconciliation adjustments
	                    ClaimProcRecAdjust cpra = new ClaimProcRecAdjust();
	                    cpra.setAdjustmentAmount(claimProcedureRepository.getBilledAmount(claimProc.getClaimProcId()));
	                    cpra.setAdjustmentGroup(adjustmentGroup);
	                    cpra.setAdjustmentReason(adjustmentReason);
	                    cpra.setCreatedBy(userName);
	                    cpra.setDateCreated(date);
	                    cpra.setClaimProcRecon(cpr);
	                    cpr.getClaimProcRecAdjusts().add(cpra);

	                    //create remit remarks
	                    ClaimProcRemitRemark cprr1 = new ClaimProcRemitRemark();
	                    cprr1.setRemittanceRemark(remittanceRemark1);
	                    cprr1.setCreatedBy(userName);
	                    cprr1.setDateCreated(date);
	                    cpr.getClaimProcRemitRemarks().add(cprr1);
	                    cprr1.setClaimProcRecon(cpr);

	                    ClaimProcRemitRemark cprr2 = new ClaimProcRemitRemark();
	                    cprr2.setRemittanceRemark(remittanceRemark2);
	                    cprr2.setCreatedBy(userName);
	                    cprr2.setDateCreated(date);
	                    cpr.getClaimProcRemitRemarks().add(cprr2);
	                    cprr2.setClaimProcRecon(cpr);

	                    // for a rejected claim all the claim procedures will be
	                    // in in the REJECTED status with the claim itself in the
	                    // COMPLETED status which is needed for 835 creation.
	                    claimProc.setLineStatus(ClaimDetailsConstants.REJECTED_STATUS_CODE);
	                }
	                
	                //Save claim procedure
	                claimProcedureRepository.save(claimProcedureList);
	                logger.info("Updated the ClaimProcedure for claimIds :"+claimIds);
				}	
				//Save claim
				claimRepository.save(claimsList);
				logger.info("Updated the Claim for claimIds :"+claimIds);

				return true;
	}	
	
	private Claim getClaimFromList(List<Claim> claimsList, long claimIndex) {
		for ( Claim claim:claimsList) {
			if(claim.getClaimIndex() == claimIndex) {
				return claim;
			}
		}
		
		return null;
	}
	
	private HashMap<Long, String> getReviewersComments(List<RejectedClaimRequest> rejectedClaimRequestList) {
		HashMap<Long, String> rejectedClaimMap = new HashMap<Long, String>();
		for(RejectedClaimRequest rejectedClaimRequest:rejectedClaimRequestList) {
			rejectedClaimMap.put(rejectedClaimRequest.getClaimId(), rejectedClaimRequest.getReviewersComments());
		}
		
		return rejectedClaimMap;
	}
	
	/**
	 * Perform reverse disapproval action requested by the user
	 * @param rejectedClaimRequest
	 * @throws GenericException
	 */
	@Override
	public boolean reverseDisapprovals(
			List<RejectedClaimRequest> rejectedClaimRequestList, String userName) throws GenericException {
	
		// Get all the entities
		HashMap<Long, String> rejectedClaimMap = getReviewersComments(rejectedClaimRequestList);
		
		List<WsFacStatus> wsFacStatusList = claimByStatusCodeRepository.findAll(rejectedClaimMap.keySet());		
		List<Claim> claimsList = claimRepository.findAll(rejectedClaimMap.keySet());
		logger.debug("claims are updated by userName"+userName);
		String claimIds = rejectedClaimMap.keySet().toString();//Arrays.toString(rejectedClaimMap.keySet().toArray()).replaceAll("[\\[\\]]", "");
		Date date = new Date();
				
				
				// Update the WsFacStatus
				for (WsFacStatus wsFacStatus : wsFacStatusList) {
					wsFacStatus.setClaimStatusCd(ClaimDetailsConstants.AWAITING_STATUS_CODE);
				}
				claimByStatusCodeRepository.save(wsFacStatusList);
				logger.info("Updated the WsFacStatus for claimIds "+claimIds);
				
				List<ClaimQueue> claimQueueList = claimQueueRepository.findRejectedClaimQueue(rejectedClaimMap.keySet());

				for(ClaimQueue claimQueue: claimQueueList) {
	                claimQueue.setDequeueComments(rejectedClaimMap.get(claimQueue.getClaim().getClaimIndex()));
	                claimQueue.setDequeueDate(date);
	                claimQueue.setOutcome(ClaimDetailsConstants.REVERSED_STATUS_CODE);
	                claimQueue.setModifiedBy(userName);
	                claimQueue.setDateModified(date);
				}
				//save claim queue
				claimQueueRepository.save(claimQueueList);

				
				// Update Claim
				for ( Claim claim:claimsList) {
					claim.setClaimStatusCd(ClaimDetailsConstants.AWAITING_STATUS_CODE);
					claim.setDateModified(date);
					claim.setModifiedBy(userName);
						                
	                //claim procedure
	                List<ClaimProcedure> claimProcedureList = claimProcedureRepository.findClaimProcedure(claim.getClaimIndex());
	                for (ClaimProcedure claimProc : claimProcedureList) {
	                    claimProc.setLineStatus(ClaimDetailsConstants.AWAITING_STATUS_CODE);
	                }
	                
	                //Save claim procedure
	                claimProcedureRepository.save(claimProcedureList);
	                logger.info("Updated the ClaimProcedure for claimIds :"+claimIds);
				}	
				//Save claim
				claimRepository.save(claimsList);
				logger.info("Updated the Claim for claimIds :"+claimIds);

				return true;
	}
}
