package gov.va.med.fee.service.impl;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.dao.DataAccessResourceFailureException;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import gov.va.med.domain.fee.Claim;
import gov.va.med.domain.fee.ClaimProcedure;
import gov.va.med.domain.fee.WsFacStatus;
import gov.va.med.fee.constants.AwaitingProcessingConstants;
import gov.va.med.fee.constants.ClaimDetailsConstants;
import gov.va.med.fee.dao.IClaimByStatusCodeRepository;
import gov.va.med.fee.dao.IClaimProcedureRepository;
import gov.va.med.fee.dao.IClaimProcedureRepositoryTemplate;
import gov.va.med.fee.dao.IClaimRepository;
import gov.va.med.fee.dao.IRejectedClaimRepository;
import gov.va.med.fee.exceptions.GenericException;
import gov.va.med.fee.model.request.ClaimRequest;
import gov.va.med.fee.model.response.ClaimDetailsResponse;
import gov.va.med.fee.model.response.ClaimLineItem;
import gov.va.med.fee.model.response.ClaimLineItems;
import gov.va.med.fee.model.response.ClaimRejectedResponse;
import gov.va.med.fee.model.response.ClaimResponse;
import gov.va.med.fee.service.IClaimDetailsService;
import gov.va.med.fee.service.IClaimService;

/**
 * 
 * @author Muneshwar Baiah
 *
 */

@Service
@Transactional
public class ClaimServiceImpl implements IClaimService {

	public static final Logger logger = LogManager.getLogger(ClaimServiceImpl.class);
	
	SimpleDateFormat dateFormat = new SimpleDateFormat("MM/dd/yyyy");

	@Autowired
	IClaimDetailsService claimDetailsService;
	
	@Autowired
	IClaimByStatusCodeRepository claimByStatusCodeRepository;
	
	@Autowired
	IClaimRepository claimRepository;
	
	@Autowired
	IClaimProcedureRepositoryTemplate claimProcedureTemplate;
	
	@Autowired
	IRejectedClaimRepository rejectedClaimRepository;
	
	@Autowired
	IClaimProcedureRepository claimProcedureRepository;

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.fee.service.IAwaitingProcessingService#
	 * getAwaitingProcessingClaims(gov.va.med.fee.model.request.
	 * AwaitingProcessingRequest)
	 */
	@Override
	public List<ClaimResponse> getClaimsByStatus(ClaimRequest claimRequest,String claimStatus)
			throws GenericException {
		List<String> activeStationList = claimRequest.getActiveStations();
		List<Long> visns = claimRequest.getVisns();
		List<String> privacyRestrictionsList = claimRequest.getPrivacyRestrictions();
		List<WsFacStatus> wsFacStatusResultSet = null;
		List<ClaimResponse> awaitingProcessingClaims = new ArrayList<ClaimResponse>();

		try {
			logger.debug(
					"getClaimsByStatus invoking the claimByStatusCodeRepository.findAwaitingProcessingClaims():");
			// Need to send the active Stations, visns and the
			// privacyRestrictions for querying
			if ( claimStatus.equals(ClaimDetailsConstants.AGED_STATUS_CODE))
				wsFacStatusResultSet = claimByStatusCodeRepository.findAgedClaims(
						ClaimDetailsConstants.COMPLETE_STATUS_CODE, activeStationList, privacyRestrictionsList, visns);
			else
			wsFacStatusResultSet = claimByStatusCodeRepository.findAwaitingProcessingClaims(
					claimStatus, activeStationList,privacyRestrictionsList , visns);
		} catch (IllegalArgumentException e) {
			// this exception is thrown by the JPA find method when the request
			// parameters are invalid
			logger.error("getClaimsByStatus invalid_input_error for " + e);
			throw new GenericException(ClaimDetailsConstants.INVALID_REQUEST, e.getMessage(), HttpStatus.BAD_REQUEST);
		} catch (DataAccessResourceFailureException e) {
			// this exception is thrown by the database connection failure
			logger.error(
					"getClaimsByStatus database_connection_error for " + e);
			throw new GenericException(ClaimDetailsConstants.DATABASE_CONNECTION_FAILURE, e.getMessage(),
					HttpStatus.INTERNAL_SERVER_ERROR);
		} catch (DataAccessException e) {
			// this is root for all the data access exceptions by spring
			// framework
			logger.error("getClaimsByStatus data_access_error for " + e);
			throw new GenericException(ClaimDetailsConstants.DATA_ACCESS_ERROR, e.getMessage(),
					HttpStatus.INTERNAL_SERVER_ERROR);
		} catch (Exception e) {
			e.printStackTrace();
			logger.error("getClaimsByStatus internal_server_error for " + e);
			throw new GenericException(ClaimDetailsConstants.INTERNAL_SERVER_ERROR, e.getMessage(),
					HttpStatus.INTERNAL_SERVER_ERROR);
		}
		if (wsFacStatusResultSet != null && !wsFacStatusResultSet.isEmpty()) {
			logger.info("getClaimsByStatus found awaiting processing claims");
			logger.debug(
					"getClaimsByStatus awaiting processing claims found are :"
							+ wsFacStatusResultSet.toString());
			logger.debug(
					"getClaimsByStatus constructing the awaiting processing claims response");
			awaitingProcessingClaims = convertToAwaitingProcessing(wsFacStatusResultSet);
		} else {
			logger.info(
					"getClaimsByStatus no awaiting processing claims found");
			throw new GenericException(AwaitingProcessingConstants.CLAIMS_NOT_FOUND,
					"No awaiting processing claims found with given inputs", HttpStatus.NOT_FOUND);
		}
		return awaitingProcessingClaims;
	}

	/**
	 * @param wsFacStatusResultSet
	 * @return List<AwaitingProcessing>
	 */
	private List<ClaimResponse> convertToAwaitingProcessing(List<WsFacStatus> wsFacStatusResultSet) {
		List<ClaimResponse> awaitingProcessingClaims = new ArrayList<ClaimResponse>();
		
		for (WsFacStatus wsFacStatus : wsFacStatusResultSet) {
			ClaimResponse awaitingClaim = new ClaimResponse();
			awaitingClaim.setClaimId(wsFacStatus.getClaimIndex());
			awaitingClaim.setClaimStatus(wsFacStatus.getClaimStatusCd());
			awaitingClaim.setClaimType(wsFacStatus.getServiceTypeCd());
			
			if( wsFacStatus.getHeroFlag() != null && (wsFacStatus.getHeroFlag().equals('1') || wsFacStatus.getHeroFlag() == '1'))
			awaitingClaim.setHero('Y');
			else
			awaitingClaim.setHero('N');
			
			awaitingClaim.setVeteranName(wsFacStatus.getPatientName());
			awaitingClaim.setSsn(wsFacStatus.getPersonId());
			awaitingClaim.setClaimDate(dateFormat.format(wsFacStatus.getDateCreated()));
			awaitingClaim.setClaimAmount(wsFacStatus.getTotalBilledCharge());
			awaitingClaim.setProviderName(wsFacStatus.getProviderName());
			awaitingClaim.setLines(wsFacStatus.getTotalLines());
			awaitingClaim.setFacility(wsFacStatus.getVaFacilityCd());
			//removed assignedTo as per change request 596357
			//awaitingClaim.setAssignedTo(wsFacStatus.getUserName());
			awaitingClaim.setVisnIdCd(wsFacStatus.getVisnIdCd());
			awaitingClaim.setDateInProcess(dateFormat.format(wsFacStatus.getDateInProcess()));
			awaitingClaim.setLinesClosed(wsFacStatus.getLinesClosed());
			
			awaitingProcessingClaims.add(awaitingClaim);
		}

		return awaitingProcessingClaims;
	}

	/**
	 * Update Claims status to the INPROCESS status for all the selected claims
	 * 
	 * @param ids
	 *            Id's of the claims that are selected to be updated
	 * @param claimStatusCd
	 *            claim status that need to be updated
	 * @return updated list of claims
	 * @throws GenericException
	 */
	@Override
	public List<ClaimResponse> updateClaimsStatus(
			ClaimRequest claimRequest, String claimStatusCd) throws GenericException {
		// Get all the entities
		List<WsFacStatus> wsFacStatusList = claimByStatusCodeRepository.findAll(claimRequest.getIds());		
		List<Claim> claimsList = claimRepository.findAll(claimRequest.getIds());
		String userName = claimRequest.getUserName();
		logger.debug("claims are updated by userName"+userName);
		String claimIds = Arrays.toString(claimRequest.getIds().toArray()).replaceAll("[\\[\\]]", "");
		Date date = new Date();
				
				// Update the WsFacStatus
			for (WsFacStatus wsFacStatus : wsFacStatusList) {
					wsFacStatus.setClaimStatusCd(claimStatusCd);
					wsFacStatus.setDateModified(date);
					wsFacStatus.setDateInProcess(date);
				}
				claimByStatusCodeRepository.save(wsFacStatusList);
				logger.info("Updated the WsFacStatus for claimIds "+claimIds);
				
				// Update Claim
				for ( Claim claim:claimsList) {
					claim.setClaimStatusCd(claimStatusCd);
					claim.setDateModified(date);
					claim.setModifiedBy(userName);
				}
				claimRepository.save(claimsList);
				logger.info("Updated the Claim for claimIds :"+claimIds);
				
				// Update Claim Procedure
				claimProcedureTemplate.updateClaimProcedureByClaimIndex(claimIds, claimStatusCd);
				logger.info("Updated the ClaimProcedure for claimIds :"+claimIds);

		List<ClaimResponse> awaitingProcessingClaims = getClaimsByStatus(claimRequest, ClaimDetailsConstants.AWAITING_STATUS_CODE);

		return awaitingProcessingClaims;
	}
		
	/**
	 * Update Claims status to the INPROCESS status for all the selected claims
	 * 
	 * @param ids
	 *            Id's of the claims that are selected to be updated
	 * @param claimStatusCd
	 *            claim status that need to be updated
	 * @return ClaimDetailsResponse
	 * @throws GenericException
	 */
	@Override
	public ClaimDetailsResponse updateClaimStatus(
			ClaimRequest claimRequest, String claimStatusCd) throws GenericException {
		// Get all the entities
		List<WsFacStatus> wsFacStatusList = claimByStatusCodeRepository.findAll(claimRequest.getIds());		
		List<Claim> claimsList = claimRepository.findAll(claimRequest.getIds());
		String userName = claimRequest.getUserName();
		logger.debug("claims are updated by userName"+userName);
		String claimIds = Arrays.toString(claimRequest.getIds().toArray()).replaceAll("[\\[\\]]", "");
		Date date = new Date();
				
				// Update the WsFacStatus
			for (WsFacStatus wsFacStatus : wsFacStatusList) {
					wsFacStatus.setClaimStatusCd(claimStatusCd);
					wsFacStatus.setDateModified(date);
					wsFacStatus.setDateInProcess(date);
				}
				claimByStatusCodeRepository.save(wsFacStatusList);
				logger.info("Updated the WsFacStatus for claimIds "+claimIds);
				
				// Update Claim
				for ( Claim claim:claimsList) {
					claim.setClaimStatusCd(claimStatusCd);
					claim.setDateModified(date);
					claim.setModifiedBy(userName);
				}
				claimRepository.save(claimsList);
				logger.info("Updated the Claim for claimId :"+claimIds);
				
				// Update Claim Procedure
				claimProcedureTemplate.updateClaimProcedureByClaimIndex(claimIds, claimStatusCd);
				logger.info("Updated the ClaimProcedure for claimIds :"+claimIds);
				
				//converting the string to long
				logger.info("Converting the claimId to Long");
				Long claimIndex = Long.valueOf(claimIds);
				logger.info("Getting the claimdetails for the claimId :"+claimIndex);
				//claimDetailsServiceImpl = new ClaimDetailsServiceImpl();
				ClaimDetailsResponse claimDetailResponse = claimDetailsService.getClaimDetailsByClaimIndex(claimIndex);
		return claimDetailResponse;
	}
	
	public List<ClaimRejectedResponse> getClaimsRejected(ClaimRequest claimRequest, String claimStatus) throws GenericException{
		List<String> activeStationList = claimRequest.getActiveStations();
		List<Long> visns = claimRequest.getVisns();
		List<String> privacyRestrictionsList = claimRequest.getPrivacyRestrictions();
		return rejectedClaimRepository.findRejectedClaims(activeStationList, visns, privacyRestrictionsList, claimStatus);
	}
	
	public ClaimLineItems getClaimLineItems(long claimId) throws GenericException{
        //claim procedure
        List<ClaimProcedure> claimProcedureList = claimProcedureRepository.findClaimProcedure(claimId);
        
        List<ClaimLineItem> claimLineItemList = convertToClaimLineItems(claimProcedureList);
        
        ClaimLineItems claimLineItems = new ClaimLineItems();
        claimLineItems.setClaimLineItems(claimLineItemList);
        return claimLineItems;
	}
	
	private List<ClaimLineItem> convertToClaimLineItems(List<ClaimProcedure> claimProcedureList) {
		List<ClaimLineItem> claimLineItems = new ArrayList<ClaimLineItem>();
		for(ClaimProcedure cp :claimProcedureList) {
			ClaimLineItem claimLineItem = new ClaimLineItem();
			claimLineItem.setClaimProcId(cp.getClaimProcId());
			claimLineItem.setBilledAmount(cp.getBilledAmount());
			claimLineItem.setBilledUnits(cp.getBilledUnits());
			claimLineItem.setDescription(cp.getProcCodeDesc());
			claimLineItem.setServiceDateFrom(dateFormat.format(cp.getServiceFromDate()));
			claimLineItem.setServiceDateTo(dateFormat.format(cp.getServiceToDate()));
			
			claimLineItem.setClaimIndex(cp.getClaim().getClaimIndex());
			claimLineItem.setClaimProcedureSeq(cp.getClaimProcedureSeq());
			claimLineItem.setProcCode(cp.getProcCode());
			claimLineItem.setProcCodeDesc(cp.getProcCodeDesc());
			claimLineItem.setRevenueCodeDesc(cp.getRevenueCodeDesc());
			claimLineItem.setPlaceOfServiceCd(cp.getPlaceOfServiceCd());
			claimLineItem.setModifier1(cp.getModifier1());
			claimLineItem.setModifier2(cp.getModifier2());
			claimLineItem.setModifier3(cp.getModifier3());
			claimLineItem.setModifier4(cp.getModifier4());
			claimLineItem.setDenyChargeAmt(cp.getDenyChargeAmt());
			claimLineItem.setEmergencyInd(cp.getEmergencyInd());
			claimLineItem.setDiagPointer(cp.getDiagPointer());
			claimLineItem.setEpsdtInd(cp.getEpsdtInd());
			claimLineItem.setFamilyPlan(cp.getFamilyPlan());
			claimLineItem.setOralCavityCd(cp.getOralCavityCd());
			claimLineItem.setProsthesisReplDate(cp.getProsthesisReplDate());
			claimLineItem.setPrescriptionNumber(cp.getPrescriptionNumber());
			claimLineItem.setLineItemControlCode(cp.getLineItemControlCode());
			claimLineItem.setPriorPlacementDate(cp.getPriorPlacementDate());
			claimLineItem.setHeroFlag(cp.getHeroFlag());
			claimLineItem.setLineStatus(cp.getLineStatus());
			claimLineItem.setCreatedBy(cp.getCreatedBy());
			claimLineItem.setDateCreated(cp.getDateCreated());
			
			claimLineItems.add(claimLineItem);
		}
		
		return claimLineItems;
	}
}
