package gov.va.med.fee.service.impl;

import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.dao.DataAccessResourceFailureException;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import gov.va.med.domain.fee.Claim;
import gov.va.med.fee.constants.ClaimDetailsConstants;
import gov.va.med.fee.dao.IClaimRepository;
import gov.va.med.fee.exceptions.GenericException;
import gov.va.med.fee.model.response.AdditionalInfoResponse;
import gov.va.med.fee.model.response.ClaimDetailsResponse;
import gov.va.med.fee.model.response.ClaimRejectionHistoryResponse;
import gov.va.med.fee.model.response.DiagnosisCodes;
import gov.va.med.fee.model.response.LineItemResponse;
import gov.va.med.fee.model.response.OhiInfoResponse;
import gov.va.med.fee.model.response.ProviderInfoResponse;
import gov.va.med.fee.model.response.ReRoutingHistory;
import gov.va.med.fee.model.response.VeteranInfoResponse;
import gov.va.med.fee.service.IAdditionalInfoService;
import gov.va.med.fee.service.IBillingProviderService;
import gov.va.med.fee.service.IClaimDetailsService;
import gov.va.med.fee.service.IDiagnosisCodeService;
import gov.va.med.fee.service.ILineItemsService;
import gov.va.med.fee.service.IOhiService;
import gov.va.med.fee.service.IProviderInfoService;
import gov.va.med.fee.service.IReRoutingHistoryService;
import gov.va.med.fee.service.IRejectionHistoryService;
import gov.va.med.fee.service.IVeteranInfoService;

/**
 * 
 * @author Muneshwar Baiah
 * @author Vamsi Krishna Gangarapu
 * 
 */

@Service
@Transactional
public class ClaimDetailsServiceImpl implements IClaimDetailsService {

	private static final Logger logger = LogManager.getLogger(ClaimDetailsServiceImpl.class);

	@Autowired
	IClaimRepository claimRepository;

	@Autowired
	IVeteranInfoService veteranInfoService;

	@Autowired
	IProviderInfoService providerInfoService;

	@Autowired
	IBillingProviderService billingProviderService;

	@Autowired
	IDiagnosisCodeService diagnosisCodeService;

	@Autowired
	IAdditionalInfoService additionalInfoService;

	@Autowired
	IReRoutingHistoryService reRoutingHistoryService;

	@Autowired
	ILineItemsService lineItemsService;

	@Autowired
	IRejectionHistoryService iRejectionHistoryService;
	
	@Autowired
	IOhiService iOhiService;

	/**
	 * This method queries the database using the spring data JPA and constructs
	 * the appropriate response.
	 * 
	 * @param claimIndex
	 * @return claimDetailsResponse
	 * @throws GenericException
	 */
	@Override
	public ClaimDetailsResponse getClaimDetailsByClaimIndex(Long claimIndex) throws GenericException {
		logger.info("ClaimDetailsServiceImpl.getClaimDetailsByClaimIndex() getting claim details for claim index :"
				+ claimIndex);
		ClaimDetailsResponse claimDetailsResponse = null;
		VeteranInfoResponse veteranInfo = null;
		ProviderInfoResponse serviceProviderInfo = null;
		ProviderInfoResponse billingProviderInfo = null;
		ProviderInfoResponse payToProviderInfo = null;
		List<DiagnosisCodes> diagnosisCodeResponse = null;
		AdditionalInfoResponse additionalInfoResponse = null;
		List<ReRoutingHistory> reRoutingHistory = null;
		List<LineItemResponse> lineItemsInfo = null;
		List<ClaimRejectionHistoryResponse> rejectionHistory = null;
		List<OhiInfoResponse> ohiInfo = null;
		Claim claim = null;
		try {
			logger.debug(
					"ClaimDetailsServiceImpl.getClaimDetailsByClaimIndex() invoking the claimRepository.findone() for claim index :"
							+ claimIndex);
			// claim = claimRepository.findOne(claimIndex);
			claim = claimRepository.findByClaimIndex(claimIndex);
		} catch (IllegalArgumentException e) {
			// this exception is thrown by the JPA find method when the
			// claimIndex is invalid format or null
			logger.error("ClaimDetailsServiceImpl.getClaimDetailsByClaimIndex() invalid_input_error for " + claimIndex
					+ " " + e);
			throw new GenericException(ClaimDetailsConstants.INVALID_REQUEST, e.getMessage(), HttpStatus.BAD_REQUEST);
		} catch (DataAccessResourceFailureException e) {
			// this exception is thrown by the database connection failure
			logger.error("ClaimDetailsServiceImpl.getClaimDetailsByClaimIndex() database_connection_error for "
					+ claimIndex + " " + e);
			throw new GenericException(ClaimDetailsConstants.DATABASE_CONNECTION_FAILURE, e.getMessage(),
					HttpStatus.INTERNAL_SERVER_ERROR);
		} catch (DataAccessException e) {
			// this is root for all the data access exceptions by spring
			// framework
			logger.error("ClaimDetailsServiceImpl.getClaimDetailsByClaimIndex() data_access_error for " + claimIndex
					+ " " + e);
			throw new GenericException(ClaimDetailsConstants.DATA_ACCESS_ERROR, e.getMessage(),
					HttpStatus.INTERNAL_SERVER_ERROR);
		} catch (Exception e) {
			logger.error("ClaimDetailsServiceImpl.getClaimDetailsByClaimIndex() internal_server_error for " + claimIndex
					+ " " + e);
			throw new GenericException(ClaimDetailsConstants.INTERNAL_SERVER_ERROR, e.getMessage(),
					HttpStatus.INTERNAL_SERVER_ERROR);
		}

		if (claim != null) {

			veteranInfo = veteranInfoService.getVeteranInfo(claim);
			serviceProviderInfo = providerInfoService.getServiceProviderResponseInfo(claim);
			billingProviderInfo = billingProviderService.getBillingProviderResponse(claim);
			payToProviderInfo = providerInfoService.getPayToProviderResponseInfo(claim);
			diagnosisCodeResponse = diagnosisCodeService.getDiagnosisCodes(claim);
			additionalInfoResponse = additionalInfoService.getAdditionalInfoDetails(claim);
			reRoutingHistory = reRoutingHistoryService.getReRoutingHistoryDetails(claim);
			lineItemsInfo = lineItemsService.getClaimLineItems(claim);
			rejectionHistory = iRejectionHistoryService.getRejectionHistory(claim);
			ohiInfo = iOhiService.getOhiInfo(claim);
			

			claimDetailsResponse = new ClaimDetailsResponse();
			logger.info(
					"ClaimDetailsServiceImpl.getClaimDetailsByClaimIndex() claim found for claim index :" + claimIndex);
			logger.debug("ClaimDetailsServiceImpl.getClaimDetailsByClaimIndex() claimId :" + claim.getClaimIndex()
					+ ", claimStatus :" + claim.getClaimStatusCd() + ", patientControlNumber :"
					+ claim.getPatientControlNumber() + " preAuthorization :" + claim.getPriorAuthNbr());
			logger.debug(
					"ClaimDetailsServiceImpl.getClaimDetailsByClaimIndex() constructing the claim details response");

			// Constructing the claims details response
			claimDetailsResponse.setClaimId(claim.getClaimIndex());
			claimDetailsResponse.setClaimStatus(claim.getClaimStatusCd());
			claimDetailsResponse.setPatientControlNumber(claim.getPatientControlNumber());
			claimDetailsResponse.setPreAuthorization(claim.getPriorAuthNbr());
			claimDetailsResponse.setClaimType(claim.getServiceType().getServiceTypeCd());
			claimDetailsResponse.setVaImpTypeCd(claim.getVaImpTypeCd());
			claimDetailsResponse.setVeteranInfo(veteranInfo);
			claimDetailsResponse.setServiceProviderInfo(serviceProviderInfo);
			claimDetailsResponse.setBillingProviderInfo(billingProviderInfo);
			claimDetailsResponse.setPayToProviderInfo(payToProviderInfo);
			claimDetailsResponse.setDiagnosisCodes(diagnosisCodeResponse);
			claimDetailsResponse.setAdditionalInfo(additionalInfoResponse);
			claimDetailsResponse.setReRoutingHistory(reRoutingHistory);
			claimDetailsResponse.setRejectionHistory(rejectionHistory);
			claimDetailsResponse.setLineItemsInfo(lineItemsInfo);
			claimDetailsResponse.setOhiInfo(ohiInfo);

			logger.debug("ClaimDetailsServiceImpl.getClaimDetailsByClaimIndex() returning the claimDetailsResponse :"
					+ claimDetailsResponse);
		} else {
			logger.info("ClaimDetailsServiceImpl.getClaimDetailsByClaimIndex() claim_not_found for " + claimIndex);
			throw new GenericException(ClaimDetailsConstants.CLAIM_NOT_FOUND,
					"No claim found with claim id " + claimIndex, HttpStatus.NOT_FOUND);
		}
		return claimDetailsResponse;
	}
}
