/**
 *  @author Eyuel Taddese
 *
 */
package gov.va.med.fee.service.impl;

import java.net.URLEncoder;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.dao.DataAccessException;
import org.springframework.dao.DataAccessResourceFailureException;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import gov.va.med.domain.fee.AppUser;
import gov.va.med.domain.fee.SsoAuthentication;
import gov.va.med.domain.fee.UserRoleUsages;
import gov.va.med.fee.dao.IAppUserRepository;
import gov.va.med.fee.dao.IAuthenticatorRepository;
import gov.va.med.fee.dao.IUserFacilityDataRepository;
import gov.va.med.fee.dao.IUserRolePermissionsRepository;
import gov.va.med.fee.exceptions.GenericException;
import gov.va.med.fee.model.response.AuthenticationParamResponse;
import gov.va.med.fee.model.response.UserInfoResponse;
import gov.va.med.fee.model.response.UserRolePermissions;
import gov.va.med.fee.model.response.UserRoles;
import gov.va.med.fee.model.response.UserStationDetails;
import gov.va.med.fee.service.IAuthenticatorService;
import gov.va.med.fee.util.IDCheckAESEncryption;
import net.minidev.json.JSONObject;

@Service
public  class AuthenticatorServiceImpl implements IAuthenticatorService {
	
	private static final Logger logger = LogManager.getLogger(AuthenticatorServiceImpl.class);
	
	@Autowired 
	IAuthenticatorRepository authenticatorRepository;
	
	@Autowired
	IUserFacilityDataRepository userFacilityRepository;
	
	@Autowired
	IAppUserRepository appUserRepository;
	
	@Autowired
	IUserRolePermissionsRepository userRolePermissionRepository;
	
	@Value("${postBack.url}")
    private String postBackUrl;
    
    @Value("${captured.url}")
    private String capturedUrl;
    
    @Value("${aspCheckPage.url}")
    private String aspCheckPageUrl;
    
	private boolean isAuthenticated = false;
	
	private List<AppUser> appUsers;
	private List<UserRoles> userRoles;
	private AppUser appUser;
	
	/**
	 * 
	 */
	public AuthenticatorServiceImpl() {
		appUser = new AppUser();
		appUsers = new ArrayList<>();
		userRoles = new ArrayList<>();
	}
	
	/**
	 * @param authenticatorRepository
	 */
	public void setAuthenticatorDAO(IAuthenticatorRepository authenticatorRepository) {
		this.authenticatorRepository = authenticatorRepository;
	}
	
	/**
	 * @param appUserRepository
	 */
	public void setAppUserRoleRepository(IAppUserRepository appUserRepository) {
		this.appUserRepository = appUserRepository;
	}
	
	/**
	 * The method accept GUID and query from SsoAuthentication table
	 * and return the UserName. I used JDBCTemplate and manually mapped the result to the object
	 * @param guid
	 * @return String
	 */
	@Override
	public String getUsernameFromAspAuth(String guid) throws GenericException {
		//For testing only
		//String guid1 = "DNS   TADDEE";
		try {
			String userName = null;
			List<SsoAuthentication> results = authenticatorRepository.findByGuid(guid);
			
			if(results.size() != 0) {
				SsoAuthentication s = results.get(0);
				userName = s.getNtname();
				logger.info("getUsernameFromAspAuth() Retrieved UserName: "+userName +" from Asp Auth");
			}
			return userName;
		} catch(IllegalArgumentException e) {
			// this exception is thrown by the JPA find method when the claimIndex is invalid format or null
			logger.info("authenticate : invalid_input_error for "+guid+" "+e.getMessage());
			throw new GenericException("invalid_input_error",e.getMessage(), HttpStatus.BAD_REQUEST);	
		} catch(DataAccessResourceFailureException e) {
			// this exception is thrown by the database connection failure
			logger.info("authenticate  : database_connection_error "+e.getMessage());
			throw new GenericException("database_connection_error",e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);		
		} catch(DataAccessException e){
			// this is root for all the data access exceptions by spring framework
			logger.info("authenticate : data_access_error "+e.getMessage());
			throw new GenericException("data_access_error", e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);	
		}  catch(Exception e) {
			logger.info("authenticate : internal_server_error  "+e.getMessage());
			throw new GenericException("internal_server_error", e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);
		}
	}
	/**
	 * This method will authenticate GUID of User and get AppUser
	 * @param guid
	 * @return boolean
	 */
	@Override
	public final boolean authenticate(String guid) throws GenericException {
		logger.info("autoLogin : Performing AutoLogin");
		try {
			logger.info(" guid from the request is :"+guid);
			String decryptedGuid = IDCheckAESEncryption.decrypt(guid);
			logger.info("decryptedGuid is :"+decryptedGuid);
			String userName = getUsernameFromAspAuth(decryptedGuid);
			if(userName != null) {
				// Get AppUser using userName
				 appUsers = appUserRepository.findByAppUserName(userName);
				 isAuthenticated = true;
			} else {
				logger.error("authenticate(): could not obtain userName from ASP. Authentication module error ");
				isAuthenticated = false;
				throw new GenericException("Asp.Authentication Failed","Could not obtain userName for guid:"+guid+" from ASP Authentication module",HttpStatus.BAD_REQUEST);
			}	
			return isAuthenticated;
		}  catch(IllegalArgumentException e) {
			// this exception is thrown by the JPA find method when the claimIndex is invalid format or null
			logger.info("authenticate : invalid_input_error for "+guid+" "+e.getMessage());
			throw new GenericException("invalid_input_error",e.getMessage(), HttpStatus.BAD_REQUEST);	
		} catch(DataAccessResourceFailureException e) {
			// this exception is thrown by the database connection failure
			logger.info("authenticate  : database_connection_error "+e.getMessage());
			throw new GenericException("database_connection_error",e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);		
		} catch(DataAccessException e){
			// this is root for all the data access exceptions by spring framework
			logger.info("authenticate : data_access_error "+e.getMessage());
			throw new GenericException("data_access_error", e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);	
		} catch(GenericException e ) {
			throw e;
		} catch(Exception e) {
			logger.info("authenticate : internal_server_error  "+e.getMessage());
			throw new GenericException("internal_server_error", e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);
		}
		
	}
	
	/**
	 * Update the lastlogin of AppUser
	 * @param appuser
	 */
	private void updateLastLogin(AppUser appuser) throws GenericException {
		try {
			logger.info("UpdateLastLogin() : Updating userLastLogin for "+appuser.getUserName());
			appuser.setLastLoginDate(new Date());
			appUserRepository.save(appuser);
			
		} catch(DataAccessResourceFailureException e) {
			// this exception is thrown by the database connection failure
			logger.info("updateLastLogin  : database_connection_error "+e.getMessage());
			throw new GenericException("database_connection_error",e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);		
		} catch(DataAccessException e){
			// this is root for all the data access exceptions by spring framework
			logger.info("updateLastLogin : data_access_error "+e.getMessage());
			throw new GenericException("data_access_error", "Unable to update user lastLoginDate", HttpStatus.INTERNAL_SERVER_ERROR);	
		} catch(Exception e) {
			logger.info("updateLastLogin : internal_server_error  "+e.getMessage());
			throw new GenericException("internal_server_error", e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);
		}
	}
	
	/**
	 * This method return list of Roles assigned 
	 * @param guid
	 * @return List<UserRoles>
	 */
	@Override
	public UserInfoResponse getUserRoles() throws GenericException {
		logger.info( "Getting user role details ");
		UserInfoResponse userInfoResponse = null;
		Set<String> roles = new HashSet<String>();
		Set<Long> assignedRoles = null;
		try {
			if(isAuthenticated) {
				//AppUser appUser = new AppUser();
				this.userRoles = new ArrayList<>();
				if(appUsers.size() > 0) {
					appUser = appUsers.get(0);
					
					//Get user who is enabled
					if(appUser.getEnabled().equalsIgnoreCase("Y")) {
						Set<UserRoleUsages> userRoleUsages = appUser.getUserRoleUsages();
						// Update App_User LastLoginDate
						updateLastLogin(appUser);
						
						if(userRoleUsages.size() > 0) {
							assignedRoles = new HashSet<Long>();
							// Add the role name description and send it to the controller
							for(UserRoleUsages u : userRoleUsages) {
								UserRoles userRole = new UserRoles();
								userRole.setRoleName(u.getAppRole().getRoleName());
								userRole.setDescription(u.getAppRole().getDescription());
								userRoles.add(userRole);
								
								// add the user roles to the set
								assignedRoles.add(u.getAppRole().getAppRoleId());
								roles.add(u.getAppRole().getRoleName());
							}
						} else {
							logger.error("getUserRoles(): No Role found for User "+appUser.getUserName());
							throw new GenericException("UserRole Error","No UserRole found for "+appUser.getUserName(),HttpStatus.BAD_REQUEST);
						}
						
						userInfoResponse = new UserInfoResponse();
						userInfoResponse.setUserName(appUser.getUserName());
						
						SimpleDateFormat dateFormat = new SimpleDateFormat("MM/dd/yyyy HH:mm:ss");
						
						userInfoResponse.setLastLogin(dateFormat.format(appUser.getLastLoginDate()));
						userInfoResponse.setUserRoles(roles);
						
						// set permissions of the user
						logger.debug("Setting permissions of the user");
						userInfoResponse.setPermissions(getPermissions(assignedRoles));
						logger.debug("Invoking the getUserStationsDetails() ");
						UserStationDetails userStationDetails = userFacilityRepository.getUserStationsDetails(appUser.getUserName(),userRoleUsages);
						logger.debug("The roles of the user are :"+userInfoResponse.toString());
						userInfoResponse.setUserFacilityData(userStationDetails);
					}
				}
				
				return userInfoResponse;
			}
			else {
				logger.error("getUserRoles(): GUID is not Authenticated ");
			}
			return userInfoResponse;
		} 	catch(GenericException e) {
			logger.info("updateLastLogin : internal_server_error  "+e.getMessage());
			throw new GenericException("internal_server_error", e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);
		}   catch(Exception e) {
			logger.info("updateLastLogin : internal_server_error  "+e.getMessage());
			throw new GenericException("internal_server_error", e.getMessage(), HttpStatus.INTERNAL_SERVER_ERROR);
		}

	}
		
	/* (non-Javadoc)
	 * @see gov.va.med.fee.service.IAuthenticatorService#getAuthParams(java.lang.String)
	 */
	public AuthenticationParamResponse getAuthParams(String sessionId) throws GenericException{
		logger.info("getAuthParams() :"+sessionId);
	
		String aspCheckPageUrl= this.aspCheckPageUrl;
		String encryptedPostBackUrl = null;
		String encryptedCapturedUrl = null;
		try {
			encryptedPostBackUrl = URLEncoder.encode(IDCheckAESEncryption.encrypt(this.postBackUrl), "UTF-8");
			encryptedCapturedUrl = URLEncoder.encode(IDCheckAESEncryption.encrypt(this.capturedUrl), "UTF-8");
		} catch (Exception e) {
			logger.error("Exception while encripting urls"+e.getMessage());
			throw new GenericException("getAuthParamsFailed","Retrieve Authentication Params Failed",HttpStatus.INTERNAL_SERVER_ERROR);
		}
			
		AuthenticationParamResponse response = new AuthenticationParamResponse();
		response.setEncryptedCapturedUrl(encryptedCapturedUrl);
		response.setEncryptedPostBackUrl(encryptedPostBackUrl);
		response.setSessionId(sessionId);
		response.setAspCheckPageUrl(aspCheckPageUrl);
		
		return response;
	}

	/**
	 * @return the appUsers
	 */
	public List<AppUser> getAppUsers() {
		return appUsers;
	}

	/**
	 * @return the appUser
	 * @throws GenericException 
	 */
	public AppUser getAppUser(String decryptedGuid) throws GenericException {
		String userName = getUsernameFromAspAuth(decryptedGuid);
		if (userName != null) {
			// Get AppUser using userName
			appUsers = appUserRepository.findByAppUserName(userName);
			if (appUsers.size() > 0) {
				appUser = appUsers.get(0);
			}
		}

		return appUser;
	}
	


	
	/**
	 * @param assignedRolesSet
	 * @return
	 */
	public JSONObject getPermissions(Set<Long> assignedRolesSet) {
		
		Set<String> assignedRolesString = new HashSet<String>();
		if ( assignedRolesSet != null ) {
			assignedRolesSet.forEach(role -> {
				if( role != null ) {
					assignedRolesString.add(role.toString());
				}
			});
		}
		
		logger.debug(" assignedRolesString is :"+assignedRolesString);
		String roleIds = String.join(",", assignedRolesString);
		logger.debug(" roleIds is :"+roleIds);
		List<UserRolePermissions> userRolePermissions = userRolePermissionRepository.getUserRolePermissions(roleIds);
		
		logger.debug(" userRolePermissions from the repository are : ", userRolePermissions.toString());
		
		HashSet<String> pages = new HashSet<String>();
		Map<String, Object> permissions = new HashMap<String, Object>();
		Map<String,Boolean> pagePermissions = new HashMap<String,Boolean>();
		
		for( UserRolePermissions userRolePermission : userRolePermissions ) {
			pages.add(userRolePermission.getPageName());
		}
		
		logger.debug(" List of the pages from the repository are : "+pages.toString());
		
		for( String page : pages ) {
			logger.debug(" Getting the permissions for the page : "+page);
			List<UserRolePermissions> filteredPermissions = userRolePermissions.stream().filter( permission -> permission.getPageName().equals(page)).collect(Collectors.toList());
			logger.debug(" filteredPermissions are : "+filteredPermissions);
			pagePermissions = new HashMap<String,Boolean>();
			
			for( UserRolePermissions filtered : filteredPermissions ) {	
				// when multiple roles has the same permissions,the below condition will set the permission as enabled if one of the role has it as enabled
				if( !(pagePermissions.containsKey(filtered.getPermissionName())) ||
						(pagePermissions.containsKey(filtered.getPermissionName()) && !(pagePermissions.get(filtered.getPermissionName()))))
				pagePermissions.put(filtered.getPermissionName(), filtered.getEnabled());
				else
				pagePermissions.put(filtered.getPermissionName(), Boolean.TRUE);
			}
			
			logger.debug(" Permissions for the page : "+page+" are :"+pagePermissions.toString());
			permissions.put( page, pagePermissions);
		}
	    
	    JSONObject permissionsJson = new JSONObject();
	    permissionsJson.putAll( permissions );
	    
	    logger.debug("Permissions of the user are : "+permissionsJson.toString());
	    return permissionsJson;
	}
	

	/**
	 * This method is introduced to retrieve the updated station count details for a logged in user
	 * @param userName
	 * @return
	 * @throws GenericException
	 */
	public UserStationDetails getUpdatedUserStationDetails(String userName) throws GenericException {
		logger.debug("Getting the UserStationDetails for userName :",userName);
		
		if(userName != null) {
			// Get AppUser using userName
			 appUsers = appUserRepository.findByAppUserName(userName);
		} else {
			logger.error("authenticate(): could not obtain details for userName : "+userName);
			throw new GenericException("Error getting UserStationDetails for :"+userName,"No User Found with userName :"+userName, HttpStatus.BAD_REQUEST);
		}	
		
		appUser = appUsers.get(0);
		Set<UserRoleUsages> userRoleUsages = appUser.getUserRoleUsages();		
		UserStationDetails userStationDetails = userFacilityRepository.getUserStationsDetails(userName,userRoleUsages);
		logger.debug("UserStationDetails obtained for the user are : "+userStationDetails);
		return userStationDetails;
	}
}

