package gov.va.med.fee.dao.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;

import javax.sql.DataSource;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.stereotype.Repository;

import gov.va.med.fee.dao.ICodeLookupRepository;
import gov.va.med.fee.model.response.CodeLookupResponse;

/**
 * @author Vamsi Krishna Gangarapu
 *
 */
@Repository
public class MedicalCodeLookupRepositoryImpl implements ICodeLookupRepository {

	public static final Logger logger = LogManager.getLogger(MedicalCodeLookupRepositoryImpl.class);
	private JdbcTemplate jdbcTemplate;

	/**
	 * @param dataSource
	 */
	@Autowired
	public void setJdbcTemplate(DataSource dataSource) {
		this.jdbcTemplate = new JdbcTemplate(dataSource);
	}


	/* (non-Javadoc)
	 * @see gov.va.med.fee.dao.ICodeLookupRepository#getCodes(java.lang.String, java.lang.String, java.lang.Integer, java.lang.String, java.lang.Integer)
	 */
	@Override
	public List<CodeLookupResponse> getCodes(String medicalCode, String codeType, Integer pageSize,
			String sortColumn, Boolean descending, Integer pageNumber)  {
		logger.info("getCodes() : Retreiving the medical codes from the database");
		List<CodeLookupResponse> medicalCodeResponseList = new LinkedList<CodeLookupResponse>();

		try {
			StringBuilder sqlQuery = getMedicalCodesSqlQuery(medicalCode, codeType, pageSize, sortColumn,descending,
					pageNumber);
			logger.debug("getCodes() : Query used to get the medical codes is : "+sqlQuery);
			medicalCodeResponseList = jdbcTemplate.query(sqlQuery.toString(),
					new ResultSetExtractor<List<CodeLookupResponse>>() {

				@Override
				public List<CodeLookupResponse> extractData(ResultSet arg0)
						throws SQLException, DataAccessException {
					List<CodeLookupResponse> medicalCodeResponseList = new ArrayList<CodeLookupResponse>();
					CodeLookupResponse medicalCodeResponse = null;

					while (arg0.next()) {
						medicalCodeResponse = new CodeLookupResponse();

						medicalCodeResponse.setCodeType(arg0.getString("MEDICAL_CODE_TYPE"));
						medicalCodeResponse.setDrugExempt(arg0.getString("DRG_EXEMPT_FLAG"));												
						medicalCodeResponse.setCode(arg0.getString("MEDICAL_CD"));
						medicalCodeResponse.setDescription(arg0.getString("DESCRIPTION"));

						try {
							Date date = null;
							String formattedDate = null;
							if ( arg0.getString("EXPIRATION_DATE") != null ) {
								date = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS").parse(arg0.getString("EXPIRATION_DATE"));
								formattedDate = new SimpleDateFormat("MM/dd/yyyy").format(date);
								medicalCodeResponse.setExpirationDate(formattedDate);
							}

							if ( arg0.getString("START_DATE") != null ) {
								date = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS").parse(arg0.getString("START_DATE"));
								formattedDate = new SimpleDateFormat("MM/dd/yyyy").format(date);
								medicalCodeResponse.setStartDate(formattedDate);
							}
						} catch (ParseException e) {
							logger.error("Date formatting exception occured :"+e.getMessage());
						}

						medicalCodeResponseList.add(medicalCodeResponse);
					}
					return medicalCodeResponseList;
				}
			});
		} catch (Exception e) {
			logger.debug("getCodes() : Exception occured :"+e);
			e.printStackTrace();
			throw e;
		}

		logger.debug("getCodes() : medicalCodeResponseList is "+medicalCodeResponseList);
		return medicalCodeResponseList;

	}

	private StringBuilder getMedicalCodesSqlQuery(String medicalCode, String codeType, Integer pageSize,
			String sortColumn, Boolean descending, Integer pageNumber) {

		StringBuilder sql = new StringBuilder();
		StringBuilder sqlQuery = new StringBuilder();

		StringBuilder sortByCriteria = new StringBuilder();
		sortByCriteria.append(" order by " + getMedicalCodeTableColumnName(sortColumn));

		if ( descending == null || descending == false ) {
			sortByCriteria.append(" ASC "); 
		} else {
			sortByCriteria.append(" DESC");
		}

		sql.append("select medicalCode.*, ROW_NUMBER() OVER ("+sortByCriteria+") r from MEDICAL_CODE medicalCode ");

		if (medicalCode != null && !(medicalCode.equalsIgnoreCase(null))) {
			sql.append(" WHERE MEDICAL_CD like "+"'"+medicalCode.toUpperCase()+'%'+"'");
		}

		if (medicalCode != null && codeType != null && !(medicalCode.equalsIgnoreCase(null)) && !(codeType.equalsIgnoreCase(null))) {

			sql.append(" AND MEDICAL_CODE_TYPE like "+"'"+codeType.toUpperCase()+'%'+"'");
		} else {
			if(codeType != null && !(codeType.equalsIgnoreCase(null))) {
				sql.append(" WHERE MEDICAL_CODE_TYPE like "+"'"+codeType.toUpperCase() + '%'+"'");
			}
		}



		if ((pageSize != null) && (pageNumber != null) && pageNumber >= 1) {

			int currentUpperLimit = pageSize * pageNumber;
			int currentLowerLimit = currentUpperLimit - pageSize;
			sqlQuery.append(
					"select * from (" + sql + ") where r >" + currentLowerLimit + " AND r<=" + currentUpperLimit);

		}
		return sqlQuery;

	}

	@Override
	public Long getCount(String medicalCode, String codeType, Integer pageSize,
			String sortColumn, Boolean descending, Integer pageNumber) {

		StringBuilder sql = new StringBuilder();
		sql.append("select count(*) from MEDICAL_CODE medicalCode ");

		if (medicalCode != null && !(medicalCode.equalsIgnoreCase(null))) {
			sql.append(" WHERE MEDICAL_CD like "+"'"+medicalCode.toUpperCase()+'%'+"'");
		}

		if (medicalCode != null && codeType != null && !(medicalCode.equalsIgnoreCase(null)) && !(codeType.equalsIgnoreCase(null))) {

			sql.append(" AND MEDICAL_CODE_TYPE like "+"'"+codeType.toUpperCase()+'%'+"'");
		} else {
			if(codeType != null && !(codeType.equalsIgnoreCase(null))) {
				sql.append(" WHERE MEDICAL_CODE_TYPE like "+"'"+codeType.toUpperCase() + '%'+"'");
			}
		}

		sql.append(" order by " + getMedicalCodeTableColumnName(sortColumn));

		if ( descending == null || descending == false ) {
			sql.append(" ASC "); 
		} else {
			sql.append(" DESC");
		}

		Long count = jdbcTemplate.queryForObject(sql.toString(), Long.class);
		return count;
	}

	/**
	 * @param param
	 * @return
	 */
	String getMedicalCodeTableColumnName(String param) {
		if ( param != null ) {
			switch( param ) {
			case "codeType":
				return "MEDICAL_CODE_TYPE";
			case "code":
				return "MEDICAL_CD";
			case "description":
				return "DESCRIPTION";
			case "drugExempt":
				return "DRG_EXEMPT_FLAG";
			case "startDate":
				return "START_DATE";
			case "expirationDate":
				return "EXPIRATION_DATE";
			default:
				return "MEDICAL_CD_ID";
			} }
		else {
			return "MEDICAL_CD_ID";
		}
	}


}
