package gov.va.med.fee.controller;

import java.util.ArrayList;
import java.util.List;

import javax.validation.Valid;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import gov.va.med.fee.constants.ClaimDetailsConstants;
import gov.va.med.fee.exceptions.GenericException;
import gov.va.med.fee.model.request.ContactInfoRequest;
import gov.va.med.fee.model.request.VaFacilityRequest;
import gov.va.med.fee.model.request.VaFacilitySearchRequest;
import gov.va.med.fee.model.request.ZipCodeRequest;
import gov.va.med.fee.model.response.GenericResponse;
import gov.va.med.fee.model.response.VaFacilities;
import gov.va.med.fee.model.response.VaFacilityResponse;
import gov.va.med.fee.model.response.VaFacilityTypeResponse;
import gov.va.med.fee.model.response.VaFacilityVisnResponse;
import gov.va.med.fee.service.IVaFacilityService;
import net.minidev.json.JSONObject;

@RestController
@RequestMapping("/api/v1/VaFacility")
public class VaFacilityController {
	private static final Logger logger = LogManager.getLogger(VaFacilityController.class);

	@Autowired
	IVaFacilityService vafacilityService;

	/*
	 * This REST API can be used to add a facility Success: Return
	 * HttpStatus.CREATED (201) - a response to successful POST requests Fail:
	 * Return HttpStatus.BAD_REQUEST (400) - The server cannot meet the
	 * requirements
	 */
	@RequestMapping(value = "/station", method = RequestMethod.POST)
	public ResponseEntity<JSONObject> addFacility(@RequestBody VaFacilityRequest request) throws GenericException {
		List<String> error_description = validationError(request);

		if (!error_description.isEmpty()) {
			throw new GenericException("Add a new facility request error", error_description.toString(),
					HttpStatus.BAD_REQUEST);

		} else {
			Integer rowCount = new Integer(0);
			JSONObject entity = new JSONObject();

			try {
				rowCount = vafacilityService.addVaFacility(request);

			} catch (GenericException e) {
				logger.debug("Unable to Add VA facility");
				throw e;
			}

			if (rowCount > 0) {
				entity.put("result", rowCount);

			} else {
				throw new GenericException("Eror", "Unable to Add VA facility", HttpStatus.BAD_REQUEST);
			}
			return new ResponseEntity<>(entity, HttpStatus.CREATED);
		}
	}

	/**
	 * @author Vamsi Krishna Gangarapu
	 * @return
	 */
	// to be done in the next sprint
	@RequestMapping(value = "/station", method = RequestMethod.PUT)
	public ResponseEntity<JSONObject> modifyFacility(@RequestBody VaFacilityRequest request) throws GenericException {
		List<String> error_description = validationError(request);

		if (!error_description.isEmpty()) {
			throw new GenericException("Modify a new ficility request error", error_description.toString(),
					HttpStatus.BAD_REQUEST);

		} else {
			boolean rowCount = false;
			JSONObject entity = new JSONObject();

			try {
				rowCount = vafacilityService.modifyStation(request);

			} catch (GenericException e) {
				logger.debug("Unable to Modify VA facility");
				throw e;
			}

			if (rowCount) {
				entity.put("result", rowCount);

			} else {
				throw new GenericException("Eror", "Unable to Modify VA facility", HttpStatus.BAD_REQUEST);
			}
			return new ResponseEntity<>(entity, HttpStatus.OK);
		}

	}

	/*
	 * This REST call gets all Visn ID Code
	 * 
	 * @return a list of Visn ID Code
	 * 
	 * @throws GenericException
	 */
	@RequestMapping(value = "/station/visn", method = RequestMethod.GET)
	public ResponseEntity<List<VaFacilityVisnResponse>> getAllVisn() throws GenericException {
		List<VaFacilityVisnResponse> response = new ArrayList<VaFacilityVisnResponse>();

		try {
			response = vafacilityService.getAllVisns();

		} catch (GenericException e) {

			logger.debug("Unable to retrieve Visn");
			throw e;
		}
		return new ResponseEntity<List<VaFacilityVisnResponse>>(response, HttpStatus.OK);
	}

	/*
	 * This REST call gets all VA Facility Type
	 * 
	 * @return a list of facility type
	 * 
	 * @throws GenericException
	 */
	@RequestMapping(value = "/station/type", method = RequestMethod.GET)
	public ResponseEntity<List<VaFacilityTypeResponse>> getAllFacilityType() throws GenericException {
		List<VaFacilityTypeResponse> response = new ArrayList<VaFacilityTypeResponse>();

		try {
			response = vafacilityService.getAllVaFacilityType();

		} catch (GenericException e) {

			logger.debug("Unable to retrieve facility type");
			throw e;
		}
		return new ResponseEntity<List<VaFacilityTypeResponse>>(response, HttpStatus.OK);
	}

	/**
	 * @author Vamsi Krishna Gangarapu
	 * @return boolean value
	 * @throws GenericException
	 */
	// REST call for search functionality for Visn
	@RequestMapping(value = "/station/search", method = RequestMethod.POST)
	public ResponseEntity<GenericResponse> searchVisn(@RequestBody VaFacilitySearchRequest searchRequest)
			throws GenericException {
		try {
			logger.info("searchVisn() : Received the request to retreive medical codes");
			GenericResponse vaFacilityResponse = vafacilityService.getSearchVisn(searchRequest);
			logger.debug("searchVisn() : Response from vafacilityService.getSearchVisn is :" + vaFacilityResponse);
			return new ResponseEntity<GenericResponse>(vaFacilityResponse, HttpStatus.OK);
		} catch (GenericException e) {
			logger.error("searchVisn() : exception occured : " + " - " + e.getMessage());
			throw e;
		}
	}

	/**
	 * @author Vamsi Krishna Gangarapu
	 * @param stationName
	 * @return boolean value
	 * @throws GenericException
	 */
	// REST call to check if a station exists
	@RequestMapping(value = "/station/checkStationExists/{stationName}")
	public ResponseEntity<JSONObject> checkVisnExists(@PathVariable String stationName) throws GenericException {
		JSONObject entity = new JSONObject();
		try {
			boolean successResponse = vafacilityService.checkVaFacility(stationName);
			if (!successResponse) {
				entity.put("result", successResponse);

			} else {
				throw new GenericException("Error", "VA facility exists", HttpStatus.BAD_REQUEST);
			}
			return new ResponseEntity<JSONObject>(entity, HttpStatus.OK);
		} catch (GenericException e) {
			logger.error("checkVisnExists() : exception occured : " + " - " + e.getMessage());
			throw e;
		}
	}

	/**
	 * @author Vamsi Krishna Gangarapu
	 * @param contactInfo
	 * @return boolean value
	 * @throws GenericException
	 */
	// REST call to Add Contact Info
	@RequestMapping(value = { "/station/contactInfo" }, method = RequestMethod.POST)
	public ResponseEntity<JSONObject> addContactInfo(@Valid @RequestBody ContactInfoRequest contactInfo) throws GenericException {
		JSONObject jsonObject = new JSONObject();
		try {
			boolean successResponse = vafacilityService.postContactInfo(contactInfo);
			if(successResponse) {
				jsonObject.put("result", successResponse);
			} else {
				throw new GenericException("Error", "Unable to add Contact Info", HttpStatus.BAD_REQUEST);
			}
			return new ResponseEntity<JSONObject>(jsonObject, HttpStatus.OK);
		} catch (GenericException e) {
			logger.error("addContactInfo() : exception occured : " + " - " + e.getMessage());
			throw e;
		}
	}

	/**
	 * @author Vamsi Krishna Gangarapu
	 * @param contactInfo
	 * @return boolean value
	 * @throws GenericException
	 */
	// REST call to Modify Contact Info
	@RequestMapping(value = { "/station/contactInfo" }, method = RequestMethod.PUT)
	public ResponseEntity<JSONObject> modifyContactInfo(@Valid @RequestBody ContactInfoRequest contactInfo) throws GenericException {
		JSONObject jsonObject = new JSONObject();
		try {
			boolean successResponse = vafacilityService.postContactInfo(contactInfo);
			if(successResponse) {
				jsonObject.put("result", successResponse);
			} else {
				throw new GenericException("Error", "Unable to modify Contact Info", HttpStatus.BAD_REQUEST);
			}
			return new ResponseEntity<JSONObject>(jsonObject, HttpStatus.OK);
		} catch (GenericException e) {
			logger.error("addContactInfo() : exception occured : " + " - " + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * @author Vamsi Krishna Gangarapu
	 * @param stationName
	 * @return JSON object that has Contact Info
	 * @throws GenericException
	 */
	// REST call for get Contact Info
	@RequestMapping(value = "/station/contactInfo/{stationName}", method = RequestMethod.GET)
	public ResponseEntity<JSONObject> getContactInfo(@PathVariable String stationName) throws GenericException {
		JSONObject jsonObject = new JSONObject();
		try {
			jsonObject.put("contactInfo", vafacilityService.getContactInfo(stationName));
			return new ResponseEntity<JSONObject>(jsonObject, HttpStatus.OK);
		} catch (GenericException e) {
			logger.error("getContactInfo() : exception occured : " + " - " + e.getMessage());
			throw e;
		}
	}

	/**
	 * @author Vamsi Krishna Gangarapu
	 * @param stationName
	 * @return station information with contact info and zip code
	 * @throws GenericException
	 */
	@RequestMapping(value = "/stations/{stationName}", method = RequestMethod.GET)
	public ResponseEntity<VaFacilityResponse> getStationInformation(@PathVariable String stationName)
			throws GenericException {
		try {
			VaFacilityResponse vaFacilityResponse = vafacilityService.getVaFacilityInfo(stationName);
			return new ResponseEntity<VaFacilityResponse>(vaFacilityResponse, HttpStatus.OK);
		} catch (GenericException e) {
			logger.error("getContactInfo() : exception occured : " + " - " + e.getMessage());
			throw e;
		}
	}

	/**
	 * @author Vamsi Krishna Gangarapu
	 * @param stationName
	 * @return station information
	 * @throws GenericException
	 */
	@RequestMapping(value = "/station/{stationName}", method = RequestMethod.GET)
	public ResponseEntity<VaFacilityRequest> getStationInfo(@PathVariable String stationName) throws GenericException {
		try {
			VaFacilityRequest vaFacilityResponse = vafacilityService.getStationInfo(stationName);
			return new ResponseEntity<VaFacilityRequest>(vaFacilityResponse, HttpStatus.OK);
		} catch (GenericException e) {
			logger.error("getContactInfo() : exception occured : " + " - " + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * @author Vamsi Krishna Gangarapu
	 * @param stationName
	 * @return station information
	 * @throws GenericException
	 */
	@RequestMapping(value = "/station/visn/{visnNumber}", method = RequestMethod.GET)
	public ResponseEntity<VaFacilityVisnResponse> getVisnInfo(@PathVariable Long visnNumber) throws GenericException {
		try {
			VaFacilityVisnResponse vaFacilityVisnResponse = null;
			if(visnNumber != null) {
				vaFacilityVisnResponse = vafacilityService.getVisnInfo(visnNumber);
				return new ResponseEntity<VaFacilityVisnResponse>(vaFacilityVisnResponse, HttpStatus.OK);
			} else {
				logger.error("getVisnInfo() : exception occured :  - visnNumber is empty");
				throw new GenericException(ClaimDetailsConstants.BAD_REQUEST,
						"visn number is empty", HttpStatus.BAD_REQUEST);
			}
		} catch (GenericException e) {
			logger.error("getVisnInfo() : exception occured : " + " - " + e.getMessage());
			throw e;
		}

	}

	@RequestMapping(value = "/station/type/{typeNumber}", method = RequestMethod.GET)
	public ResponseEntity<VaFacilityTypeResponse> getStationTypeInfo(@PathVariable String typeNumber) throws GenericException {
		try {
			VaFacilityTypeResponse vaFacilityTypeResponse = null;
			if(typeNumber != null) {
				vaFacilityTypeResponse = vafacilityService.getStationTypeInfo(typeNumber);
				return new ResponseEntity<VaFacilityTypeResponse>(vaFacilityTypeResponse, HttpStatus.OK);
			} else {
				logger.error("getVisnInfo() : exception occured :  - visnNumber is empty");
				throw new GenericException(ClaimDetailsConstants.BAD_REQUEST,
						"visn number is empty", HttpStatus.BAD_REQUEST);
			}
		} catch (GenericException e) {
			logger.error("getVisnInfo() : exception occured : " + " - " + e.getMessage());
			throw e;
		}

	}

	
	/**
	 * @author Vamsi Krishna Gangarapu
	 * @param stationName
	 * @return station information
	 * @throws GenericException
	 */
	@RequestMapping(value = "/station/{stationName}", method = RequestMethod.DELETE)
	public ResponseEntity<JSONObject> deleteStationInfo(@PathVariable String stationName) throws GenericException {
		JSONObject jsonObject = new JSONObject();
		try {
			boolean successResponse = vafacilityService.deleteStationInfo(stationName);
			if(successResponse) {
				jsonObject.put("result", successResponse);
			} else {
				throw new GenericException("Error", "Unable to delete Station", HttpStatus.BAD_REQUEST);
			}
			return new ResponseEntity<JSONObject>(jsonObject, HttpStatus.OK);
		} catch (GenericException e) {
			logger.error("getContactInfo() : exception occured : " + " - " + e.getMessage());
			throw e;
		}

	}

	/**
	 * @author Vamsi Krishna Gangarapu
	 * @param contactInfoId
	 * @return boolean value
	 * @throws GenericException
	 */
	@RequestMapping(value = "/station/contactInfo/{contactInfoId}", method = RequestMethod.DELETE)
	public ResponseEntity<JSONObject> deleteContactInfo(@PathVariable Long contactInfoId) throws GenericException {
		JSONObject jsonObject = new JSONObject();
		try {
			boolean successResponse = vafacilityService.deleteContactInfo(contactInfoId);
			if(successResponse) {
				jsonObject.put("result", successResponse);
			} else {
				throw new GenericException("Error", "Unable to delete Contact Info", HttpStatus.BAD_REQUEST);
			}
			return new ResponseEntity<JSONObject>(jsonObject, HttpStatus.OK);
		} catch (GenericException e) {
			logger.error("addContactInfo() : exception occured : " + " - " + e.getMessage());
			throw e;
		}
	}

	// Add ZIP Code to the VA Facility(Station)
	@RequestMapping(value = "/station/zipcode", method = RequestMethod.POST)
	public ResponseEntity<JSONObject> addZipCode(@Valid @RequestBody ZipCodeRequest request) throws GenericException {
		Boolean result = Boolean.FALSE;
		JSONObject entity = new JSONObject();
		
		try {
			result = vafacilityService.addZipCode(request);
		} catch (GenericException e) {
			logger.debug("Unable to Add zip code");
			throw e;
		}

		if (result) {
			entity.put("result", "Add ZIP Code success");
		} else {
			throw new GenericException("Add zip code error", "Unable to Add Zip Code", HttpStatus.BAD_REQUEST);
		}
		
		return new ResponseEntity<>(entity, HttpStatus.OK);
	}

	// Edit ZIP Code to the VA Facility(Station)
	@RequestMapping(value = "/station/zipcode", method = RequestMethod.PUT)
	public ResponseEntity<JSONObject> editZipCode(@Valid @RequestBody ZipCodeRequest request) throws GenericException {
		Boolean result = Boolean.FALSE;
		JSONObject entity = new JSONObject();

		try {
			result = vafacilityService.editZipCode(request);
		} catch (GenericException e) {
			logger.debug("Unable to edit ZIP Code");
			throw e;
		}

		if (result) {
			entity.put("result", result);
		} else {
			throw new GenericException("Edit ZIP Code error", "Unable to edit ZIP Code", HttpStatus.BAD_REQUEST);
		}

		return new ResponseEntity<>(entity, HttpStatus.OK);
	}

	// Delete ZIP Code
	@RequestMapping(value = "/station/zipcode", method = RequestMethod.DELETE, produces = "application/json")
	public ResponseEntity<JSONObject> removeZipCode(@Valid @RequestBody ZipCodeRequest request) throws GenericException {
		logger.debug("removeZipCode - " + request.getZip_code());
		Boolean result = Boolean.FALSE;
		JSONObject entity = new JSONObject();

		try {
			result = vafacilityService.removeZipCode(request);

		} catch (GenericException e) {
			logger.debug("Unable to remove ZIP Code");
			throw e;
		}

		if (result) {
			entity.put("result", result);
		} else {
			throw new GenericException("Remove ZIP Code error", "Unable to remove ZIP Code", HttpStatus.BAD_REQUEST);
		}

		return new ResponseEntity<>(entity, HttpStatus.OK);
	}
	
	// Find ZIP Codes associate with the VA Facility(Station)
	@RequestMapping(value = "/station/zipcode/{stationNumber}", method = RequestMethod.GET)
	public ResponseEntity<JSONObject> getZipCodeList(@PathVariable("stationNumber") String stationNumber)
			throws GenericException {
		logger.debug("getZipCodeList - " + stationNumber);
		JSONObject jsonObject = new JSONObject();
		List<ZipCodeRequest> list = new ArrayList<>();

		try {
			list = vafacilityService.getZipCodeList(stationNumber);
			jsonObject.put("zipcodes", list);

		} catch (GenericException e) {
			logger.debug("Unable to get ZIP Code");
			throw e;
		}

		return new ResponseEntity<>(jsonObject, HttpStatus.OK);
	}

	/**
	 * @author Vamsi Krishna Gangarapu
	 * @param zipCode
	 * @return boolean value
	 * @throws GenericException
	 */
	// REST call to check if a station exists
	@RequestMapping(value = "/station/zipcode/checkZipExists/{zipCode}")
	public ResponseEntity<JSONObject> checkZipExists(@PathVariable String zipCode) throws GenericException {
		JSONObject jsonObject = new JSONObject();
		try {
			boolean successResponse = vafacilityService.checkZipExists(zipCode);
			if(!successResponse) {
				jsonObject.put("result", successResponse);
			} else {
				throw new GenericException("Error", "Zip code exists", HttpStatus.BAD_REQUEST);
			}
			return new ResponseEntity<JSONObject>(jsonObject, HttpStatus.OK);
		} catch (GenericException e) {
			logger.error("checkVisnExists() : exception occured : " + " - " + e.getMessage());
			throw e;
		}
	}
	
	@RequestMapping(value = "/stations")
	public ResponseEntity<JSONObject> getAllVaFacilites() throws GenericException {
		JSONObject jsonObject = new JSONObject();
		try {
			List<VaFacilities> vaFacilitiesList = vafacilityService.getAllVaFacilites();
			jsonObject.put("VaFacilites", vaFacilitiesList);
			return new ResponseEntity<JSONObject>(jsonObject, HttpStatus.OK);
		} catch (Exception e) {
			logger.error("getAllVaFacilites() : exception occured : " + " - " + e.getMessage());
			throw e;
		}
	}

	// Validate inputs
	private List<String> validationError(VaFacilityRequest request) {
		List<String> error_description = new ArrayList<>();

		// User must need to provide a valid App User Name
		if (request.getAppUserName() == null || request.getAppUserName().isEmpty()) {
			error_description.add("Missing App User Name");
		}

		// Station number field is a required field
		if (request.getVaFacilityCd() == null || request.getVaFacilityCd().isEmpty()
				|| request.getVaFacilityCd().length() > 12) {
			error_description.add("Missing Station Number");
		}

		// Visn field is a required field
		if (Long.valueOf(request.getVisnIdCd()) == null) {
			error_description.add("Missing VisN field");
		}

		// Rerouting Allowed is a required field
		if (Boolean.valueOf(request.isReroutingAllowed()) == null) {
			error_description.add("Missing Rerouting Allowed field");
		}

		// Type is a required field
		if (request.getVaTypeCd() != null && request.getVaTypeCd().isEmpty()) {
			error_description.add("Missing Type field");
		}

		if (request.getShortName() != null && request.getShortName().length() > 80) {
			error_description.add("Short Name is too long");
		}

		if (request.getVaFacilityName() != null && request.getVaFacilityName().length() > 100) {
			error_description.add("Name is too long");
		}

		if (request.getAgedDefinition() != null && String.valueOf(request.getAgedDefinition()).length() > 10) {
			error_description.add("Aged Definition is too long");
		}

		if (request.getParentVaFacilityCd() != null && request.getParentVaFacilityCd().length() > 36) {
			error_description.add("Parent Station is too long");
		}

		return error_description;
	}

}