package gov.va.med.fee.controller;

import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import gov.va.med.fee.dao.IAppUserRepository;
import gov.va.med.fee.exceptions.GenericException;
import gov.va.med.fee.model.request.UserReqRequest;
import gov.va.med.fee.model.response.UserReqResponse;
import gov.va.med.fee.service.IUserRequestService;
import net.minidev.json.JSONObject;


@RestController
@RequestMapping("/api/v1/userRequest")
public class UserRequestController {
	private static final Logger logger = LogManager.getLogger(UserRequestController.class);
	
	@Autowired
	IUserRequestService userRequestService;
	@Autowired
	IAppUserRepository appUserRepository;	
	
	/* Restful service to get available roles and available facilities
	 * Success: Return HttpStatus.OK (200) - Standard response for successful HTTP requests
	 * Fail: Return HttpStatus.BAD_REQUEST (400) - The server cannot or will not process the request due to an error 
	 * */
	@RequestMapping(value="/form",method = RequestMethod.GET, produces = "application/json")
	public ResponseEntity<UserReqResponse> loadRequestUser(HttpServletRequest request) throws GenericException {
		                
        UserReqResponse response = null;
		try {
			response = userRequestService.loadUserRequestForm();
		} catch (Exception e) {
			logger.debug("Unable to Load Roles and Facilities");
			throw e;
		}
		
		if(response == null) {
			return new ResponseEntity<UserReqResponse>(response, HttpStatus.BAD_REQUEST);
		} 		
		
		return new ResponseEntity<UserReqResponse>(response, HttpStatus.OK);
	}
		
	/* Restful service to create user request
	 * Success: Return HttpStatus.CREATED (201) - a response to successful POST requests
	 * Fail: Return HttpStatus.BAD_REQUEST (400) - The server cannot meet the requirements
	 * */
	@RequestMapping(method = RequestMethod.POST, produces = "application/json")
	public ResponseEntity<JSONObject> requestAddUser(@RequestBody UserReqRequest userRequest) throws GenericException {
		List<String> error_description = validationError(userRequest, false);
		
		if(!error_description.isEmpty()) {
			throw new GenericException("Create user request error", error_description.toString(), HttpStatus.BAD_REQUEST);
		
		} else {		
			Integer result = new Integer(0);
			JSONObject entity = new JSONObject();
			try {
				result = userRequestService.requestAddUser(userRequest);
			} catch (GenericException e) {
				logger.debug("Unable to Request Add User");
				throw e;
			}
			 		
			if(result > 0) {
				entity.put("result", true);
			} else {
				throw new GenericException("Duplicate","Duplicate Window_Nt_Name", HttpStatus.BAD_REQUEST);
			}
			
			return new ResponseEntity<>(entity, HttpStatus.CREATED); 
		}
	}
		
	/* Restful service to modify user request
	 * Success: Return HttpStatus.OK (200) - a response to successful POST requests
	 * 	return UserReqResponse
	 * Fail: Return HttpStatus.BAD_REQUEST (400) - The server cannot meet the requirements
	 * */
	@RequestMapping(method = RequestMethod.PUT, produces = "application/json")
	public ResponseEntity<JSONObject> modifyUserRequest(@RequestBody UserReqRequest userRequest) throws GenericException {
		List<String> error_description = validationError(userRequest, false);
		
		if(!error_description.isEmpty()) {
			throw new GenericException("Modify user request error", error_description.toString(), HttpStatus.BAD_REQUEST);
		
		} else {		
			Integer result = new Integer(0);
			JSONObject entity = new JSONObject();
			try {
				result = userRequestService.requestModifyUser(userRequest);
			} catch (GenericException e) {
				logger.debug("Unable to Request Modify User");
				throw e;
			}
			 		
			if(result > 0) {
				entity.put("result", true);
			} else {
				throw new GenericException("Modify error","Unable to create or update an user request", HttpStatus.BAD_REQUEST);
			}
			
			return new ResponseEntity<>(entity, HttpStatus.OK); 
		}
		
		/* else {	
			Integer result = new Integer(0);
			try {
				result = userRequestService.requestModifyUser(userRequest);
			} catch (GenericException e) {
				logger.debug("Unable to Request Modify User");
				throw e;
			}			
			
			logger.debug("Modify User Request row = " + result);
			
			//Should look for app user info by the window_nt_name, but comment out for later implementation if needed
			UserReqResponse response = null;
			try {
				//get UserToEditByUserName
				response = userRequestService.loadModifyUserRequestFormByName(userRequest.getWindow_nt_name());
			} catch (GenericException e) {
				logger.debug("Unable to Request Modify User, response is empty");
				throw e;
			}		
			return new ResponseEntity<>(response, HttpStatus.OK);  
		}
		*/
	}
		
	/* Calling after Search for user request, send an appUserId to load modify user request
	 Search User For Request = findUserToEditByUserId -
	 userRequest.rolesAvailableToUserForUserModifyRequest,
	 userRequest.facilitiesAvailableToUser*/
	@RequestMapping(value = "/form/{userName}", method = RequestMethod.GET, produces = "application/json")
	public ResponseEntity<UserReqResponse> findUserToEditByUserName(@PathVariable("userName") String userName) throws GenericException {
		logger.debug("findUserToEditByUserId - " +userName);
		
		UserReqResponse response = null;
		try {
			response = userRequestService.loadModifyUserRequestFormByName(userName);
		} catch (GenericException e) {
			logger.debug("Unable Find User To Edit By User Id");
			throw e;
		}			
		return new ResponseEntity<>(response, HttpStatus.OK); 
	}
	
	//Validate inputs
	private List<String> validationError(UserReqRequest userRequest, boolean isModify) {
		List<String> error_description = new ArrayList<>();
		
		if(userRequest.getWindow_nt_name() == null || userRequest.getWindow_nt_name().isEmpty()) {
			error_description.add("Missing Window_nt_name");
		}
		
		if(userRequest.getCurrent_user_name() == null || userRequest.getCurrent_user_name().isEmpty()) {
			error_description.add("Missing Current_user_name");
		}
		
		if(userRequest.getWindow_nt_name() != null && userRequest.getWindow_nt_name().length() > 30) {
			error_description.add("Window NT Name is too long");
		}
		
		if(userRequest.getLast_name() != null && userRequest.getLast_name().length() > 30) {
			error_description.add("Last name is too long");
		}
		
		if(userRequest.getFirst_name() != null && userRequest.getFirst_name().length() > 30) {
			error_description.add("First name is too long");
		}
		
		if(userRequest.getMiddle_name() != null && userRequest.getMiddle_name().length() > 30) {
			error_description.add("Middle name is too long");
		}
		
		if(userRequest.getEmail() != null && userRequest.getEmail().length() > 100) {
			error_description.add("Email is too long");
		}
		
		if(userRequest.getPhone() != null && userRequest.getPhone().length() > 20) {
			error_description.add("Phone number is too long");
		}		
		
		if(userRequest.getRequestedFacilities() != null && userRequest.getRequestedFacilities().length() > 4000) {
			error_description.add("Requested facilities is too long");
		}		
		
		if(isModify) {
			if(userRequest.getDisable_comments() != null && userRequest.getDisable_comments().length() > 200) {
				error_description.add("Disable comments is too long");
			}
		}
				
		return error_description;
	}	
}
