package gov.va.med.fee.controller;

import java.util.ArrayList;
import java.util.List;


import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import gov.va.med.fee.exceptions.GenericException;
import gov.va.med.fee.model.request.RerouteClaimRequest;
import gov.va.med.fee.model.response.RerouteClaimResponse;
import gov.va.med.fee.service.IRerouteClaimsService;

@RestController
@RequestMapping("/api/v1/rerouteClaims")
public class RerouteClaimsController {
	private static final Logger logger = LogManager.getLogger(RerouteClaimsController.class);

	@Autowired
	IRerouteClaimsService rerouteClaimsService;
	
	/* Restful service to populate rereroute claims
	 * Success: Return HttpStatus.OK (200) - Standard response for successful HTTP requests
	 * Fail: Return HttpStatus.BAD_REQUEST (400) - The server cannot or will not process the request due to an error 
	 * */
	@RequestMapping(method = RequestMethod.POST, produces = "application/json")
	public ResponseEntity<RerouteClaimResponse> getReroutes(@RequestBody List<Long> claimsToReroute) throws GenericException {
		logger.debug("getReroutes request = " + claimsToReroute);                 
		RerouteClaimResponse response = null;   
		
		//Return backend validation error message
		if(claimsToReroute == null || claimsToReroute.isEmpty()) {
			throw new GenericException("Getting reroute claims error", "Empty reroute claim request", HttpStatus.BAD_REQUEST);
		} else {
			try {
				response = rerouteClaimsService.populateReroutes(claimsToReroute);
				logger.debug("RerouteClaimResponse" + response.toString());
			} catch (GenericException e) {
				logger.debug("Unable to populate reroute claims");
				throw e;
			}	
		}
		
		return new ResponseEntity<RerouteClaimResponse>(response, HttpStatus.OK);
	}
	
	@RequestMapping(value = "/reroutes", method = RequestMethod.PUT, produces = "application/json")
	public ResponseEntity<RerouteClaimResponse> processReroutes(@RequestBody RerouteClaimRequest request) throws GenericException {
		logger.debug("processReroutes request = " + request); 
		RerouteClaimResponse response = null;
		
		List<String> error_descriptions = getValidationError(request);
		
		if(!error_descriptions.isEmpty()) {
			throw new GenericException("Process Reroutes error", error_descriptions.toString(), HttpStatus.BAD_REQUEST);
		} else {	
			try {
				response = rerouteClaimsService.reroute(request);
			} catch (GenericException e) {
				logger.debug("Unable to reroute claims");
				throw e;
			}	
		}
	
		return new ResponseEntity<RerouteClaimResponse>(response, HttpStatus.OK);
	}
	
	/* Ajax call - restful service to populate Facility list
	 * Request should include selected visnId and selected facility code
	 * Success: Return HttpStatus.OK (200) - Standard response for successful HTTP requests
	 * Fail: Return HttpStatus.BAD_REQUEST (400) - The server cannot or will not process the request due to an error 
	 * */
	@RequestMapping(value = "/facilities", method = RequestMethod.POST, produces = "application/json")
	public ResponseEntity<RerouteClaimResponse> getFacilities(@RequestBody RerouteClaimRequest request) throws GenericException {
		logger.debug("getFacilities request = " + request.getSelectedVisn());                
		
		RerouteClaimResponse response = new RerouteClaimResponse();
		try {
			List<String> facilitiesList  = rerouteClaimsService.getFacilitiesList(request.getSelectedVisn());
			response.setFacilities(facilitiesList);
		} catch (GenericException e) {
			logger.debug("Unable to populate facilitiesList");
			throw e;
		}
		
		return new ResponseEntity<RerouteClaimResponse>(response, HttpStatus.OK);
	}
		
	@RequestMapping(method = RequestMethod.PUT, produces = "application/json")
	public ResponseEntity<RerouteClaimResponse> reroutes(@RequestBody RerouteClaimRequest request) throws GenericException {
		logger.debug("reroutes request = " + request);                 
		RerouteClaimResponse response = null;   
		
		List<String> error_description = getValidationError(request);
		
		if(!error_description.isEmpty()) {
			throw new GenericException("Reroute claims error", error_description.toString(), HttpStatus.BAD_REQUEST);
		
		} else {
			
		}
		return new ResponseEntity<RerouteClaimResponse>(response, HttpStatus.OK);
	}
	
	//Validate inputs
	private List<String> getValidationError(RerouteClaimRequest request) {
		List<String> error_descriptions = new ArrayList<>();

		if(request.getCurrent_user_name() == null || request.getCurrent_user_name().isEmpty()) {
			error_descriptions.add("Missing Current_user_name");
		}
		
		if (request.getSelectedVisn() == null || "".equals(request.getSelectedVisn())) {
			error_descriptions.add("Missing Selected VISN");
		}

		if (request.getSelectedFacility() == null || "".equals(request.getSelectedFacility())) {
			error_descriptions.add("Missing Selected Facility");
		}
		

		if (request.getSelectedReason() == null || "".equals(request.getSelectedReason())) {
			error_descriptions.add("Missing Selected Reason");
		}
		
		if(request.getComment() != null && request.getComment().length() > 250) {
			error_descriptions.add("Comments field exceeds 250 characters");
		}
		
		return error_descriptions;
	}

}
