package gov.va.med.fee.controller;

import java.io.ByteArrayOutputStream;
import java.io.FileOutputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.util.Calendar;
import java.util.Date;
import java.util.Map;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import gov.va.med.fee.exceptions.GenericException;
import gov.va.med.fee.model.request.PrintClaimsRequest;
import gov.va.med.fee.model.request.SearchCriteriaRequest;
import gov.va.med.fee.model.response.JasperReportsJsonResponse;
import gov.va.med.fee.service.IAuthenticatorService;
import gov.va.med.fee.service.IFeeReportService;
import gov.va.med.fee.service.IPrintClaimsService;

@RestController
@RequestMapping("/api/v1")
public class PrintReportsController {
	private static final Logger logger = LogManager.getLogger(PrintReportsController.class);

	@Autowired
	IFeeReportService feeReportService;

	@Autowired
	IPrintClaimsService printClaimsService;

	@Autowired
	IAuthenticatorService authenticatorService;

	@RequestMapping(value = "/reports", method = RequestMethod.POST)
	public ResponseEntity<byte[]> getReport(@RequestBody SearchCriteriaRequest criteria) {

		try {
			logger.debug("SearchCriteriaRequest -" + criteria.toString());

			if (criteria.getDateValue().getFromDate() == null) {
				Date date = new Date();
				LocalDate fromDate = LocalDate.now().minusDays(criteria.getDateValue().getDays());
				criteria.getDateValue().setFromDate(java.sql.Date.valueOf((fromDate)));
				criteria.getDateValue().setToDate(date);
			}
			feeReportService.setCriteria(criteria);
			Map<String, Object> resultMap = feeReportService.displayJasper();
			String filename = (String) resultMap.get("fileName");
			byte[] contents = (byte[]) resultMap.get("output");
			// outputStream.close();

			HttpHeaders headers = new HttpHeaders();

			if ("C".equalsIgnoreCase(criteria.getFormat())) {
				filename += ".csv";
				headers.setContentType(MediaType.parseMediaType("text/plain"));
			} else if ("E".equalsIgnoreCase(criteria.getFormat())) {
				filename += ".xlsx";
				headers.setContentType(MediaType.parseMediaType("application/vnd.ms-excel"));
			} else {
				filename += ".pdf";
				headers.setContentType(MediaType.parseMediaType("application/pdf"));
			}

			headers.setContentDispositionFormData(filename, filename);
			headers.setCacheControl("must-revalidate, post-check=0, pre-check=0");

			return new ResponseEntity<byte[]>(contents, headers, HttpStatus.OK);

		} catch (SQLException e) {
			logger.error("PrintReportsController SQLException -");
			e.printStackTrace();

		}
		return new ResponseEntity<byte[]>(HttpStatus.INTERNAL_SERVER_ERROR);
	}

	@RequestMapping(value = "/reports/claims", method = RequestMethod.POST)
	public ResponseEntity<Object> getReport(@RequestBody PrintClaimsRequest printClaimsRequest)
			throws GenericException {

		HttpStatus httpStatus = HttpStatus.NOT_FOUND;
		ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
		String guid = printClaimsRequest.getGuid();
		HttpHeaders headers = new HttpHeaders();
		byte[] contents = null;

		try {
			logger.debug(
					"PrintReportsController:getReport() invoking the authenticatorService.getUsernameFromAspAuth() with claim index"
							+ printClaimsRequest.getIds());
			if (!(printClaimsRequest.getIds().isEmpty())) {
				String appUserName = authenticatorService.getUsernameFromAspAuth(guid);
				if (appUserName == null || appUserName.isEmpty()) {
					appUserName = "VHAADMIN";
				}
				logger.debug(
						"PrintReportsController:getReport() invoking the printClaimsService.displayClaims() with claim index"
								+ printClaimsRequest.getIds());
				outputStream = printClaimsService.displayClaims(printClaimsRequest.getIds(), appUserName);

				String filename = "Claims.pdf";
				contents = outputStream.toByteArray();

				headers.setContentType(MediaType.parseMediaType("application/pdf"));
				headers.setContentDispositionFormData(filename, filename);
				headers.setCacheControl("must-revalidate, post-check=0, pre-check=0");
				httpStatus = HttpStatus.OK;

			}
		} catch (GenericException e) {
			logger.error("PrintClaimsController error -");
			throw e;
		}

		return new ResponseEntity<Object>(contents, headers, httpStatus);

	}

	@RequestMapping(value = "/reports/claims/pdf", method = RequestMethod.POST)
	public ResponseEntity<JasperReportsJsonResponse> getReportPdf(@RequestBody PrintClaimsRequest printClaimsRequest) {

		HttpStatus httpStatus = HttpStatus.NOT_FOUND;
		ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
		JasperReportsJsonResponse jasperReportsJsonResponse = new JasperReportsJsonResponse();
		String guid = printClaimsRequest.getGuid();

		try {
			logger.info("PrintReportsController:getReport() received the request for claim index :"
					+ printClaimsRequest.getIds());

			logger.debug(
					"PrintReportsController:getReport() invoking the authenticatorService.getUsernameFromAspAuth() with claim index"
							+ printClaimsRequest.getIds());
			if (!(printClaimsRequest.getIds().isEmpty())) {
				String appUserName = authenticatorService.getUsernameFromAspAuth(guid);
				if (appUserName == null || appUserName.isEmpty()) {
					appUserName = "VHAADMIN";
				}
				logger.info("PrintReportsController:getReport() received the request for claim index :"
						+ printClaimsRequest.getIds());

				logger.debug(
						"PrintReportsController:getReport() invoking the printClaimsService.displayClaims() with claim index"
								+ printClaimsRequest.getIds());
				outputStream = printClaimsService.displayClaims(printClaimsRequest.getIds(), appUserName);

				String filename = "Claims.pdf";
				String file = "Claims";
				byte[] contents = outputStream.toByteArray();
				String pathName = byteArrayToFile(contents, file);

				HttpHeaders headers = new HttpHeaders();
				headers.setContentType(MediaType.parseMediaType("application/pdf"));
				headers.setContentDispositionFormData(filename, filename);
				headers.setCacheControl("must-revalidate, post-check=0, pre-check=0");

				if (pathName != null) {
					jasperReportsJsonResponse.setPathName(pathName);
					httpStatus = HttpStatus.OK;
				} else {
					logger.info("PrintReportsController:getReport() received the request for claim index :"
							+ printClaimsRequest.getIds());

					logger.debug(
							"PrintReportsController:getReport() invoking the printClaimsService.displayClaims() with claim index"
									+ printClaimsRequest.getIds());
					jasperReportsJsonResponse.setPathName("File Not Found: Report not generated");
					return new ResponseEntity<JasperReportsJsonResponse>(jasperReportsJsonResponse,
							HttpStatus.NOT_FOUND);
				}
			}
		} catch (Exception e) {
			logger.error("PrintClaimsController error -");
			e.printStackTrace();
		}
		return new ResponseEntity<JasperReportsJsonResponse>(jasperReportsJsonResponse, httpStatus);
	}

	@RequestMapping(value = "/reports/pdf", method = RequestMethod.POST)
	public ResponseEntity<JasperReportsJsonResponse> getAgingReportPdf(@RequestBody SearchCriteriaRequest criteria) {

		HttpStatus httpStatus = HttpStatus.NOT_FOUND;
		JasperReportsJsonResponse jasperReportsJsonResponse = new JasperReportsJsonResponse();
		try {
			logger.debug("SearchCriteriaRequest -" + criteria.toString());

			if (criteria.getDateValue().getFromDate() == null) {
				Date date = new Date();
				LocalDate fromDate = LocalDate.now().minusDays(criteria.getDateValue().getDays());
				criteria.getDateValue().setFromDate(java.sql.Date.valueOf((fromDate)));
				criteria.getDateValue().setToDate(date);
			}
			
			feeReportService.setCriteria(criteria);
			Map<String, Object> resultMap = feeReportService.displayJasper();
			String filename = (String) resultMap.get("fileName");
			byte[] contents = (byte[]) resultMap.get("output");
			String file = "Aging";
			String pathName = byteArrayToFile(contents, file);
			// outputStream.close();

			HttpHeaders headers = new HttpHeaders();

			if ("C".equalsIgnoreCase(criteria.getFormat())) {
				filename += ".csv";
				headers.setContentType(MediaType.parseMediaType("text/plain"));
			} else if ("E".equalsIgnoreCase(criteria.getFormat())) {
				filename += ".xlsx";
				headers.setContentType(MediaType.parseMediaType("application/vnd.ms-excel"));
			} else {
				filename += ".pdf";
				headers.setContentType(MediaType.parseMediaType("application/pdf"));
			}

			headers.setContentDispositionFormData(filename, filename);
			headers.setCacheControl("must-revalidate, post-check=0, pre-check=0");

			if (pathName != null) {
				jasperReportsJsonResponse.setPathName(pathName);
				httpStatus = HttpStatus.OK;
			} else {
				jasperReportsJsonResponse.setPathName("File Not Found: Report not generated");
				return new ResponseEntity<JasperReportsJsonResponse>(jasperReportsJsonResponse, HttpStatus.NOT_FOUND);
			}

		} catch (SQLException e) {
			logger.error("PrintReportsController SQLException -");
			e.printStackTrace();

		}
		return new ResponseEntity<JasperReportsJsonResponse>(jasperReportsJsonResponse, httpStatus);
	}

	private String byteArrayToFile(byte[] outputStream, String filename) {
		String pathname = null;
		String osName = null;
		try {
			final String DATE_FORMAT_NOW = "yyyy-MM-dd HH:mm:ss";
			Calendar cal = Calendar.getInstance();
			SimpleDateFormat sdf = new SimpleDateFormat(DATE_FORMAT_NOW);
			String timeName = sdf.format(cal.getTime()).toString().trim();
			timeName = timeName.replaceAll("\\s+", "");
			timeName = timeName.replaceAll(":", "-");
			// get environment to determine pathname
			osName = System.getProperty("os.name").toLowerCase();
			if (osName.indexOf("win") >= 0) {
				// windows environment
				pathname = "./usr/share/tomcat/latest/webapps/generatedReports/";
				Files.createDirectories(Paths.get(pathname));
				pathname = pathname + filename + timeName + ".pdf";
				FileOutputStream fos = new FileOutputStream(pathname);
				fos.write(outputStream);
				fos.close();
			} else {
				// other ie Unix, OSX, etc.
				pathname = "/usr/share/tomcat/latest/webapps/generatedReports/";
				Files.createDirectories(Paths.get(pathname));
				pathname = pathname + filename + timeName + ".pdf";
				FileOutputStream fos = new FileOutputStream(pathname);
				fos.write(outputStream);
				fos.close();
				// make sure ownership of file is set to apache:apache
				Process p = Runtime.getRuntime().exec(new String[] { "bash", "-c", "chown apache:apache " + pathname });
			}
		} catch (Exception e) {
			System.err.println("Error: " + e.getMessage());
		}
		return pathname;

	}
}
