package gov.va.med.fee.controller;

import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import gov.va.med.fee.exceptions.GenericException;
import gov.va.med.fee.model.request.RejectedClaimRequest;
import gov.va.med.fee.model.response.DisapproveClaims;
import gov.va.med.fee.model.response.DisapproveClaimsPage;
import gov.va.med.fee.service.IDisapproveService;

/**
 * 
 * @author vhaisbjahank  
 * This is the controller for disapproving one or max 100 claims.
 * This controller will generate the reason codes for disapprove.
 * This controller will get all the claims by their ClaimIds to disapprove. 
 * 
 * 
 * @Modifiedauthor Vamsi Krishna Gangarapu
 * corrected populateAdjustmentReasons()
 * Added getClaimsToDisapprove() and disapproveClaims()
 * 
 *
 */

@RestController
@RequestMapping("/api/v1")
public class DisapproveClaimController {

	@Autowired
	IDisapproveService disapproveService;

	private static final Logger logger = LogManager.getLogger(DisapproveClaimController.class);

	
	/**
	 * Retrieve all the claims to disapprove by their ClaimId
	 * @return List of Claims 
	 * @throws GenericException
	 */
	@RequestMapping(value = "/disapproveClaim/claims", method = RequestMethod.POST, produces = "application/json")
	public ResponseEntity<DisapproveClaimsPage> getClaimsToDisapprove(@RequestBody List<Long> claimIds) throws GenericException  {
		try {
			DisapproveClaimsPage claimsToDisapprove = disapproveService.getClaimsToDisapprove(claimIds);
			return new ResponseEntity<DisapproveClaimsPage>(claimsToDisapprove, HttpStatus.OK);
		} catch (GenericException e) {
			logger.error("getClaimsToDisapprove() exception occured : "
					+" - " + e.getMessage());
			throw e;
		}
			
	}
	
	/**
	 * Get all reason code
	 * @return a list of reason codes
	 * @throws GenericException
	 */
	@RequestMapping(value = "/disapproveClaim/reasons", method = RequestMethod.GET, produces = "application/json")
	public ResponseEntity<DisapproveClaimsPage> populateAdjustmentReasons() throws GenericException {
		
		logger.info("populateAdjustmentReasons()");
		
		try {			
			DisapproveClaimsPage reasons = disapproveService.populateAdjustmentReasons();
			
			logger.debug("populateAdjustmentReasons() :", reasons.toString());
			return new ResponseEntity<DisapproveClaimsPage>(reasons, HttpStatus.OK);
			
		} catch (GenericException e) {
			logger.error("populateAdjustmentReasons() exception occured : "
					+" - " + e.getMessage());
			
			throw e; 
		}
	}
	
	@RequestMapping(value = "/disapproveClaim", method = RequestMethod.POST, produces = "application/json")
	public ResponseEntity<DisapproveClaimsPage> disapproveClaims(@RequestBody DisapproveClaims disapproveClaims) throws GenericException  {
		try {
			DisapproveClaimsPage successResponse = disapproveService.disapproveClaim(disapproveClaims);
			return new ResponseEntity<DisapproveClaimsPage>(successResponse, HttpStatus.OK);
		} catch (GenericException e) {
			logger.error("getClaimsToDisapprove() exception occured : "
					+" - " + e.getMessage());
			throw e;
		}
	}
	
	@RequestMapping(value = "/okDisapproval/{userName}", method = RequestMethod.POST, produces = "application/json")
	public ResponseEntity<Boolean> okDisapproval(@PathVariable String userName, @RequestBody List<RejectedClaimRequest> rejectedClaimRequestList) throws GenericException  {
		try {
			boolean successResponse = disapproveService.okDisapprovals(rejectedClaimRequestList, userName);
			return new ResponseEntity<Boolean>(successResponse, HttpStatus.OK);
		} catch (GenericException e) {
			logger.error("okDisapproval() exception occured : "
					+" - " + e.getMessage());
			throw e;
		}
	}
	
	@RequestMapping(value = "/reverseDisapproval/{userName}", method = RequestMethod.POST, produces = "application/json")
	public ResponseEntity<Boolean> reverseDisapproval(@PathVariable String userName, @RequestBody List<RejectedClaimRequest> rejectedClaimRequestList) throws GenericException  {
		try {
			boolean successResponse = disapproveService.reverseDisapprovals(rejectedClaimRequestList, userName);
			return new ResponseEntity<Boolean>(successResponse, HttpStatus.OK);
		} catch (GenericException e) {
			logger.error("reverseDisapproval() exception occured : "
					+" - " + e.getMessage());
			throw e;
		}
	}	
}
