package gov.va.med.fee.controller;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import gov.va.med.fee.exceptions.GenericException;
import gov.va.med.fee.model.response.AuthenticationParamResponse;
import gov.va.med.fee.model.response.UserInfoResponse;
import gov.va.med.fee.model.response.UserStationDetails;
import gov.va.med.fee.service.IAuthenticatorService;

/**
 *  @author Eyuel Taddese
 *  This is a controller that will handle the User Authentication and send Roles of the User
 *
 */

@RestController
@RequestMapping("/api/v1")
public class AuthenticatorController {

	private static final Logger logger = LogManager.getLogger(AuthenticatorController.class);
	
	@Autowired
	IAuthenticatorService authenticatorService;
	
	public AuthenticatorController() {
		
	}
	/**
	 * This method return the list of UserRoles
	 * @param guid
	 * @return ResponseEntity<List<UserRoles>>
	 * @throws GenericException 
	 */
	@RequestMapping(value="/roles",method = RequestMethod.GET, produces = "application/json")
	public ResponseEntity<UserInfoResponse> getUserRoles(@RequestParam String guid) throws GenericException {
		logger.info("getUserRoles : "+"getting userroles for "+guid);
		
		try {
			//Authenticate the User First
			boolean isAuthenticated = authenticatorService.authenticate(guid);
			UserInfoResponse userRolesResponse = null;
			if(isAuthenticated) {
				//Get UserRoles
				userRolesResponse = authenticatorService.getUserRoles();
				if(userRolesResponse != null && userRolesResponse.getUserRoles().size() > 0) {
					return new ResponseEntity<UserInfoResponse>(userRolesResponse,HttpStatus.OK);
				}
				else {
					logger.warn("getUserRoles : 0 Roles found for "+guid);
					return new ResponseEntity<UserInfoResponse>(HttpStatus.NOT_FOUND);
				}
			}
			else {
				logger.error("getUserRoles : Authentication failed for GUID: "+guid);
				throw new GenericException("", "", HttpStatus.ACCEPTED);
			}
		} catch(GenericException u) {
			logger.error("getUserRoles : Exception is raised ");
			throw u;
		}
	
	}
	
	/**
	 * Get authentication param's that are required to invoke third party service 
	 * @param request Http servlet request object
	 * @return Authentication parameters (encryptedDefaultUrl, encryptedPostBackUrl, sessionId, aspCheckPageUrl and errorPageUrl)
	 * @throws GenericException
	 */
	@RequestMapping(value="/authParams",method = RequestMethod.GET, produces = "application/json")
	public ResponseEntity<AuthenticationParamResponse> getAuthParams(HttpServletRequest request) throws GenericException {
		logger.info("Received the request for authParam");
		String sessionId = getSessionId(request);
		AuthenticationParamResponse authenticationParamResponse = null;
		try {
			authenticationParamResponse = authenticatorService.getAuthParams(sessionId);
		} catch (Exception e) {
			logger.error("Error in getting auth params");
			e.printStackTrace();
		}
		
		return new ResponseEntity<AuthenticationParamResponse>(authenticationParamResponse, HttpStatus.OK);
	}
	
	/**
	 * This service is added to get the latest station and claim count details for the logged-in user
	 * @param userName
	 * @return
	 * @throws GenericException
	 */
	@RequestMapping(value="/refreshStats", method = RequestMethod.GET, produces = "application/json")
	public ResponseEntity<UserStationDetails> getUpdatedStats(@RequestParam String userName) throws GenericException {
		logger.info("Received the request to get the updated counts for userName "+userName);
		try {
		UserStationDetails userStationDetails = authenticatorService.getUpdatedUserStationDetails(userName);
		return new ResponseEntity<UserStationDetails>(userStationDetails,HttpStatus.OK);
		} catch(GenericException e) {
			logger.error("getUpdatedStats : Exception is raised ");
			throw e;
		}
	}
	
	private String getSessionId(HttpServletRequest hreq){
        HttpSession session = hreq.getSession();
        String id = session.getId() + session.hashCode();
        return id;
	}
		
}

