package gov.va.med.fee.controller;

import java.util.List;



import javax.validation.Valid;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import gov.va.med.fee.model.AppUserRequestDTO;
import gov.va.med.fee.service.IAppUserRequestService;
import gov.va.med.fee.vo.ErrorVo;

@RestController
@RequestMapping("/api/v1/appUserRequest")
public class AppUserRequestController {

	@Autowired
	IAppUserRequestService appUserRequestService;

	private static final Logger log = LogManager.getLogger(AppUserRequestController.class);

	/**
	 * Create App user request
	 * @param req App user request data that needs to be persisted
	 * @return App user request object that created in database
	 */
    @RequestMapping(method = RequestMethod.POST)
    public ResponseEntity<?> create(@RequestBody AppUserRequestDTO req) {
        ResponseEntity<?> result = null;
        try {

        	AppUserRequestDTO obj = appUserRequestService.create(req);
            result = new ResponseEntity<AppUserRequestDTO>(obj, HttpStatus.OK);
            log.info("Successfully created app user request with request id: " + obj.getAppUserRequestId());

        } catch (Exception e) {
            log.error("Error creating app user request for : " + req.getUserName(), e);
            ErrorVo vo = new ErrorVo("ERR", "There was an error in creating app user request, please contact support.");
            result = new ResponseEntity<ErrorVo>(vo, HttpStatus.INTERNAL_SERVER_ERROR);
        }
        return result;
    }

    /**
     * Delete App User Request
     * @param requestId Id of the app user request that needs to be deleted
     * @return Deleted app user request object
     */
    @RequestMapping(value = "{requestId}", method = RequestMethod.DELETE)
    public ResponseEntity<?> delete(@PathVariable("requestId") long requestId) {
        ResponseEntity<?> result = null;
        try {

        	AppUserRequestDTO obj = appUserRequestService.delete(requestId);
            result = new ResponseEntity<AppUserRequestDTO>(obj, HttpStatus.OK);
            log.info("Successfully deleted app user request for request id: " + obj.getAppUserRequestId());

        } catch (Exception e) {
            log.error("Error deleting app user request for requestId : " + requestId, e);
            ErrorVo vo = new ErrorVo("ERR", "There was an error in deleting app user request, please contact support.");
            result = new ResponseEntity<ErrorVo>(vo, HttpStatus.INTERNAL_SERVER_ERROR);
        }
        return result;
    }
    
    /**
     * Update App User Request
     * @param req App user request data that needs to be updated
     * @return App user request object that updated in database
     */
    @RequestMapping(value = "/update", method = RequestMethod.POST)
     public ResponseEntity<?> update(@RequestBody @Valid AppUserRequestDTO req) {
        ResponseEntity<?> result = null;
        try {

        	AppUserRequestDTO obj = appUserRequestService.update(req);
            result = new ResponseEntity<AppUserRequestDTO>(obj, HttpStatus.OK);
            log.info("Successfully updated app user request for request id: " + obj.getAppUserRequestId());

        } catch (Exception e) {
            log.error("Error updating app user request for requestId : " + req.getAppUserRequestId(), e);
            ErrorVo vo = new ErrorVo("ERR", "There was an error in updating app user request, please contact support.");
            result = new ResponseEntity<ErrorVo>(vo, HttpStatus.INTERNAL_SERVER_ERROR);
        }
        return result;
    }

    /**
     * Get all App User Requests
     * @return All App User Requests
     */
    @RequestMapping(method = RequestMethod.GET)
    public ResponseEntity<?> findAll() {
       ResponseEntity<?> result = null;
       try {

       	   List<AppUserRequestDTO> obj = appUserRequestService.findAll();
           result = new ResponseEntity<List<AppUserRequestDTO>>(obj, HttpStatus.OK);
           log.info("Retrieved all the app user requests, count: " + obj.size());

       } catch (Exception e) {
           log.error("Error retrieving app user requests:", e);
           ErrorVo vo = new ErrorVo("ERR", "There was an error retrieving app user requests, please contact support.");
           result = new ResponseEntity<ErrorVo>(vo, HttpStatus.INTERNAL_SERVER_ERROR);
       }
       return result;
   } 

    /**
     * Get App User Request
     * @param requestId id of the app user request object
     * @return App User Request object
     */
    @RequestMapping(value = "{requestId}", method = RequestMethod.GET)
    public ResponseEntity<?> find(@PathVariable("requestId") long requestId) {
       ResponseEntity<?> result = null;
       try {

       	   AppUserRequestDTO obj = appUserRequestService.findById(requestId);
           result = new ResponseEntity<AppUserRequestDTO>(obj, HttpStatus.OK);
           log.info("Successfully retrieved app user request for request id: " + requestId);

       } catch (Exception e) {
           log.error("Error retrieving app user request for requestId : " + requestId, e);
           ErrorVo vo = new ErrorVo("ERR", "There was an error retrieving app user request, please contact support.");
           result = new ResponseEntity<ErrorVo>(vo, HttpStatus.INTERNAL_SERVER_ERROR);
       }
       return result;
   }	
	
}
