package gov.va.cpss.service;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assume.assumeTrue;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;

import javax.sql.DataSource;

import org.apache.log4j.Logger;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.JobExecution;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.dao.CBSMessageDAO;
import gov.va.cpss.dao.CBSStmtDAO;
import gov.va.cpss.dao.impl.CBSDAOImplUtils;
import gov.va.cpss.dao.impl.CBSDAOTestWrapper;
import gov.va.cpss.dao.impl.SendCBSServiceTestWrapper;
import gov.va.cpss.model.BatchRun;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.model.cbs.CBSMessage;
import gov.va.cpss.model.cbs.CBSStmt;

/**
 * Unit tests the SendCBSService class
 * 
 * @author Brad Pickle
 *
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml", "/dao-test-context.xml" })
public class SendCBSServiceTest {

	@Value("${run.integration.test:false}")
	private Boolean runIntegrationTest;

	private static int MESSAGE_STATUS_INITIAL_ID = 1;

	private static int MESSAGE_STATUS_SUCCESS_ID = 7;

	private static int MESSAGE_STATUS_ERROR_ID = 8;

	private static int STMT_STATUS_NEW_ID = 2;

	private static int STMT_STATUS_SENT_ID = 4;

	@SuppressWarnings("unused")
	private Logger logger = Logger.getLogger(this.getClass().getCanonicalName());

	@Autowired
	private CBSDAOImplUtils cbsDAOImplUtils;

	@Autowired
	private SendCBSService sendCBSService;

	@Autowired
	private CBSMessageDAO cbsMessageDao;

	@Autowired
	private CBSStmtDAO cbsStmtDao;

	private JdbcTemplate jdbcTemplate;

	@Autowired
	public void setDataSource(DataSource dataSource) {
		jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/**
	 * Only run these tests if property is set to run integration test.
	 */
	@Before
	public final void beforeTest() {
		assumeTrue(runIntegrationTest);
	}

	/**
	 * Tests the startIsolatedSendCBSJob() method of SendCBSService. Reading
	 * ProcessStatus.INITIAL and ERROR will fail.
	 */
	@Test(expected = RuntimeException.class)
	public final void testStartIsolatedSendCBSJobNoStatus() {
		final ProcessStatus.Status[] statuses = { ProcessStatus.Status.ERROR, ProcessStatus.Status.INITIAL };

		for (ProcessStatus.Status status : statuses) {
			final ProcessStatus.Status failOnStatus = status;

			(new SendCBSServiceTestWrapper(sendCBSService) {

				@Override
				protected void testImpl() {
					failOnMessageStatus(failOnStatus);

					sendCBSService.startIsolatedSendCBSJob((BatchRun) null);
				}

			}).run(jdbcTemplate);
		}
	}

	/**
	 * Tests the startIsolatedSendCBSJob() method of SendCBSService.
	 */
	@Test
	public final void testStartIsolatedSendCBSJob() {
		(new CBSDAOTestWrapper() {

			@Override
			protected void test() {
				final int batchRunId = 1;
				final Timestamp startDate = new Timestamp((new Date()).getTime());

				final BatchRun batchRun = new BatchRun();
				batchRun.setId(batchRunId);
				batchRun.setStartDate(startDate);

				// This CBSMessage should be updated to ERROR status
				final CBSMessage existingInitialCbsMessage = createTestCBSMessage(getCbsMessageIdS(),
						MESSAGE_STATUS_INITIAL_ID);

				final CBSMessage cbsMessage = sendCBSService.startIsolatedSendCBSJob(batchRun);
				assertTrue(cbsMessage.getId() > 0);
				getCbsMessageIdS().add(cbsMessage.getId());

				assertNotNull(cbsMessage.getFileName());
				assertEquals(batchRunId, cbsMessage.getBatchRunId());
				assertEquals(MESSAGE_STATUS_INITIAL_ID, cbsMessage.getStatusId());
				// assertNotNull(cbsMessage.getCreatedBy());
				assertNotNull(cbsMessage.getCreatedDate());
				// assertNotNull(cbsMessage.getModifiedBy());
				assertNotNull(cbsMessage.getModifiedDate());

				final CBSMessage savedMessage = cbsMessageDao.get(cbsMessage.getId());
				assertNotNull(savedMessage);

				cbsMessage.setModifiedBy(savedMessage.getModifiedBy());
				cbsMessage.setModifiedDate(savedMessage.getModifiedDate());

				cbsDAOImplUtils.compareCBSMessages(cbsMessage, savedMessage);

				// Verify the existing CBSMessage in INITIAL was updated to
				// ERROR
				final CBSMessage existingErrorCbsMessage = cbsMessageDao.get(existingInitialCbsMessage.getId());
				assertNotNull(existingErrorCbsMessage);
				assertEquals(MESSAGE_STATUS_ERROR_ID, existingErrorCbsMessage.getStatusId());
				existingInitialCbsMessage.setStatusId(existingErrorCbsMessage.getStatusId());
				existingInitialCbsMessage.setModifiedDate(existingErrorCbsMessage.getModifiedDate());
				cbsDAOImplUtils.compareCBSMessages(existingInitialCbsMessage, existingErrorCbsMessage);
			}

		}).run(jdbcTemplate);
	}

	/**
	 * Tests the endSendCBSJob() method of SendCBSService. Testing a null
	 * JobExecution.
	 */
	@Test
	public final void testEndSendCBSJobNullExecution() {
		(new SendCBSServiceTestWrapper(sendCBSService) {

			@Override
			protected void testImpl() {
				final CBSMessage cbsMessage = createTestCBSMessage(getCbsMessageIdS(), MESSAGE_STATUS_INITIAL_ID);

				final boolean result = sendCBSService.endSendCBSJob((JobExecution) null, cbsMessage);
				assertEndCBSJobFailed(result, cbsMessage);
			}

		}).run(jdbcTemplate);
	}

	/**
	 * Tests the endSendCBSJob() method of SendCBSService. Testing FAILED exit
	 * status.
	 */
	@Test
	public final void testEndSendCBSJobNonSuccessExistStatus() {
		(new SendCBSServiceTestWrapper(sendCBSService) {

			@Override
			protected void testImpl() {
				final CBSMessage cbsMessage = createTestCBSMessage(getCbsMessageIdS(), MESSAGE_STATUS_INITIAL_ID);

				final long jobExecutionId = 1L;
				final ExitStatus jobExecutionExitStatus = ExitStatus.FAILED;

				final JobExecution jobExecution = new JobExecution(jobExecutionId);
				jobExecution.setExitStatus(jobExecutionExitStatus);

				final boolean result = sendCBSService.endSendCBSJob(jobExecution, cbsMessage);
				assertEndCBSJobFailed(result, cbsMessage);
			}

		}).run(jdbcTemplate);
	}

	/**
	 * Tests the endSendCBSJob() method of SendCBSService. Testing failure to
	 * read ProcessStatus ERROR.
	 */
	@Test
	public final void testEndSendCBSFailReadMessageStatus() {
		(new SendCBSServiceTestWrapper(sendCBSService) {

			@Override
			protected void testImpl() {
				final CBSMessage cbsMessage = createTestCBSMessage(getCbsMessageIdS(), MESSAGE_STATUS_INITIAL_ID);

				final long jobExecutionId = 1L;
				final ExitStatus jobExecutionExitStatus = ExitStatus.FAILED;

				final JobExecution jobExecution = new JobExecution(jobExecutionId);
				jobExecution.setExitStatus(jobExecutionExitStatus);

				// Make getting this status return null
				failOnMessageStatus(ProcessStatus.Status.ERROR);

				final boolean result = sendCBSService.endSendCBSJob(jobExecution, cbsMessage);
				assertEndCBSJobFailed(result, (CBSMessage) null);
			}

		}).run(jdbcTemplate);
	}

	/**
	 * Tests the deleteCBSOutputResource() method of SendCBSService.
	 */
	@Test
	public final void testDeleteCBSOutputResource() {
		(new SendCBSServiceTestWrapper(sendCBSService) {

			@Override
			protected void testImpl() {
				final CBSMessage cbsMessage = createTestCBSMessage(getCbsMessageIdS(), MESSAGE_STATUS_INITIAL_ID);
				touchStatementFiles(cbsMessage);
				assertTrue(sendCBSService.deleteCBSOutputResource(sendCBSService.getCBSOutputResource(cbsMessage.getCBSFileName())));
				assertTrue(sendCBSService.deleteCBSOutputResource(sendCBSService.getCBSOutputResource(cbsMessage.getSBSFileName())));
				assertFalse(getCbsFile().exists());
				assertFalse(getSbsFile().exists());
			}

		}).run(jdbcTemplate);
	}

	/**
	 * Tests the updateAllStatementStatusToSentForMessageId() method of SendCBSService. Testing failure to
	 * read CBSStmtStatus SENT.
	 */
	@Test
	public final void testUpdateAllStatementStatusToSentForMessageIdFailReadStmtStatus() {
		(new SendCBSServiceTestWrapper(sendCBSService) {

			@Override
			protected void testImpl() {
				final CBSMessage cbsMessage = createTestCBSMessage(getCbsMessageIdS(), MESSAGE_STATUS_INITIAL_ID);

				// Make getting this status return null
				failOnStmtStatus(ProcessStatus.Status.SENT); 

				assertEquals(-1, sendCBSService.updateAllStatementStatusToSentForMessageId(cbsMessage.getId()));
			}

		}).run(jdbcTemplate);
	}

	/**
	 * Tests the updateAllStatementStatusToSentForMessageId() method of SendCBSService.
	 */
	@Test
	public final void testUpdateAllStatementStatusToSentForMessageId() {
		(new SendCBSServiceTestWrapper(sendCBSService) {

			@Override
			protected void testImpl() {
				final long stationNum = 1L;
				final int batchRunId = 1;
				final String fileName = "cbsTestFileName.txt";

				// Create a statement
				final CBSStmt cbsStmt = cbsDAOImplUtils.createSimpleTestCBSStmt(1, STMT_STATUS_NEW_ID, stationNum);
				cbsDAOImplUtils.saveCBSStmt(cbsStmt);
				getCbsStmtIdS().add(cbsStmt.getId());
				getAccountIdS().add(cbsStmt.getAccountId());

				// Create the CBSMessage record for the statement
				final List<CBSMessage> cbsMessagesL = new ArrayList<CBSMessage>();
				final long messageId = cbsDAOImplUtils.insertTestCBSMessage(cbsMessagesL, MESSAGE_STATUS_SUCCESS_ID,
						batchRunId, fileName);
				getCbsMessageIdS().add(messageId);
				cbsDAOImplUtils.updateMessageIdForStmt(cbsStmt.getId(), messageId);
				cbsStmt.setMessageId(messageId);

				CBSMessage cbsMessage = cbsMessagesL.get(0);

				assertEquals(1, sendCBSService.updateAllStatementStatusToSentForMessageId(cbsMessage.getId()));

				CBSStmt updatedCBSStmt = cbsStmtDao.get(cbsStmt.getId());

				assertNotNull(updatedCBSStmt);
				assertEquals(STMT_STATUS_SENT_ID, updatedCBSStmt.getStatusId());

				cbsStmt.setStatusId(updatedCBSStmt.getStatusId());
				cbsStmt.setModifiedBy(updatedCBSStmt.getModifiedBy());
				cbsStmt.setModifiedDate(updatedCBSStmt.getModifiedDate());

				cbsDAOImplUtils.compareCBSStmt(cbsStmt, updatedCBSStmt);
			}

		}).run(jdbcTemplate);
	}

	/**
	 * Tests the setMessageStatus() method of SendCBSService. Testing failure to
	 * read ProcessStatus ERROR.
	 */
	@Test
	public final void testSetMessageStatusFailReadMessageStatus() {
		(new SendCBSServiceTestWrapper(sendCBSService) {

			@Override
			protected void testImpl() {
				final CBSMessage cbsMessage = createTestCBSMessage(getCbsMessageIdS(), MESSAGE_STATUS_INITIAL_ID);

				// Make getting this status return null
				failOnMessageStatus(ProcessStatus.Status.SUCCESS);

				assertFalse(sendCBSService.setMessageStatus(cbsMessage.getId(), ProcessStatus.Status.SUCCESS));
			}

		}).run(jdbcTemplate);
	}

	/**
	 * Tests the SetMessageStatus() method of SendCBSService. Testing successful
	 * conditions.
	 */
	@Test
	public final void testSetMessageStatusSuccess() {
		(new SendCBSServiceTestWrapper(sendCBSService) {

			@Override
			protected void testImpl() {
				final CBSMessage cbsMessage = createTestCBSMessage(getCbsMessageIdS(), MESSAGE_STATUS_INITIAL_ID);

				assertTrue(sendCBSService.setMessageStatus(cbsMessage.getId(), ProcessStatus.Status.SUCCESS));
				
				CBSMessage updatedCBSMessage = cbsMessageDao.get(cbsMessage.getId());
				assertNotNull(updatedCBSMessage);
				assertEquals(MESSAGE_STATUS_SUCCESS_ID, updatedCBSMessage.getStatusId());

				cbsMessage.setStatusId(updatedCBSMessage.getStatusId());
				cbsMessage.setModifiedBy(updatedCBSMessage.getModifiedBy());
				cbsMessage.setModifiedDate(updatedCBSMessage.getModifiedDate());

				cbsDAOImplUtils.compareCBSMessages(cbsMessage, updatedCBSMessage);

			}

		}).run(jdbcTemplate);
	}

	/**
	 * Tests the getMessageStatementsForSite() method of SendCBSService. Tests
	 * loadSiteStatements false.
	 */
	@Test
	public final void testGetMessageStatementsForSite() {
		for (boolean loadSiteStatements : new boolean[] { false, true }) {
			(new CBSDAOTestWrapper() {

				@Override
				protected void test() {
					final long stationNum1 = 1L;
					final long stationNum2 = 2L;
					final long stationNum3 = 3L;

					final int stmtStatusNewId = 2;
					final int stmtStatusSentId = 3;

					final int messageStatusInitialId = 1;
					final int messageStatusSuccessId = 2;

					final int sentBatchRunId = 1;
					final String sentFileName = "cbsTestFileNameSent.txt";

					final List<CBSMessage> cbsMessagesL = new ArrayList<CBSMessage>();

					// Create an already sent statement for station 1
					final CBSStmt sentCBSStmt1 = cbsDAOImplUtils.createSimpleTestCBSStmt(1, stmtStatusSentId,
							stationNum1);
					cbsDAOImplUtils.saveCBSStmt(sentCBSStmt1);
					getCbsStmtIdS().add(sentCBSStmt1.getId());
					getAccountIdS().add(sentCBSStmt1.getAccountId());

					// Create the CBSMessage record for sent statement
					final long sentMessageId = cbsDAOImplUtils.insertTestCBSMessage(cbsMessagesL,
							messageStatusSuccessId, sentBatchRunId, sentFileName);
					getCbsMessageIdS().add(sentMessageId);
					cbsDAOImplUtils.updateMessageIdForStmt(sentCBSStmt1.getId(), sentMessageId);
					sentCBSStmt1.setMessageId(sentMessageId);

					final List<CBSStmt> simpleResultsL = sendCBSService.getMessageStatementsForSite(sentMessageId,
							"" + stationNum1, loadSiteStatements);
					assertNotNull(simpleResultsL);
					assertEquals(simpleResultsL.size(), 1);
					assertNotNull(simpleResultsL.get(0));

					compareCBSStmtForLoadSiteStatements(sentCBSStmt1, simpleResultsL.get(0), loadSiteStatements);

					// Create a NEW statement for station 1
					final CBSStmt newCBSStmt1 = cbsDAOImplUtils.createSimpleTestCBSStmt(2, stmtStatusNewId,
							stationNum1);
					cbsDAOImplUtils.saveCBSStmt(newCBSStmt1);
					getCbsStmtIdS().add(newCBSStmt1.getId());
					getAccountIdS().add(newCBSStmt1.getAccountId());

					// Create a NEW statement for station 1 and station2.
					// station 1
					// is
					// primary
					final CBSStmt newCBSStmt1and2 = cbsDAOImplUtils.createDoubleTestCBSStmt(3, stmtStatusNewId,
							stationNum1, stationNum2);
					cbsDAOImplUtils.saveCBSStmt(newCBSStmt1and2);
					getCbsStmtIdS().add(newCBSStmt1and2.getId());
					getAccountIdS().add(newCBSStmt1and2.getAccountId());

					// Create a NEW statement for station 2 and station3.
					// station 2
					// is
					// primary
					final CBSStmt newCBSStmt2and3 = cbsDAOImplUtils.createDoubleTestCBSStmt(4, stmtStatusNewId,
							stationNum2, stationNum3);
					cbsDAOImplUtils.saveCBSStmt(newCBSStmt2and3);
					getCbsStmtIdS().add(newCBSStmt2and3.getId());
					getAccountIdS().add(newCBSStmt2and3.getAccountId());

					// Create the CBSMessage record for NEW statement
					final int newBatchRunId = 2;
					final String newFileName = "cbsTestFileNameNew.txt";
					final long newMessageId = cbsDAOImplUtils.insertTestCBSMessage(cbsMessagesL, messageStatusInitialId,
							newBatchRunId, newFileName);
					getCbsMessageIdS().add(newMessageId);
					cbsDAOImplUtils.updateMessageIdForStmt(newCBSStmt1.getId(), newMessageId);
					cbsDAOImplUtils.updateMessageIdForStmt(newCBSStmt1and2.getId(), newMessageId);
					cbsDAOImplUtils.updateMessageIdForStmt(newCBSStmt2and3.getId(), newMessageId);
					newCBSStmt1.setMessageId(newMessageId);
					newCBSStmt1and2.setMessageId(newMessageId);
					newCBSStmt2and3.setMessageId(newMessageId);

					final List<CBSStmt> doubleStat1ResultsL = sendCBSService.getMessageStatementsForSite(newMessageId,
							"" + stationNum1, loadSiteStatements);
					assertNotNull(doubleStat1ResultsL);
					assertEquals(doubleStat1ResultsL.size(), 2);
					final List<CBSStmt> newStmts = new ArrayList<CBSStmt>();
					newStmts.add(newCBSStmt1);
					newStmts.add(newCBSStmt1and2);
					for (CBSStmt cbsStmt : doubleStat1ResultsL) {
						assertNotNull(cbsStmt);
						CBSStmt expectedCbsStmt = CBSDAOImplUtils.getStmtWithId(newStmts, cbsStmt.getId());
						assertNotNull(expectedCbsStmt);
						compareCBSStmtForLoadSiteStatements(expectedCbsStmt, cbsStmt, loadSiteStatements);
					}

					final List<CBSStmt> doubleStat2ResultsL = sendCBSService.getMessageStatementsForSite(newMessageId,
							"" + stationNum2, loadSiteStatements);
					assertNotNull(doubleStat2ResultsL);
					assertEquals(doubleStat2ResultsL.size(), 1);
					compareCBSStmtForLoadSiteStatements(newCBSStmt2and3, doubleStat2ResultsL.get(0),
							loadSiteStatements);

					final List<CBSStmt> doubleStat3ResultsL = sendCBSService.getMessageStatementsForSite(newMessageId,
							"" + stationNum3, loadSiteStatements);
					assertNotNull(doubleStat3ResultsL);
					assertEquals(doubleStat3ResultsL.size(), 0);
				}

			}).run(jdbcTemplate);
		}
	}

	private void compareCBSStmtForLoadSiteStatements(CBSStmt newCBSStmt, CBSStmt expectedCBSStmt,
			boolean loadSiteStatements) {
		if (loadSiteStatements) {
			cbsDAOImplUtils.compareCBSStmtDeep(newCBSStmt, expectedCBSStmt);
		} else {
			cbsDAOImplUtils.compareCBSStmt(newCBSStmt, expectedCBSStmt);
		}
	}

	private void assertEndCBSJobFailed(final boolean result, final CBSMessage cbsMessage) {
		assertFalse(result);

		if (cbsMessage != null) {
			CBSMessage updatedCBSMessage = cbsMessageDao.get(cbsMessage.getId());
			assertNotNull(updatedCBSMessage);
			assertEquals(MESSAGE_STATUS_ERROR_ID, updatedCBSMessage.getStatusId());

			cbsMessage.setStatusId(updatedCBSMessage.getStatusId());
			cbsMessage.setModifiedBy(updatedCBSMessage.getModifiedBy());
			cbsMessage.setModifiedDate(updatedCBSMessage.getModifiedDate());

			cbsDAOImplUtils.compareCBSMessages(cbsMessage, updatedCBSMessage);
		}
	}

	private CBSMessage createTestCBSMessage(final Set<Long> cbsMessageIdS, final int statusId) {
		final long batchRunId = 1;
		final String fileName = "cbsTestFileName.txt,sbsTestFileName.txt";

		// Assume functioning CBSMessage
		final CBSMessage cbsMessage = new CBSMessage();

		cbsMessage.setStatusId(statusId);
		cbsMessage.setBatchRunId(batchRunId);
		cbsMessage.setFileName(fileName);

		final long id = cbsMessageDao.insert(cbsMessage);
		cbsMessageIdS.add(id);

		return cbsMessage;
	}

}
