package gov.va.cpss.service;

import static org.junit.Assert.assertEquals;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.cpss.model.cbs.CBSSiteStmt;
import gov.va.cpss.model.cbs.CBSStmt;
import gov.va.cpss.model.fps.PSPatient;

/**
 * Integration Unit Tests to test business logic of Generate CBS batch
 * processing.
 * 
 * @author DNS  
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "/test-context.xml", "/cpss-context.xml", "/cpss-batch.xml" })
public class CbsServiceTest {

	private CbsServiceTestPatients testPatients = new CbsServiceTestPatients();

	@Autowired
	private CbsService cbsService;

	@Test
	public final void testCalculateTotals() {

		Map<Long, List<PSPatient>> patientMap = new HashMap<>();
		patientMap.put(1234L, testPatients.getTestPatientList());

		CBSStmt statement = cbsService.buildStatementForPatientMap(1, patientMap.entrySet().iterator().next());

		assertEquals("1001.00", statement.getAmountDue().getDoubleAsString());
		assertEquals("101.00", statement.getPrevBalance().getDoubleAsString());
		assertEquals("11.00", statement.getTotalCharges().getDoubleAsString());
		assertEquals("111.10", statement.getTotalCredits().getDoubleAsString());
		assertEquals("101.10", statement.getNewBalance().getDoubleAsString());
	}

	@Test
	public final void testPrimarySiteFlags() {

		Map<Long, List<PSPatient>> patientMap = new HashMap<>();
		patientMap.put(1234L, testPatients.getTestPatientList());

		CBSStmt statement = cbsService.buildStatementForPatientMap(1, patientMap.entrySet().iterator().next());

		List<CBSSiteStmt> siteStatementL = statement.getSiteStmtL();

		CBSSiteStmt primarySiteStatement = null;
		CBSSiteStmt primaryAddressSiteStatement = null;

		int primarySiteStatementCounter = 0;
		int primaryAddressSiteStatementCounter = 0;
		for (CBSSiteStmt siteStatement : siteStatementL) {
			if (siteStatement.getIsPrimary().isTrue()) {
				primarySiteStatement = siteStatement;
				primarySiteStatementCounter++;
			}
			if (siteStatement.getIsPrimaryAddress().isTrue()) {
				primaryAddressSiteStatement = siteStatement;
				primaryAddressSiteStatementCounter++;
			}
		}

		assertEquals(1, primarySiteStatementCounter);
		assertEquals(1, primaryAddressSiteStatementCounter);

		assertEquals(testPatients.getPatient3().getLastBillPrepDate(), primarySiteStatement.getLastBillPrepDate());
		assertEquals(testPatients.getPatient3().getLastBillPrepDate(),
				primaryAddressSiteStatement.getLastBillPrepDate());
	}

	@Test
	public final void testPrimarySiteFlagsReversedInput() {

		Map<Long, List<PSPatient>> patientMap = new HashMap<>();
		List<PSPatient> reversedList = testPatients.getTestPatientList();
		Collections.reverse(reversedList);

		patientMap.put(1234L, reversedList);

		CBSStmt statement = cbsService.buildStatementForPatientMap(1, patientMap.entrySet().iterator().next());

		List<CBSSiteStmt> siteStatementL = statement.getSiteStmtL();

		CBSSiteStmt primarySiteStatement = null;
		CBSSiteStmt primaryAddressSiteStatement = null;

		int primarySiteStatementCounter = 0;
		int primaryAddressSiteStatementCounter = 0;
		for (CBSSiteStmt siteStatement : siteStatementL) {
			if (siteStatement.getIsPrimary().isTrue()) {
				primarySiteStatement = siteStatement;
				primarySiteStatementCounter++;
			}
			if (siteStatement.getIsPrimaryAddress().isTrue()) {
				primaryAddressSiteStatement = siteStatement;
				primaryAddressSiteStatementCounter++;
			}
		}

		assertEquals(1, primarySiteStatementCounter);
		assertEquals(1, primaryAddressSiteStatementCounter);

		assertEquals(testPatients.getPatient3().getLastBillPrepDate(), primarySiteStatement.getLastBillPrepDate());
		assertEquals(testPatients.getPatient3().getLastBillPrepDate(),
				primaryAddressSiteStatement.getLastBillPrepDate());
	}

	@Test
	public final void testPrimarySiteArFlagsInput() {

		Map<Long, List<PSPatient>> patientMap = new HashMap<>();
		List<PSPatient> reversedList = testPatients.getArTestPatientList();
		Collections.reverse(reversedList);

		patientMap.put(1234L, reversedList);

		CBSStmt statement = cbsService.buildStatementForPatientMap(1, patientMap.entrySet().iterator().next());

		List<CBSSiteStmt> siteStatementL = statement.getSiteStmtL();

		CBSSiteStmt primarySiteStatement = null;
		CBSSiteStmt primaryAddressSiteStatement = null;

		int primarySiteStatementCounter = 0;
		int primaryAddressSiteStatementCounter = 0;
		for (CBSSiteStmt siteStatement : siteStatementL) {
			if (siteStatement.getIsPrimary().isTrue()) {
				primarySiteStatement = siteStatement;
				primarySiteStatementCounter++;
			}
			if (siteStatement.getIsPrimaryAddress().isTrue()) {
				primaryAddressSiteStatement = siteStatement;
				primaryAddressSiteStatementCounter++;
			}
		}

		assertEquals(1, primarySiteStatementCounter);
		assertEquals(1, primaryAddressSiteStatementCounter);

		assertEquals(testPatients.getPatient3().getLastBillPrepDate(), primarySiteStatement.getLastBillPrepDate());
		assertEquals(testPatients.getPatient1().getLastBillPrepDate(),
				primaryAddressSiteStatement.getLastBillPrepDate());
	}

	@Test
	public final void testPrimarySiteMultipleArFlagsInput() {

		Map<Long, List<PSPatient>> patientMap = new HashMap<>();
		List<PSPatient> reversedList = testPatients.getMultipleArTestPatientList();
		Collections.reverse(reversedList);

		patientMap.put(1234L, reversedList);

		CBSStmt statement = cbsService.buildStatementForPatientMap(1, patientMap.entrySet().iterator().next());

		List<CBSSiteStmt> siteStatementL = statement.getSiteStmtL();

		CBSSiteStmt primarySiteStatement = null;
		CBSSiteStmt primaryAddressSiteStatement = null;

		int primarySiteStatementCounter = 0;
		int primaryAddressSiteStatementCounter = 0;
		for (CBSSiteStmt siteStatement : siteStatementL) {
			if (siteStatement.getIsPrimary().isTrue()) {
				primarySiteStatement = siteStatement;
				primarySiteStatementCounter++;
			}
			if (siteStatement.getIsPrimaryAddress().isTrue()) {
				primaryAddressSiteStatement = siteStatement;
				primaryAddressSiteStatementCounter++;
			}
		}

		assertEquals(1, primarySiteStatementCounter);
		assertEquals(1, primaryAddressSiteStatementCounter);

		assertEquals(testPatients.getPatient3().getLastBillPrepDate(), primarySiteStatement.getLastBillPrepDate());
		assertEquals(testPatients.getPatient3().getLastBillPrepDate(),
				primaryAddressSiteStatement.getLastBillPrepDate());
	}

	@Test
	public final void testMergeWithExistingStatement() {

		Map<Long, List<PSPatient>> patientMap = new HashMap<>();
		patientMap.put(1234L, testPatients.getTestPatientList());

		// Newly Processed Statement.
		CBSStmt statement = cbsService.buildStatementForPatientMap(1, patientMap.entrySet().iterator().next());

		List<CBSSiteStmt> siteStatementL = statement.getSiteStmtL();

		CBSSiteStmt primarySiteStatement = null;
		CBSSiteStmt primaryAddressSiteStatement = null;

		int primarySiteStatementCounter = 0;
		int primaryAddressSiteStatementCounter = 0;
		for (CBSSiteStmt siteStatement : siteStatementL) {
			if (siteStatement.getIsPrimary().isTrue()) {
				primarySiteStatement = siteStatement;
				primarySiteStatementCounter++;
			}
			if (siteStatement.getIsPrimaryAddress().isTrue()) {
				primaryAddressSiteStatement = siteStatement;
				primaryAddressSiteStatementCounter++;
			}
		}

		assertEquals(1, primarySiteStatementCounter);
		assertEquals(1, primaryAddressSiteStatementCounter);

		assertEquals(testPatients.getPatient3().getLastBillPrepDate(), primarySiteStatement.getLastBillPrepDate());
		assertEquals(testPatients.getPatient3().getLastBillPrepDate(),
				primaryAddressSiteStatement.getLastBillPrepDate());

		// Existing Statement
		patientMap = new HashMap<>();
		patientMap.put(1234L, testPatients.getTestPatientList2());

		CBSStmt existingStatement = cbsService.buildStatementForPatientMap(1,
				patientMap.entrySet().iterator().next());

		assertEquals("122.08", existingStatement.getAmountDue().getDoubleAsString());
		assertEquals("30.57", existingStatement.getPrevBalance().getDoubleAsString());
		assertEquals("10.90", existingStatement.getTotalCharges().getDoubleAsString());
		assertEquals("12.90", existingStatement.getTotalCredits().getDoubleAsString());
		assertEquals("0.51", existingStatement.getNewBalance().getDoubleAsString());

		List<CBSSiteStmt> existingSiteStatementL = existingStatement.getSiteStmtL();

		CBSSiteStmt existingPrimarySiteStatement = null;
		CBSSiteStmt existingPrimaryAddressSiteStatement = null;

		int existingPrimarySiteStatementCounter = 0;
		int existingPrimaryAddressSiteStatementCounter = 0;
		for (CBSSiteStmt siteStatement : existingSiteStatementL) {
			if (siteStatement.getIsPrimary().isTrue()) {
				existingPrimarySiteStatement = siteStatement;
				existingPrimarySiteStatementCounter++;
			}
			if (siteStatement.getIsPrimaryAddress().isTrue()) {
				existingPrimaryAddressSiteStatement = siteStatement;
				existingPrimaryAddressSiteStatementCounter++;
			}
		}

		assertEquals(1, existingPrimarySiteStatementCounter);
		assertEquals(1, existingPrimaryAddressSiteStatementCounter);

		assertEquals(testPatients.getPatient6().getLastBillPrepDate(),
				existingPrimarySiteStatement.getLastBillPrepDate());
		assertEquals(testPatients.getPatient5().getLastBillPrepDate(),
				existingPrimaryAddressSiteStatement.getLastBillPrepDate());

		// Now combine the two statements.
		cbsService.combineWithExistingStatement(statement, existingStatement);

		assertEquals("1123.08", statement.getAmountDue().getDoubleAsString());
		assertEquals("131.57", statement.getPrevBalance().getDoubleAsString());
		assertEquals("21.90", statement.getTotalCharges().getDoubleAsString());
		assertEquals("124.00", statement.getTotalCredits().getDoubleAsString());
		assertEquals("101.61", statement.getNewBalance().getDoubleAsString());

		List<CBSSiteStmt> mergedSiteStatementL = statement.getSiteStmtL();

		CBSSiteStmt mergedPrimarySiteStatement = null;
		CBSSiteStmt mergedPrimaryAddressSiteStatement = null;

		int mergedPrimarySiteStatementCounter = 0;
		int mergedPrimaryAddressSiteStatementCounter = 0;
		for (CBSSiteStmt siteStatement : mergedSiteStatementL) {
			if (siteStatement.getIsPrimary().isTrue()) {
				mergedPrimarySiteStatement = siteStatement;
				mergedPrimarySiteStatementCounter++;
			}
			if (siteStatement.getIsPrimaryAddress().isTrue()) {
				mergedPrimaryAddressSiteStatement = siteStatement;
				mergedPrimaryAddressSiteStatementCounter++;
			}
		}

		assertEquals(1, mergedPrimarySiteStatementCounter);
		assertEquals(1, mergedPrimaryAddressSiteStatementCounter);

		assertEquals(testPatients.getPatient3().getLastBillPrepDate(),
				mergedPrimarySiteStatement.getLastBillPrepDate());
		assertEquals(testPatients.getPatient5().getLastBillPrepDate(),
				mergedPrimaryAddressSiteStatement.getLastBillPrepDate());
	}

}
