package gov.va.cpss.job.sendcbs;

import static org.junit.Assert.*;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.junit.Test;

import gov.va.cpss.cobol.Money;
import gov.va.cpss.dao.impl.CBSDAOImplUtils;
import gov.va.cpss.model.BooleanChar;
import gov.va.cpss.model.cbs.CBSSiteStmt;
import gov.va.cpss.model.cbs.CBSStmt;
import gov.va.cpss.model.fps.PS1;
import gov.va.cpss.model.fps.PSDetails;
import gov.va.cpss.model.fps.PSPatient;
import gov.va.cpss.model.fps.PSRecord;
import gov.va.cpss.model.fps.PSSite;
import gov.va.cpss.model.fps.PSSiteStmt;

/**
 * Tests for SendCBSWritePSRecordSource.
 * 
 * @author Brad Pickle
 *
 */
public class SendCBSWritePSRecordSourceTest {

	private static int STMT_STATUS_NEW_ID = 2;

	/**
	 * Test SendCBSWritePSRecordSourceTest using test statements created with
	 * getCBSStmtList1(). Expect successful run.
	 */
	@Test
	public void test() {
		final SendCBSWritePSRecordSource psRecordSource = new SendCBSWritePSRecordSource();
		
		final SendCBSRuntimeState runtimeState = new SendCBSRuntimeStateImpl();
		psRecordSource.setSendCBSRuntimeState(runtimeState);

		final List<PSSite> psSitesList = runtimeState.getPSSitesList();
		final Map<String, List<CBSStmt>> cbsListsBySite = runtimeState.getCBSListsBySite();
		final Map<String, List<CBSStmt>> sbsListsBySite = runtimeState.getSBSListsBySite();
		
		getCBSStmts1(psSitesList, cbsListsBySite, sbsListsBySite);
				
		// PS
		PSRecord psRecord = psRecordSource.nextPSRecord();
		assertNotNull(psRecord);
		assertTrue(psRecord instanceof PSSite);
		
		// P1
		psRecord = psRecordSource.nextPSRecord();
		assertNotNull(psRecord);
		assertTrue(psRecord instanceof PS1);
		PS1 ps1 = (PS1)psRecord;
		
		assertEquals(PSRecord.DataType.P1, ps1.getType());
		assertEquals("1000000001TESTL", ps1.getPatientAccount());
		assertEquals("ICN1", ps1.getIcnNumber());
		assertEquals("11TESTL", ps1.getAcntNumDisp());
		assertEquals(BooleanChar.N, ps1.getLateSiteFlag());
		
		// P1
		psRecord = psRecordSource.nextPSRecord();
		assertNotNull(psRecord);
		assertTrue(psRecord instanceof PS1);
		ps1 = (PS1)psRecord;
		
		assertEquals(PSRecord.DataType.P1, ps1.getType());
		assertEquals("1000000002TESTL", ps1.getPatientAccount());
		assertEquals("ICN2", ps1.getIcnNumber());
		assertEquals("12TESTL", ps1.getAcntNumDisp());
		assertEquals(BooleanChar.Y, ps1.getLateSiteFlag());
		
		// PS
		psRecord = psRecordSource.nextPSRecord();
		assertNotNull(psRecord);
		assertTrue(psRecord instanceof PSSite);
		
		// PH
		psRecord = psRecordSource.nextPSRecord();
		assertNotNull(psRecord);
		assertTrue(psRecord instanceof PSPatient);
		PSPatient psPatient = (PSPatient)psRecord;
		
		assertEquals(PSRecord.DataType.PH, psPatient.getType());
		
		assertEquals("1000000003TESTL", psPatient.getPatientAccount());
		assertEquals("TESTLast", psPatient.getPatientLastName());
		assertEquals("TestFirst", psPatient.getPatientFirstName());
		assertEquals("TestMiddle", psPatient.getPatientMiddleName());
		
		assertEquals("TestAddress1", psPatient.getAddress1());
		assertEquals("TestAddress2", psPatient.getAddress2());
		assertEquals("TestAddress3", psPatient.getAddress3());
		assertEquals("TestCity", psPatient.getCity());
		assertEquals("OH", psPatient.getState());
		assertEquals("12345", psPatient.getZipCode());
		assertEquals("TestCountry", psPatient.getCountryName());

		assertEquals("00000540{", psPatient.getAmountDue().getCobol());
		assertEquals("00000800{", psPatient.getPrevBalance().getCobol());
		assertEquals("00000540{", psPatient.getTotalCharges().getCobol());
		assertEquals("00000800}", psPatient.getTotalCredits().getCobol());
		assertEquals("00000540{", psPatient.getNewBalance().getCobol());

		assertEquals("Test Special Notes", psPatient.getSpecialNotes());
		assertEquals("", psPatient.getSpecialNotesForLateSites1());
		assertEquals("", psPatient.getSpecialNotesForLateSites2());
		assertEquals("010203", psPatient.getRightsObligationParagraphCodes());

		assertEquals(6, psPatient.getNumOfLines());
		assertEquals(100000000003L, psPatient.getDfnNumber());
		assertEquals(BooleanChar.N, psPatient.getLargeFontInd());
		assertEquals("ICN3", psPatient.getIcnNumber());
		assertEquals(BooleanChar.Y, psPatient.getAddressFlag());
		assertEquals("05252016", psPatient.getLastBillPrepDate().toString());
		assertEquals("13TESTL", psPatient.getAcntNumDisp());
		assertEquals(2, psPatient.getNumConsolidatedSites());
		assertEquals(BooleanChar.N, psPatient.getLateSiteFlag());
	
		// PV
		psRecord = psRecordSource.nextPSRecord();
		assertNotNull(psRecord);
		assertTrue(psRecord instanceof PSSiteStmt);
		PSSiteStmt psSiteStmt = (PSSiteStmt)psRecord;
		
		assertEquals(PSRecord.DataType.PV, psSiteStmt.getType());
		assertEquals("1", psSiteStmt.getFacilityNum());
		assertEquals("00000270{", psSiteStmt.getStationTotalAmount().getCobol());
		assertEquals(2, psSiteStmt.getNumOfLines());
		assertEquals("1000000003TESTL", psSiteStmt.getPatientAccount());
		assertEquals(100000000003L, psSiteStmt.getDfnNumber());
		assertEquals("ICN3", psSiteStmt.getIcnNumber());

		// PD
		psRecord = psRecordSource.nextPSRecord();
		assertNotNull(psRecord);
		assertTrue(psRecord instanceof PSDetails);
		PSDetails psDetails = (PSDetails)psRecord;
		
		assertEquals(PSRecord.DataType.PD, psDetails.getType());
		assertEquals("05012016", psDetails.getDatePosted().toString());
		assertEquals("PAYMENT", psDetails.getTransDesc());
		assertEquals("00000400}", psDetails.getTransAmount().getCobol());
		assertEquals("REF_1", psDetails.getReferenceNum());
		assertEquals("ICN3", psDetails.getIcnNumber());
		assertEquals(1, psDetails.getSeqNum());
		
		// PD
		psRecord = psRecordSource.nextPSRecord();
		assertNotNull(psRecord);
		assertTrue(psRecord instanceof PSDetails);
		psDetails = (PSDetails)psRecord;
		
		assertEquals(PSRecord.DataType.PD, psDetails.getType());
		assertEquals("05012016", psDetails.getDatePosted().toString());
		assertEquals("CHARGE", psDetails.getTransDesc());
		assertEquals("00000270{", psDetails.getTransAmount().getCobol());
		assertEquals("REF_2", psDetails.getReferenceNum());
		assertEquals("ICN3", psDetails.getIcnNumber());
		assertEquals(2, psDetails.getSeqNum());
		
		// PV
		psRecord = psRecordSource.nextPSRecord();
		assertNotNull(psRecord);
		assertTrue(psRecord instanceof PSSiteStmt);
		psSiteStmt = (PSSiteStmt)psRecord;
		
		assertEquals(PSRecord.DataType.PV, psSiteStmt.getType());
		assertEquals("2", psSiteStmt.getFacilityNum());
		assertEquals("00000270{", psSiteStmt.getStationTotalAmount().getCobol());
		assertEquals(2, psSiteStmt.getNumOfLines());
		assertEquals("2000000003TESTL", psSiteStmt.getPatientAccount());
		assertEquals(200000000003L, psSiteStmt.getDfnNumber());
		assertEquals("ICN3", psSiteStmt.getIcnNumber());

		// PD
		psRecord = psRecordSource.nextPSRecord();
		assertNotNull(psRecord);
		assertTrue(psRecord instanceof PSDetails);
		psDetails = (PSDetails)psRecord;
		
		assertEquals(PSRecord.DataType.PD, psDetails.getType());
		assertEquals("05012016", psDetails.getDatePosted().toString());
		assertEquals("PAYMENT", psDetails.getTransDesc());
		assertEquals("00000400}", psDetails.getTransAmount().getCobol());
		assertEquals("REF_1", psDetails.getReferenceNum());
		assertEquals("ICN3", psDetails.getIcnNumber());
		assertEquals(1, psDetails.getSeqNum());
		
		// PD
		psRecord = psRecordSource.nextPSRecord();
		assertNotNull(psRecord);
		assertTrue(psRecord instanceof PSDetails);
		psDetails = (PSDetails)psRecord;
		
		assertEquals(PSRecord.DataType.PD, psDetails.getType());
		assertEquals("05012016", psDetails.getDatePosted().toString());
		assertEquals("CHARGE", psDetails.getTransDesc());
		assertEquals("00000270{", psDetails.getTransAmount().getCobol());
		assertEquals("REF_2", psDetails.getReferenceNum());
		assertEquals("ICN3", psDetails.getIcnNumber());
		assertEquals(2, psDetails.getSeqNum());

		// PS
		psRecord = psRecordSource.nextPSRecord();
		assertNotNull(psRecord);
		assertTrue(psRecord instanceof PSSite);
		
		// PH
		psRecord = psRecordSource.nextPSRecord();
		assertNotNull(psRecord);
		assertTrue(psRecord instanceof PSPatient);
		psPatient = (PSPatient)psRecord;
		
		assertEquals(PSRecord.DataType.PH, psPatient.getType());
		
		assertEquals("2000000004TESTL", psPatient.getPatientAccount());
		assertEquals("TESTLast", psPatient.getPatientLastName());
		assertEquals("TestFirst", psPatient.getPatientFirstName());
		assertEquals("TestMiddle", psPatient.getPatientMiddleName());
		
		assertEquals("TestAddress1", psPatient.getAddress1());
		assertEquals("TestAddress2", psPatient.getAddress2());
		assertEquals("TestAddress3", psPatient.getAddress3());
		assertEquals("TestCity", psPatient.getCity());
		assertEquals("OH", psPatient.getState());
		assertEquals("12345", psPatient.getZipCode());
		assertEquals("TestCountry", psPatient.getCountryName());

		assertEquals("00000540{", psPatient.getAmountDue().getCobol());
		assertEquals("00000800{", psPatient.getPrevBalance().getCobol());
		assertEquals("00000540{", psPatient.getTotalCharges().getCobol());
		assertEquals("00000800}", psPatient.getTotalCredits().getCobol());
		assertEquals("00000540{", psPatient.getNewBalance().getCobol());

		assertEquals("Test Special Notes", psPatient.getSpecialNotes());
		assertEquals("", psPatient.getSpecialNotesForLateSites1());
		assertEquals("", psPatient.getSpecialNotesForLateSites2());
		assertEquals("010203", psPatient.getRightsObligationParagraphCodes());

		assertEquals(6, psPatient.getNumOfLines());
		assertEquals(200000000004L, psPatient.getDfnNumber());
		assertEquals(BooleanChar.N, psPatient.getLargeFontInd());
		assertEquals("ICN4", psPatient.getIcnNumber());
		assertEquals(BooleanChar.Y, psPatient.getAddressFlag());
		assertEquals("05252016", psPatient.getLastBillPrepDate().toString());
		assertEquals("24TESTL", psPatient.getAcntNumDisp());
		assertEquals(2, psPatient.getNumConsolidatedSites());
		assertEquals(BooleanChar.N, psPatient.getLateSiteFlag());
	
		// PV
		psRecord = psRecordSource.nextPSRecord();
		assertNotNull(psRecord);
		assertTrue(psRecord instanceof PSSiteStmt);
		psSiteStmt = (PSSiteStmt)psRecord;
		
		assertEquals(PSRecord.DataType.PV, psSiteStmt.getType());
		assertEquals("2", psSiteStmt.getFacilityNum());
		assertEquals("00000270{", psSiteStmt.getStationTotalAmount().getCobol());
		assertEquals(2, psSiteStmt.getNumOfLines());
		assertEquals("2000000004TESTL", psSiteStmt.getPatientAccount());
		assertEquals(200000000004L, psSiteStmt.getDfnNumber());
		assertEquals("ICN4", psSiteStmt.getIcnNumber());

		// PD
		psRecord = psRecordSource.nextPSRecord();
		assertNotNull(psRecord);
		assertTrue(psRecord instanceof PSDetails);
		psDetails = (PSDetails)psRecord;
		
		assertEquals(PSRecord.DataType.PD, psDetails.getType());
		assertEquals("05012016", psDetails.getDatePosted().toString());
		assertEquals("PAYMENT", psDetails.getTransDesc());
		assertEquals("00000400}", psDetails.getTransAmount().getCobol());
		assertEquals("REF_1", psDetails.getReferenceNum());
		assertEquals("ICN4", psDetails.getIcnNumber());
		assertEquals(1, psDetails.getSeqNum());
		
		// PD
		psRecord = psRecordSource.nextPSRecord();
		assertNotNull(psRecord);
		assertTrue(psRecord instanceof PSDetails);
		psDetails = (PSDetails)psRecord;
		
		assertEquals(PSRecord.DataType.PD, psDetails.getType());
		assertEquals("05012016", psDetails.getDatePosted().toString());
		assertEquals("CHARGE", psDetails.getTransDesc());
		assertEquals("00000270{", psDetails.getTransAmount().getCobol());
		assertEquals("REF_2", psDetails.getReferenceNum());
		assertEquals("ICN4", psDetails.getIcnNumber());
		assertEquals(2, psDetails.getSeqNum());
		
		// PV
		psRecord = psRecordSource.nextPSRecord();
		assertNotNull(psRecord);
		assertTrue(psRecord instanceof PSSiteStmt);
		psSiteStmt = (PSSiteStmt)psRecord;
		
		assertEquals(PSRecord.DataType.PV, psSiteStmt.getType());
		assertEquals("3", psSiteStmt.getFacilityNum());
		assertEquals("00000270{", psSiteStmt.getStationTotalAmount().getCobol());
		assertEquals(2, psSiteStmt.getNumOfLines());
		assertEquals("3000000004TESTL", psSiteStmt.getPatientAccount());
		assertEquals(300000000004L, psSiteStmt.getDfnNumber());
		assertEquals("ICN4", psSiteStmt.getIcnNumber());

		// PD
		psRecord = psRecordSource.nextPSRecord();
		assertNotNull(psRecord);
		assertTrue(psRecord instanceof PSDetails);
		psDetails = (PSDetails)psRecord;
		
		assertEquals(PSRecord.DataType.PD, psDetails.getType());
		assertEquals("05012016", psDetails.getDatePosted().toString());
		assertEquals("PAYMENT", psDetails.getTransDesc());
		assertEquals("00000400}", psDetails.getTransAmount().getCobol());
		assertEquals("REF_1", psDetails.getReferenceNum());
		assertEquals("ICN4", psDetails.getIcnNumber());
		assertEquals(1, psDetails.getSeqNum());
		
		// PD
		psRecord = psRecordSource.nextPSRecord();
		assertNotNull(psRecord);
		assertTrue(psRecord instanceof PSDetails);
		psDetails = (PSDetails)psRecord;
		
		assertEquals(PSRecord.DataType.PD, psDetails.getType());
		assertEquals("05012016", psDetails.getDatePosted().toString());
		assertEquals("CHARGE", psDetails.getTransDesc());
		assertEquals("00000270{", psDetails.getTransAmount().getCobol());
		assertEquals("REF_2", psDetails.getReferenceNum());
		assertEquals("ICN4", psDetails.getIcnNumber());
		assertEquals(2, psDetails.getSeqNum());

		// No more records
		psRecord = psRecordSource.nextPSRecord();
		assertNull(psRecord);
	}

	private void getCBSStmts1(final List<PSSite> psSitesList, final Map<String, List<CBSStmt>> cbsListsBySite,
			final Map<String, List<CBSStmt>> sbsListsBySite) {
		final long accountNum1 = 1L;
		final long accountNum2 = 2L;
		final long accountNum3 = 3L;
		final long accountNum4 = 4L;

		final String ICN1 = "ICN1";
		final String ICN2 = "ICN2";
		final String ICN3 = "ICN3";
		final String ICN4 = "ICN4";

		final long stationNum1 = 1L;
		final long stationNum2 = 2L;
		final long stationNum3 = 3L;

		Date statementDate = null;
		try {
			statementDate = (new SimpleDateFormat("MMddyyyy")).parse("05252016");
		} catch (ParseException e) {
		}

		// Create 2 NEW statements for station 1
		final CBSStmt newCBSStmt1 = CBSDAOImplUtils.createSimpleTestCBSStmtObject(accountNum1, 1, STMT_STATUS_NEW_ID,
				stationNum1, ICN1, statementDate);

		final CBSStmt newCBSStmt2 = CBSDAOImplUtils.createSimpleTestCBSStmtObject(accountNum2, 2, STMT_STATUS_NEW_ID,
				stationNum1, ICN2, statementDate);
		psSitesList.add(createPSSite(PSRecord.DataType.P1, 1, 1, newCBSStmt2.getPrimarySiteStmt(), 2,
				newCBSStmt2.getNewBalance().getDouble()));

		newCBSStmt2.setLateStmtMsg(true);

		final List<CBSStmt> cbsStmtL1 = new ArrayList<CBSStmt>(2);
		cbsStmtL1.add(newCBSStmt1);
		cbsStmtL1.add(newCBSStmt2);
		sbsListsBySite.put(newCBSStmt1.getPrimarySiteStmt().getStationNum(), cbsStmtL1);

		// Create a NEW statement for station 1 and station2.
		// station 1 is primary
		final CBSStmt newCBSStmt1and2 = CBSDAOImplUtils.createDoubleTestCBSStmtObject(accountNum3, ICN3, 3,
				STMT_STATUS_NEW_ID, stationNum1, stationNum2);
		psSitesList.add(createPSSite(PSRecord.DataType.PH, 1, 1, newCBSStmt1and2.getPrimarySiteStmt(), 1,
				newCBSStmt1and2.getNewBalance().getDouble()));
		final List<CBSStmt> cbsStmtL2 = new ArrayList<CBSStmt>(1);
		cbsStmtL2.add(newCBSStmt1and2);
		cbsListsBySite.put(newCBSStmt1and2.getPrimarySiteStmt().getStationNum(), cbsStmtL2);

		// Create a NEW statement for station 2 and station3.
		// station 2 is primary
		final CBSStmt newCBSStmt2and3 = CBSDAOImplUtils.createDoubleTestCBSStmtObject(accountNum4, ICN4, 4,
				STMT_STATUS_NEW_ID, stationNum2, stationNum3);
		psSitesList.add(createPSSite(PSRecord.DataType.PH, 1, 1, newCBSStmt2and3.getPrimarySiteStmt(), 1,
				newCBSStmt2and3.getNewBalance().getDouble()));
		final List<CBSStmt> cbsStmtL3 = new ArrayList<CBSStmt>(1);
		cbsStmtL3.add(newCBSStmt2and3);
		cbsListsBySite.put(newCBSStmt2and3.getPrimarySiteStmt().getStationNum(), cbsStmtL3);

	}

	private PSSite createPSSite(final PSRecord.DataType statementType, final int psSeqNum, final int psTotSeqNum,
			final CBSSiteStmt primarySite, final int psTotStatement, final double psStatementVal) {
		PSSite psSite = new PSSite();
		psSite.setType(PSRecord.DataType.PS);
		psSite.setStatementType(statementType);

		psSite.setSeqNum(psSeqNum);
		psSite.setTotSeqNum(psTotSeqNum);
		psSite.setFacilityNum(primarySite.getStationNum());
		psSite.setFacilityPhoneNum(primarySite.getStationPhoneNum());
		psSite.setStatementDate(primarySite.getStatementDate());
		psSite.setProcessDate(primarySite.getProcessDate());
		psSite.setTotStatement(psTotStatement);
		psSite.setStatementVal(new Money(psStatementVal, 11));

		return psSite;
	}

}
